/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.common.selection

import com.macrofocus.common.properties.MutableProperty
import com.macrofocus.common.properties.PropertyEvent
import com.macrofocus.common.properties.PropertyListener

class PropertySingleSelection<E>(private val property: MutableProperty<E>) : AbstractMutableSingleSelection<E>(),
    MutableSingleSelection<E> {
    override fun clearSelection(): Boolean {
        return if (isEnabled && isActive) {
            setSelectedOrNull(null)
            true
        } else {
            false
        }
    }

    override val isActive: Boolean
        get() = selected != null && isEnabled

    override fun isSelected(element: E?): Boolean {
        return property.value == element
    }

    override fun setSelectedElement(element: E) {
        this.selected = element
    }

    override fun setSelectedState(element: E, selected: Boolean) {
        if (isEnabled) {
            if (selected) {
                this.selected = element
            } else {
                if (isSelected(element)) {
                    clearSelection()
                }
            }
        }
    }

    override fun setSelectedIterableState(element: Iterable<E>, selected: Boolean): Boolean {
        throw UnsupportedOperationException()
    }

    override fun setSelectedElementsState(selected: Boolean, vararg elements: E): Boolean {
        throw UnsupportedOperationException()
    }

    override fun setSelectedElements(vararg elements: E) {
        throw UnsupportedOperationException()
    }

    override fun setSelectedIterable(elements: Iterable<E>) {
        throw UnsupportedOperationException()
    }

    override var selected: E?
        get() = property.value
        set(element) {
            setSelectedOrNull(element)
        }

    fun setSelectedOrNull(element: E?) {
        if (isEnabled) {
            if (selected !== element) {
                val previousSelected = selected
                val previousActive = previousSelected != null
                property.value = element!!
                notifySingleSelectedChanged(
                    SingleSelectionEvent<E>(
                        this,
                        previousActive,
                        element != null,
                        previousSelected,
                        element
                    )
                )
            }
        }
    }

    override val selectedCount: Int
        get() = if (isActive) 1 else 0

    override val selectedSet: Set<E>
        get() = if (isActive) setOf(selected!!) else emptySet()

    override operator fun iterator(): Iterator<E> {
        return object : Iterator<E> {
            var fetched = isActive
            override fun hasNext(): Boolean {
                return !fetched
            }

            override fun next(): E {
                fetched = true
                return selected!!
            }
        }
    }

    init {
        val self: MutableSingleSelection<E> = this
        property.addPropertyListener(object : PropertyListener<E> {
            override fun propertyChanged(event: PropertyEvent<E>) {
                notifySingleSelectedChanged(
                    SingleSelectionEvent<E>(
                        self,
                        event.oldValue != null,
                        event.newValue != null,
                        event.oldValue,
                        event.newValue
                    )
                )
            }
        })
    }
}