/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.common.collection

/**
 * Created by luc on 25/11/15.
 */
object Iterables {
    fun forIntArray(vararg array: Int): Iterable<Int> {
        return IntegerIterable(*array)
    }

//    fun <T> forArray(vararg array: T?): Iterable<T?>? {
//        return GenericIterable<T?>(*array)
//    }

//    fun <T> exclude(original: Iterable<T?>?, value: T?): Iterable<T?>? {
//        return conditional(
//            original,
//            EqualCondition<T?>(value)
//        )
//    }

    fun <T> nonNull(original: Iterable<T?>): Iterable<T> {
        return conditional(
            original,
            NonNullCondition<T?>()
        ) as Iterable<T>
    }

    fun <T> conditional(
        original: Iterable<T>,
        condition: Condition<in T>
    ): Iterable<T> {
        return original.filter { condition.apply(it) }
//        return ConditionalIterable<T>(original, condition)
    }

//    fun <T, C> convert(
//        original: Iterable<T?>?,
//        converter: Converter<in T?, C?>?
//    ): Iterable<C?>? {
//        return ConvertIterable<C?, T?>(original, converter)
//    }

//    @SafeVarargs
    fun <E> concatenate(vararg iterables: Iterable<E>): Iterable<E> {
        return concatenate(iterables.toList())
    }

    fun <E> concatenate(iterables: List<Iterable<E>>): Iterable<E> {
        return ConcatenateIterable<E>(iterables)
    }

//    fun <E> makeList(iter: Iterable<E?>?): MutableList<E?>? {
//        val list: MutableList<E?> = java.util.ArrayList<E?>()
//        for (item in iter!!) {
//            list.add(item)
//        }
//        return list
//    }

//    fun <E> makeCollection(iter: Iterable<E?>?): MutableCollection<E?>? {
//        return makeList(iter)
//    }

    // ToDo: This doesn't seem to be compatible with Java: can result in a ClassCastException
//    fun <E> makeArray(iter: Iterable<E?>?): Array<E?>? {
//        val list = makeList(iter)
//        return list!!.toTypedArray()
//    }

    /**
     * Copies an iterable's elements into an array.
     * ToDo: This is compatible with Java, not sure about JavaScript
     *
     * @param iter the iterable to copy
     * @param type the type of the elements
     * @return a newly-allocated array into which all the elements of the iterable have been copied
     */
//    @GwtIncompatible
//    fun <E> makeTypedArray(iter: Iterable<E?>?, type: java.lang.Class<E?>?): Array<E?>? {
//        val list = makeList(iter)
//        val array = java.lang.reflect.Array.newInstance(type, list!!.size) as Array<E?>
//        return list.toArray(array)
//    }

    interface Condition<T> {
        fun apply(var1: T): Boolean
    }

    interface Converter<T, C> {
        fun convert(var1: T?): C?
    }

    class ToStringConverter :
        Converter<Any?, String?> {
        override fun convert(var1: Any?): String? {
            return var1?.toString()
        }
    }

    private class ConditionalIterable<T>(
        private val original: Iterable<T>,
        private val condition: Condition<in T>
    ) : Iterable<T?> {
        override fun iterator(): Iterator<T?> {
            return object : MutableIterator<T?> {
                var it: Iterator<T?>? = original.iterator()
                var nextEntry: T? = null
                var queryNext = true
                override fun hasNext(): Boolean {
                    if (queryNext) {
                        nextEntry = findNext()
                        queryNext = false
                    }
                    return nextEntry != null
                }

                override fun next(): T {
                    if (queryNext) {
                        nextEntry = findNext()
                    }
                    queryNext = true
                    return nextEntry!!
                }

                private fun findNext(): T? {
                    while (it!!.hasNext()) {
                        val t = it!!.next()!!
                        if (condition.apply(t)) {
                            return t
                        }
                    }
                    return null
                }

                override fun remove() {
                    throw UnsupportedOperationException()
                }
            }
        }

    }

    private class IntegerIterable(vararg array: Int) : Iterable<Int> {
        private val array: IntArray?
        override fun iterator(): Iterator<Int> {
            return object : MutableIterator<Int> {
                private var pos = 0
                override fun hasNext(): Boolean {
                    return array != null && array.size > pos
                }

                override fun next(): Int {
                    return array!![pos++]
                }

                override fun remove() {
                    throw UnsupportedOperationException()
                }
            }
        }

        init {
            this.array = array
        }
    }

//    private class GenericIterable<T>(vararg array: T?) : Iterable<T?> {
//        private val array: Array<T?>?
//        override fun iterator(): Iterator<T?> {
//            return object : MutableIterator<T?> {
//                private var pos = 0
//                override fun hasNext(): Boolean {
//                    return array != null && array.size > pos
//                }
//
//                override fun next(): T? {
//                    return array!![pos++]
//                }
//
//                override fun remove() {
//                    throw UnsupportedOperationException()
//                }
//            }
//        }
//
//        init {
//            this.array = array
//        }
//    }

//    private class ConvertIterable<C, T>(
//        private val original: Iterable<T?>?,
//        private val converter: Converter<in T?, C?>?
//    ) : Iterable<C?> {
//        override fun iterator(): Iterator<C?> {
//            return object : MutableIterator<C?> {
//                var it: MutableIterator<T?>? = original!!.iterator()
//                override fun hasNext(): Boolean {
//                    return it!!.hasNext()
//                }
//
//                override fun next(): C? {
//                    return converter!!.convert(it!!.next())
//                }
//
//                override fun remove() {
//                    throw UnsupportedOperationException()
//                }
//            }
//        }
//
//    }

    private class ConcatenateIterable<E>(iterables: List<Iterable<E>>) : Iterable<E> {
        val iterablesIterator: Iterator<Iterable<E>>?
        private val iterables: Iterable<Iterable<E>>?
        override fun iterator(): Iterator<E> {
            return if (!iterablesIterator!!.hasNext()) EmptyIterator<E>() else ConcatenateIterator().findNext()
        }

        private inner class ConcatenateIterator : Iterator<E> {
            var iterableIterator = nextIterator()

            override fun hasNext(): Boolean {
                return iterableIterator.hasNext()
            }

            override fun next(): E {
                val next = iterableIterator.next()!!
                findNext()
                return next
            }

            fun nextIterator(): Iterator<E> {
                return iterablesIterator!!.next().iterator()
            }

            fun findNext(): Iterator<E> {
                while (!iterableIterator.hasNext()) {
                    if (!iterablesIterator!!.hasNext()) {
                        break
                    }
                    iterableIterator = nextIterator()
                }
                return this
            }
        }

        private class EmptyIterator<E> : Iterator<E> {
            override fun hasNext(): Boolean {
                return false
            }

            override fun next(): E {
                throw IllegalStateException()
            }
        }

        init {
            this.iterables = iterables
            iterablesIterator = iterables.iterator()
        }
    }

    private class EqualCondition<T>(private val value: T?) : Condition<T?> {
        override fun apply(var1: T?): Boolean {
            return if (var1 == null) {
                var1 !== value
            } else {
                var1 != value
            }
        }

    }

    private class NonNullCondition<T> : Condition<T?> {
        override fun apply(var1: T?): Boolean {
            return var1 != null
        }
    }
}