/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.common.properties

open class OverrideProperty<T>(
    overrideDefaults: MutableProperty<Boolean>,
    defaultProperty: MutableProperty<T>
) : AbstractMutableProperty<T>() {
    private val overrideDefaults: MutableProperty<Boolean>
    protected open val defaultProperty: MutableProperty<T>
    private var overrideProperty: MutableProperty<T>? = null
    val overridePropertyListener: PropertyListener<T> = object : PropertyListener<T> {
        override fun propertyChanged(event: PropertyEvent<T>) {
            if (overrideDefaults.value) {
                notifyPropertyChanged(PropertyEvent(event.oldValue, event.newValue))
            }
        }
    }

    fun getOverrideProperty(): MutableProperty<T>? {
        if (overrideProperty == null) {
            overrideProperty = createOverrideProperty()
        }
        return overrideProperty
    }

    protected open fun createOverrideProperty(): MutableProperty<T> {
        val overrideProperty: MutableProperty<T> = SimpleProperty.newInstance(defaultProperty.value)
        return addProperty(overrideProperty)
    }

    protected fun addProperty(overrideProperty: MutableProperty<T>): MutableProperty<T> {
        overrideProperty.addPropertyListener(overridePropertyListener)
        return overrideProperty
    }

    protected fun removeProperty(overrideProperty: MutableProperty<T>) {
        overrideProperty.removePropertyListener(overridePropertyListener)
    }

    override var value: T
        get() =         // As long as the value as not been set using setValue, we return the default value
            if (overrideDefaults.value && overrideProperty != null) {
//            return getOverrideProperty().getValue();
                overrideProperty!!.value
            } else {
                defaultProperty.value
            }

        set(value) {if (overrideDefaults.value) {
            getOverrideProperty()!!.value = value
        } else {
            defaultProperty.value = value
        }}

    // As long as the value as not been set using setValue, we return the default value
    val overrideValue: T?
        get() =// As long as the value as not been set using setValue, we return the default value
            if (overrideDefaults.value && overrideProperty != null) {
//            return getOverrideProperty().getValue();
                overrideProperty!!.value
            } else {
                null
            }

    fun reset() {
        if (overrideDefaults.value && overrideProperty != null) {
            val oldValue: T = overrideProperty!!.value
            val newValue: T = defaultProperty.value
            removeProperty(overrideProperty!!)
            overrideProperty = null
            if (oldValue !== newValue) {
                notifyPropertyChanged(PropertyEvent(oldValue, newValue))
            }
        }
        overrideProperty = null
    }

    init {
        this.overrideDefaults = overrideDefaults
        this.defaultProperty = defaultProperty

//        createOverrideProperty();
        overrideDefaults.addPropertyListener(object : PropertyListener<Boolean> {
            override fun propertyChanged(event: PropertyEvent<Boolean>) {
                val defaultValue: T = defaultProperty.value
                val overrideValue = if (overrideProperty != null) overrideProperty!!.value else defaultValue
                if (defaultValue !== overrideValue) {
                    if (event.newValue) {
                        notifyPropertyChanged(PropertyEvent<T>(defaultValue, overrideValue))
                    } else {
                        notifyPropertyChanged(PropertyEvent<T>(overrideValue, defaultValue))
                    }
                }
            }
        })
        defaultProperty.addPropertyListener(object : PropertyListener<T> {
            override fun propertyChanged(event: PropertyEvent<T>) {
                if (!overrideDefaults.value) {
                    notifyPropertyChanged(PropertyEvent(event.oldValue, event.newValue))
                }
            }
        })
    }
}