/*
 * Copyright (c) 2016 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.docking.compose

import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.unit.dp
import com.macrofocus.common.command.Command
import com.macrofocus.common.properties.MutableProperty
import com.macrofocus.common.properties.SimpleArrayProperty
import com.macrofocus.common.properties.SimpleProperty
import com.macrofocus.docking.Dockable
import com.macrofocus.docking.DockingAnchor
import com.macrofocus.docking.DockingBar
import com.macrofocus.docking.DockingState
import org.mkui.component.CPComponent
import org.mkui.compose.ComposeComponent
import org.mkui.compose.toState

class ComposeDockingBar(private val container: DockingContainer, private val orientation: DockingAnchor) : DockingBar<CPComponent> {
    private val orientationProperty : MutableProperty<DockingAnchor> = SimpleProperty<DockingAnchor>(orientation)
    private val components = SimpleArrayProperty<CPComponent>(arrayOf())

    override val nativeComponent: CPComponent = ComposeComponent {
        val (orientation, _) = orientationProperty.toState()
        val (getComponent, _) = components.toState()

        if (orientation == DockingAnchor.LEFT || orientation == DockingAnchor.RIGHT) {
            Column(
                modifier = Modifier
                    .fillMaxSize()
                    .background(Color.Gray)
                    .padding(16.dp) // Optional: Add padding for better spacing
            ) {
                for (c in getComponent) {
                    c.nativeComponent()
                }
            }
        } else {
            Row (
                modifier = Modifier
                    .fillMaxSize()
                    .background(Color.Gray)
                    .padding(16.dp) // Optional: Add padding for better spacing
            ) {
                for (c in getComponent) {
                    c.nativeComponent()
                }
            }
        }
    }

    override val dockables: MutableList<Dockable<CPComponent>?>?
    private val dockingButtons: MutableMap<Dockable<CPComponent>, DockingButton>
    private val dockingStates: MutableMap<Dockable<CPComponent>, DockingState>

    init {
        dockables = ArrayList()
        dockingButtons = HashMap()
        dockingStates = HashMap()

        //        nativeComponent.setStyle("-fx-background-color: #778999;");
    }

    override fun attach(dockable: Dockable<CPComponent>) {
        dockable.dockingBar = this

        val first = dockables!!.isEmpty()
        dockables.add(dockable)

        val button = DockingButton(orientation)
        button.text = dockable.shortTitle
        if (dockable.iconURL != null) {
        }
        dockingButtons[dockable] = button

        button.command = Command { setActive(dockable, button.selected) }
//        button.onAction = EventHandler { event: ActionEvent? -> setActive(dockable, button.isSelected) }

        components.append(button)

        val state = DockingState()
        dockingStates[dockable] = state

        if (first) {
            setActive(dockable, true)
        }
    }

    override fun detach(dockable: Dockable<CPComponent>) {
        setActive(dockable, false)
        dockables!!.remove(dockable)
        dockingButtons.remove(dockable)
        dockingStates.remove(dockable)
    }

    override fun removeAll() {
    }

    override fun isActive(dockable: Dockable<CPComponent>): Boolean {
        val state = dockingStates[dockable]
        return state!!.isActive
    }

    override fun setActive(dockable: Dockable<CPComponent>, active: Boolean) {
        val state = dockingStates[dockable]

        if (!state!!.isFloating) {
            for (d in dockables!!) {
                val s = dockingStates[d]

                if (s!!.isActive) {
                    s.isActive = d === dockable && active
                    val button = dockingButtons[d]
                    button!!.selected = d === dockable && active
                    container.remove(d!!.component)
                }
            }
            if (active) {
                state.isActive = true
                container.center = dockable.component
                container.visible = true
                val button = dockingButtons[dockable]
                button!!.selected = true
            } else {
                container.visible = false
            }
        } else {
//            SwingUtilities.windowForComponent(dockable.getContainer()).toFront();
            val button = dockingButtons[dockable]
            button!!.selected = false
        }
    }

    override fun setFloating(dockable: Dockable<CPComponent>, floating: Boolean) {
    }

    override fun setSliding(dockable: Dockable<CPComponent>, sliding: Boolean) {
    }

    override fun dispose() {
    }
}
