/*
 * Copyright (c) 2016 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.docking.javafx

import com.macrofocus.docking.Dockable
import com.macrofocus.docking.DockingAnchor
import com.macrofocus.docking.DockingBar
import com.macrofocus.docking.DockingState
import javafx.event.ActionEvent
import javafx.event.EventHandler
import javafx.geometry.Orientation
import javafx.scene.Group
import javafx.scene.Node
import javafx.scene.control.ToolBar
import javafx.scene.image.Image
import javafx.scene.image.ImageView

class JavaFXDockingBar(private val container: DockingContainer, private val orientation: DockingAnchor) : DockingBar<Node> {
    override val nativeComponent: ToolBar = ToolBar()

    override val dockables: MutableList<Dockable<Node>?>?
    private val dockingButtons: MutableMap<Dockable<Node>, DockingButton>
    private val dockingStates: MutableMap<Dockable<Node>, DockingState>

    init {
        nativeComponent.styleClass.add("buttonArea")

        nativeComponent.orientation =
            if (orientation == DockingAnchor.LEFT || orientation == DockingAnchor.RIGHT) Orientation.VERTICAL else Orientation.HORIZONTAL

        dockables = ArrayList()
        dockingButtons = HashMap()
        dockingStates = HashMap()

        //        nativeComponent.setStyle("-fx-background-color: #778999;");
    }

    override fun attach(dockable: Dockable<Node>) {
        dockable.dockingBar = this

        val first = dockables!!.isEmpty()
        dockables.add(dockable)

        val button = DockingButton(
            orientation
        )
        button.text = dockable.shortTitle
        if (dockable.iconURL != null) {
            try {
                button.graphicProperty().setValue(ImageView(Image(dockable.iconURL)))
            } catch (e: IllegalArgumentException) {
                System.err.println("Could not load " + dockable.iconURL)
                e.printStackTrace()
            }
        }
        dockingButtons[dockable] = button

        button.onAction = EventHandler { event: ActionEvent? -> setActive(dockable, button.isSelected) }

        nativeComponent.items.add(Group(button))

        val state = DockingState()
        dockingStates[dockable] = state

        if (first) {
            setActive(dockable, true)
        }
    }

    override fun detach(dockable: Dockable<Node>) {
        setActive(dockable, false)
        dockables!!.remove(dockable)
        dockingButtons.remove(dockable)
        dockingStates.remove(dockable)
    }

    override fun removeAll() {
    }

    override fun isActive(dockable: Dockable<Node>): Boolean {
        val state = dockingStates[dockable]
        return state!!.isActive
    }

    override fun setActive(dockable: Dockable<Node>, active: Boolean) {
        val state = dockingStates[dockable]

        if (!state!!.isFloating) {
            for (d in dockables!!) {
                val s = dockingStates[d]

                if (s!!.isActive) {
                    s.isActive = d === dockable && active
                    val button = dockingButtons[d]
                    button!!.isSelected = d === dockable && active
                    container.children.remove(d!!.component)
                }
            }
            if (active) {
                state.isActive = true
                container.center = dockable.component
                container.isVisible = true
                val button = dockingButtons[dockable]
                button!!.isSelected = true
            } else {
                container.isVisible = false
            }
        } else {
//            SwingUtilities.windowForComponent(dockable.getContainer()).toFront();
            val button = dockingButtons[dockable]
            button!!.isSelected = false
        }
    }

    override fun setFloating(dockable: Dockable<Node>, floating: Boolean) {
    }

    override fun setSliding(dockable: Dockable<Node>, sliding: Boolean) {
    }

    override fun dispose() {
    }
}
