/*
 * Copyright (c) 2016 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.docking.javafx

import com.macrofocus.docking.*
import com.macrofocus.docking.javafx.splitter.JavaFXThreeComponentsSplitter
import com.macrofocus.docking.splitter.ThreeComponentsSplitter
import javafx.scene.Node
import javafx.scene.Parent
import javafx.scene.layout.GridPane
import javafx.scene.layout.Priority
import org.mkui.component.CPComponent
import javax.swing.ImageIcon
import javax.swing.JComponent

class JavaFXDockingPanel @JvmOverloads constructor(vertical: Boolean = true) : DockingPanel<Node> {
    override val nativeComponent: GridPane = GridPane()

    private val primarySplitter: ThreeComponentsSplitter<Node?> = JavaFXThreeComponentsSplitter(vertical)
    private val secondarySplitter: ThreeComponentsSplitter<Node?> = JavaFXThreeComponentsSplitter(!vertical)
    override val westDockingBar: DockingBar<Node>
    override val eastDockingBar: DockingBar<Node>
    override val southDockingBar: DockingBar<Node>
    override val northDockingBar: DockingBar<Node>

    private var vertical: Boolean? = null

    override var innerComponent: Node? = null
        get() { return field }
        set(value) {
            field = value

            updateDockingContainers()
        }
    var westContainerPanel: DockingContainer
    var eastContainerPanel: DockingContainer
    var southContainerPanel: DockingContainer
    var northContainerPanel: DockingContainer

    init {
        val style = javaClass.getResource("style.css").toExternalForm()
        nativeComponent.stylesheets.addAll(style)

        westContainerPanel = DockingContainer()
        eastContainerPanel = DockingContainer()
        southContainerPanel = DockingContainer()
        northContainerPanel = DockingContainer()

        westDockingBar = JavaFXDockingBar(westContainerPanel, DockingAnchor.LEFT)
        eastDockingBar = JavaFXDockingBar(eastContainerPanel, DockingAnchor.RIGHT)
        southDockingBar = JavaFXDockingBar(southContainerPanel, DockingAnchor.BOTTOM)
        northDockingBar = JavaFXDockingBar(northContainerPanel, DockingAnchor.TOP)

        nativeComponent.add(JavaFXDockingCorner(), 0, 0)
        setTop(northDockingBar.nativeComponent)
        nativeComponent.add(JavaFXDockingCorner(), 2, 0)
        setLeft(westDockingBar.nativeComponent)
        nativeComponent.add(JavaFXDockingCorner(), 1, 1)
        setRight(eastDockingBar.nativeComponent)
        nativeComponent.add(JavaFXDockingCorner(), 0, 2)
        setBottom(southDockingBar.nativeComponent)
        nativeComponent.add(JavaFXDockingCorner(), 2, 2)

        setOrientation(vertical)
        updateDockingBars()
    }

    private fun setBottom(nativeComponent: Node) {
        this.nativeComponent.add(nativeComponent, 1, 2)
    }

    private fun setTop(nativeComponent: Node) {
        this.nativeComponent.add(nativeComponent, 1, 0)
    }

    private fun setLeft(nativeComponent: Node) {
        this.nativeComponent.add(nativeComponent, 0, 1)
    }

    private fun setRight(nativeComponent: Node) {
        this.nativeComponent.add(nativeComponent, 2, 1)
    }

    private fun setCenter(nativeComponent: Node) {
        this.nativeComponent.add(nativeComponent, 1, 1)
        GridPane.setVgrow(nativeComponent, Priority.ALWAYS)
        GridPane.setHgrow(nativeComponent, Priority.ALWAYS)
    }

    override fun setOrientation(vertical: Boolean) {
        if (this.vertical == null || this.vertical != vertical) {
            this.vertical = vertical
            var innerComponent: Node? = null
            var secondaryFirstSize: Double? = null
            var secondaryLastSize: Double? = null
            var primaryFirstSize: Double? = null
            var primaryLastSize: Double? = null
            secondaryFirstSize = secondarySplitter.firstSizeIfVisible
            secondaryLastSize = secondarySplitter.lastSizeIfVisible
            innerComponent = secondarySplitter.innerComponent
            secondarySplitter.setComponents(null, null, null)

            primaryFirstSize = primarySplitter.firstSizeIfVisible
            primaryLastSize = primarySplitter.lastSizeIfVisible
            primarySplitter.setComponents(null, null, null)

            primarySplitter.vertical = vertical
            secondarySplitter.vertical = !vertical

            secondarySplitter.firstSize = primaryFirstSize
            secondarySplitter.lastSize = primaryLastSize
            primarySplitter.firstSize = secondaryFirstSize
            primarySplitter.lastSize = secondaryLastSize

            if (vertical) {
                secondarySplitter.setComponents(westContainerPanel, innerComponent, eastContainerPanel)
                primarySplitter.setComponents(northContainerPanel, secondarySplitter.nativeComponent, southContainerPanel)
            } else {
                secondarySplitter.setComponents(northContainerPanel, innerComponent, southContainerPanel)
                primarySplitter.setComponents(westContainerPanel, secondarySplitter.nativeComponent, eastContainerPanel)
            }


            //            primarySplitter.setDividerPositions(1f / 3f, 2f / 3f);
//            if(innerComponent != null) {
//                secondarySplitter.setDividerPositions(1 / 3f, 2 / 3f);
//            } else {
//                secondarySplitter.setDividerPositions(0.5f);
//            }


//            secondarySplitter.setShowDividerControls(true);
//            primarySplitter.setShowDividerControls(true);
            setCenter(primarySplitter.nativeComponent!!)
        }
    }

    private val horizontalSplitter: ThreeComponentsSplitter<Node?>
        private get() = if (!vertical!!) {
            primarySplitter
        } else {
            secondarySplitter
        }
    private val verticalSplitter: ThreeComponentsSplitter<Node?>
        private get() = if (vertical!!) {
            primarySplitter
        } else {
            secondarySplitter
        }


    override var horizontalFirstDividerSize: Double
        get() = horizontalSplitter.firstSizeIfVisible
        set(size) {
            horizontalSplitter.firstSize = size
        }

    override var horizontalLastDividerSize: Double
        get() = horizontalSplitter.lastSizeIfVisible
        set(size) {
            horizontalSplitter.lastSize = size
        }

    override var verticalFirstDividerSize: Double
        get() = primarySplitter.firstSizeIfVisible
        set(size) {
            verticalSplitter.firstSize = size
        }

    override var verticalLastDividerSize: Double
        get() = primarySplitter.lastSizeIfVisible
        set(size) {
            verticalSplitter.lastSize = size
        }


    override fun setHorizontalRelativeSize(relativeResize: Boolean) {
        secondarySplitter.isRelativeResize = relativeResize
    }

    override fun setVerticalRelativeResize(relativeResize: Boolean) {
        primarySplitter.isRelativeResize = relativeResize
    }

    override fun dispose() {
    }

    override fun attach(dockable: Dockable<Node>?, anchor: AttachAnchor?) {
        when (anchor) {
            AttachAnchor.TOP -> northDockingBar.attach(dockable!!)
            AttachAnchor.BOTTOM -> southDockingBar.attach(dockable!!)
            AttachAnchor.LEFT -> westDockingBar.attach(dockable!!)
            AttachAnchor.RIGHT -> eastDockingBar.attach(dockable!!)
            AttachAnchor.CENTER -> innerComponent = dockable!!.component
            null -> TODO()
        }
        updateDockingBars()
    }

    private fun updateDockingBars() {
//        nativeComponent.setWidgetHidden(getNorthDockingBar().getNativeComponent(), getNorthDockingBar().getDockables().size() == 0);
//        nativeComponent.setWidgetHidden(getSouthDockingBar().getNativeComponent(), getSouthDockingBar().getDockables().size() == 0);
//        nativeComponent.setWidgetHidden(getWestDockingBar().getNativeComponent(), getWestDockingBar().getDockables().size() == 0);
//        nativeComponent.setWidgetHidden(getEastDockingBar().getNativeComponent(), getEastDockingBar().getDockables().size() == 0);

        updateDockingContainers()
    }

    fun updateDockingContainers() {
        val westContainerPanel = if (westDockingBar.dockables!!.size > 0) this.westContainerPanel else null
        val eastContainerPanel = if (eastDockingBar.dockables!!.size > 0) this.eastContainerPanel else null
        val northContainerPanel = if (northDockingBar.dockables!!.size > 0) this.northContainerPanel else null
        val southContainerPanel = if (southDockingBar.dockables!!.size > 0) this.southContainerPanel else null

        if (vertical!!) {
            val secondarySplitter =
                if (westContainerPanel != null || innerComponent != null || eastContainerPanel != null) secondarySplitter.nativeComponent else null
            this.secondarySplitter.setComponents(westContainerPanel, innerComponent, eastContainerPanel)
            primarySplitter.setComponents(northContainerPanel, secondarySplitter, southContainerPanel)
        } else {
            val secondarySplitter =
                if (northContainerPanel != null || innerComponent != null || southContainerPanel != null) secondarySplitter.nativeComponent else null
            this.secondarySplitter.setComponents(northContainerPanel, innerComponent, southContainerPanel)
            primarySplitter.setComponents(westContainerPanel, secondarySplitter, eastContainerPanel)
        }
    }

    override fun createNakedDockable(component: Node, title: String?): Dockable<Node>? {
        return createNakedDockable(component, title, null)
    }

    override fun createNakedDockable(component: Node, title: String?, icon: String?): Dockable<Node>? {
        return JavaFXNakedDockable(component, title, icon)
    }

    override fun createDecoratedDockable(component: Node, title: String?): Dockable<Node>? {
        return createDecoratedDockable(component, title, null)
    }

    override fun createDecoratedDockable(component: Node, shortTitle: String?, title: String?, description: String?): Dockable<Node>? {
        return createDecoratedDockable(component, shortTitle, title, description, null)
    }

    fun createNakedDockable(component: JComponent, title: String?, icon: ImageIcon?): Dockable<JComponent> {
        return NakedDockable(component, title!!, null)
    }

    fun createDecoratedDockable(component: Node, title: String?, iconUrl: String?): Dockable<Node> {
        return JavaFXDockable(component, title, iconUrl)
    }

    fun createDecoratedDockable(component: Node, shortTitle: String?, title: String?, description: String?, iconUrl: String?): Dockable<Node> {
        return JavaFXDockable(component, shortTitle, title, description, iconUrl)
    }
}
