/*
 * Copyright (c) 2017 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.docking.javafx.splitter

import com.macrofocus.docking.splitter.AbstractThreeComponentsSplitter
import javafx.beans.value.ChangeListener
import javafx.geometry.Orientation
import javafx.scene.Node
import javafx.scene.control.SplitPane

class JavaFXThreeComponentsSplitter @JvmOverloads constructor(vertical: Boolean = false) :
    AbstractThreeComponentsSplitter<Node?>() {
    override val nativeComponent: SplitPane = SplitPane()

    private var myFirstComponent: Node? = null
    private var myInnerComponent: Node? = null
    private var myLastComponent: Node? = null

    private var firstDividerPosition: Double? = null
    private var lastDividerPosition: Double? = null

    val changeListener: ChangeListener<Boolean> = ChangeListener { observable, oldValue, newValue ->
        update()
        //            restoreDividerPositions();
    }

    override var vertical: Boolean
        get() = if(nativeComponent.orientation == Orientation.VERTICAL) true else false
        set(value) {
            nativeComponent.orientation = if (value) Orientation.VERTICAL else Orientation.HORIZONTAL
        }

    init {
        this.vertical = vertical
        nativeComponent.setPrefSize(1.0, 1.0)
    }


    override fun setComponents(first: Node?, inner: Node?, last: Node?) {
//        saveDividerPositions();

        firstComponent = first
        innerComponent = inner
        lastComponent = last

        //        restoreDividerPositions();
    }

    private fun restoreDividerPositions() {
        System.err.println("Restoring $firstDividerPosition, $lastDividerPosition")

        if (firstDividerVisible() && firstDividerPosition != null) {
            nativeComponent.setDividerPosition(0, firstDividerPosition!!)
            if (lastDividerVisible() && lastDividerPosition != null) {
                nativeComponent.setDividerPosition(1, lastDividerPosition!!)
            }
        } else {
            if (lastDividerVisible() && lastDividerPosition != null) {
                nativeComponent.setDividerPosition(0, lastDividerPosition!!)
            }
        }
    }

    private fun saveDividerPositions() {
        System.err.println("Saving $firstDividerPosition, $lastDividerPosition")

        if (firstDividerVisible()) {
            firstDividerPosition = nativeComponent.dividers[0].position
            if (lastDividerVisible()) {
                lastDividerPosition = nativeComponent.dividers[1].position
            }
        } else {
            if (lastDividerVisible()) {
                lastDividerPosition = nativeComponent.dividers[0].position
            }
        }
    }

    override var firstComponent: Node?
        get() = myFirstComponent
        set(component) {
            if (myFirstComponent !== component) {
                if (myFirstComponent != null) {
                    myFirstComponent!!.visibleProperty().removeListener(changeListener)
                }

                myFirstComponent = component

                if (myFirstComponent != null) {
                    myFirstComponent!!.visibleProperty().addListener(changeListener)
                }

                update()
            }
        }

    override var lastComponent: Node?
        get() = myLastComponent
        set(component) {
            if (myLastComponent !== component) {
                if (myLastComponent != null) {
                    myLastComponent!!.visibleProperty().removeListener(changeListener)
                }

                myLastComponent = component

                if (myLastComponent != null) {
                    myLastComponent!!.visibleProperty().addListener(changeListener)
                }

                update()
            }
        }

    override var innerComponent: Node?
        get() = myInnerComponent
        set(component) {
            if (myInnerComponent !== component) {
                if (myInnerComponent != null) {
                    myInnerComponent!!.visibleProperty().removeListener(changeListener)
                }

                myInnerComponent = component

                if (myInnerComponent != null) {
                    myInnerComponent!!.visibleProperty().addListener(changeListener)
                }

                update()
            }
        }

    override var firstSize: Double
        get() = nativeComponent.dividers[0].position
        set(size) {
            if (myFirstComponent != null && nativeComponent.dividers.size > 0) {
                nativeComponent.dividers[0].position = size
            }
        }

    override var lastSize: Double
        get() {
            val index = if (myInnerComponent != null) 1 else 0
            return nativeComponent.dividers[index].position
        }
        set(size) {
            if (myInnerComponent != null || myLastComponent != null) {
                val index = if (myInnerComponent != null) 1 else 0
                nativeComponent.dividers[index].position = 1.0 - size
            }
        }

    override val firstSizeIfVisible: Double
        get() = if (myFirstComponent != null) nativeComponent.dividers[0].position else 0.0

    override val lastSizeIfVisible: Double
        get() {
            val index = if (myInnerComponent != null) 1 else 0
            return if (myLastComponent != null) 1.0 - nativeComponent.dividers[index].position else 0.0
        }

    override var isRelativeResize: Boolean
        get() = true
        set(relativeResize) {
        }

    override fun setWidgetHidden(widget: Node?, hidden: Boolean) {
    }

    override fun setShowDividerControls(show: Boolean) {
    }

    private fun firstVisible(): Boolean {
        return myFirstComponent != null && myFirstComponent!!.isVisible
    }

    private fun lastVisible(): Boolean {
        return myLastComponent != null && myLastComponent!!.isVisible
    }

    private fun innerVisible(): Boolean {
        return myInnerComponent != null && myInnerComponent!!.isVisible
    }

    private fun visibleDividersCount(): Int {
        var count = 0
        if (firstDividerVisible()) count++
        if (lastDividerVisible()) count++
        return count
    }

    private fun firstDividerVisible(): Boolean {
        return (firstVisible() && innerVisible()) || (firstVisible() && lastVisible() && !innerVisible())
    }

    private fun lastDividerVisible(): Boolean {
        return innerVisible() && lastVisible()
    }

    private fun update() {
        val nodes: MutableList<Node> = ArrayList(3)

        if (myFirstComponent != null && myFirstComponent!!.isVisible) {
            nodes.add(myFirstComponent!!)
        }

        if (myInnerComponent != null && myInnerComponent!!.isVisible) {
            nodes.add(myInnerComponent!!)
        }

        if (myLastComponent != null && myLastComponent!!.isVisible) {
            nodes.add(myLastComponent!!)
        }

        nativeComponent.items.setAll(nodes)
    }
}
