/*
 * Copyright (c) 2016 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.docking.swing

import com.macrofocus.docking.swing.plaf.DockingAddon
import de.javasoft.docking.JYDockingPort
import de.javasoft.docking.controls.*
import de.javasoft.docking.ui.DockingViewUI
import de.javasoft.docking.ui.addons.JYDockingAddon
import de.javasoft.syntheticaaddons.SyntheticaAddons
import org.jdesktop.swingx.plaf.LookAndFeelAddons
import java.awt.*
import java.awt.event.HierarchyEvent
import java.awt.event.HierarchyListener
import java.io.Serializable
import javax.swing.Action
import javax.swing.Icon
import javax.swing.JComponent
import javax.swing.JPanel
import javax.swing.plaf.ComponentUI

class DockingView(paramString1: String?, paramString2: String?, paramString3: String?) : JComponent(), IDockable,
    IDockingConstants {
    private var titlebar: DockingViewTitleBar? = null
    protected var contentPane: JPanel?
    protected var userContentPane: Container? = null
    private val delegate: DockableComponentWrapper
    var paintContentPaneBackground: Boolean

    companion object {
        private const val serialVersionUID = 1910286791529421496L
        const val uiClassID = "DockingViewUI"
        val VIEW_DOCKING_STRATEGY: IDockingStrategy
        private fun createDockingStrategy(): IDockingStrategy {
            return object : DockingStrategy() {
                override fun createDockingPortImpl(paramAnonymousIDockingPort: IDockingPort?): IDockingPort {
                    return JYDockingPort()
                }
            }
        }

        init {
            SyntheticaAddons.contribute(JYDockingAddon())
            LookAndFeelAddons.contribute(DockingAddon())
            VIEW_DOCKING_STRATEGY = createDockingStrategy()
        }
    }

    override fun updateUI() {
        setUI(LookAndFeelAddons.getUI(this, DockingViewUI::class.java))

//        DockingManager.isMaximized(this);
    }

    override fun getUI(): ComponentUI {
        return this.ui as DockingViewUI
    }

    protected override fun addImpl(paramComponent: Component?, paramObject: Any?, paramInt: Int) {
        if (paramComponent is DockingViewTitleBar) {
            (paramComponent as DockingViewTitleBar).setView(this)
            super.addImpl(paramComponent, paramObject, paramInt)
        } else if (paramComponent === contentPane) {
            super.addImpl(paramComponent, paramObject, paramInt)
        } else {
            getContentPane()!!.add(paramComponent, paramObject, paramInt)
        }
    }

    override fun remove(paramComponent: Component) {
        if (paramComponent === userContentPane) contentPane!!.remove(paramComponent) else getContentPane()!!.remove(
            paramComponent
        )
    }

    protected fun createContentPane(): Container {
        return JPanel()
    }

    protected fun createLayout(): LayoutManager {
        return ViewLayout()
    }

    @Throws(IllegalArgumentException::class)
    fun setContentPane(paramContainer: Container?) {
        if (userContentPane != null) remove(userContentPane!!)
        userContentPane = paramContainer
        contentPane!!.add(userContentPane)
    }

    fun getContentPane(): Container? {
        return userContentPane
    }

    var title: String?
        get() {
            val localDockingViewTitlebar = getTitlebar()
            return localDockingViewTitlebar?.getTitleText()
        }
        set(paramString) {
            if (titlebar != null) titlebar!!.setTitleText(paramString)
        }//            DockingManager.removeDragListeners(getTitlebar());

    //            DockingManager.updateDragListeners(this);
    var isDraggingEnabled: Boolean
        get() = dragSources.contains(getTitlebar()!!)
        set(paramBoolean) {
            if (paramBoolean) {
                dragSources.add(getTitlebar()!!)
                //            DockingManager.updateDragListeners(this);
            } else {
                dragSources.remove(getTitlebar()!!)
                //            DockingManager.removeDragListeners(getTitlebar());
            }
        }

    fun setTitlebar(paramDockingViewTitlebar: DockingViewTitleBar?) {
        require(!(paramDockingViewTitlebar != null && paramDockingViewTitlebar === userContentPane)) { "Cannot use the same component as both content pane and titlebar." }
        if (titlebar != null) {
            remove(titlebar!!)
            titlebar!!.setView(null)
            dragSources!!.remove(titlebar!!)
            DockingManager.removeDragListeners(titlebar)
        }
        titlebar = paramDockingViewTitlebar
        if (paramDockingViewTitlebar != null) {
            add(paramDockingViewTitlebar)
            isDraggingEnabled = true
        }
    }

    fun getTitlebar(): DockingViewTitleBar? {
        return titlebar
    }

    fun addAction(paramAction: Action?) {
        if (titlebar != null) titlebar!!.addAction(paramAction)
    }

    fun removeAction(paramAction: Action?) {
        if (titlebar != null) titlebar!!.removeAction(paramAction)
    }

    var icon: Icon?
        get() = if (titlebar == null) null else titlebar!!.getTitleIcon()
        set(paramIcon) {
            if (titlebar != null) titlebar!!.setTitleIcon(paramIcon)
        }

    override fun setLayout(paramLayoutManager: LayoutManager) {
        getContentPane()!!.layout = paramLayoutManager
    }

    protected override fun paramString(): String {
        return "id=" + id + "," + super.paramString()
    }

    private fun clearButtonRollovers() {
        if (titlebar == null) {
            return
        }
        val arrayOfComponent = titlebar!!.getComponents()
        for (i in arrayOfComponent.indices) {
            val localObject: Any? = if (arrayOfComponent[i] is DockingButton) arrayOfComponent[i] else null
            if (localObject != null) (localObject as DockingButton).model.isRollover = false
        }
    }

    fun dock(paramIDockable: IDockable?, paramString: String?, paramFloat: Float): Boolean {
        return delegate.dock(
            paramIDockable,
            DockingManager.getPerspectiveManager().getCurrentPerspective(),
            paramString,
            paramFloat
        )
    }

    override fun getComponent(): JComponent {
        return delegate.component
    }

    override fun getID(): String {
        return delegate.id
    }

    override fun getDockingPort(): IDockingPort {
        return delegate.dockingPort
    }

    override fun getDragSources(): MutableList<JComponent> {
        return delegate.getDragSources() as MutableList<JComponent>
    }

    override fun dock(
        paramIDockable: IDockable?,
        paramPerspective: Perspective?,
        paramString: String?,
        paramFloat: Float
    ): Boolean {
        return delegate.dock(paramIDockable, paramPerspective, paramString, paramFloat)
    }

    override fun addDockingListener(paramIDockingEventListener: IDockingEventListener?) {
        delegate.addDockingListener(paramIDockingEventListener)
    }

    override fun removeDockingListener(paramIDockingEventListener: IDockingEventListener?) {
        delegate.removeDockingListener(paramIDockingEventListener)
    }

    override fun getDockingListeners(): MutableList<IDockingEventListener> {
        return delegate.getDockingListeners() as MutableList<IDockingEventListener>
    }

    override fun isDockingEnabled(): Boolean {
        return delegate.isDockingEnabled
    }

    override fun setDockingEnabled(paramBoolean: Boolean) {
        delegate.isDockingEnabled = paramBoolean
    }

    override fun setActive(paramBoolean: Boolean) {
        delegate.isActive = paramBoolean
    }

    override fun isActive(): Boolean {
        return delegate.isActive
    }

    override fun setActiveStateLocked(paramBoolean: Boolean) {
        delegate.isActiveStateLocked = paramBoolean
    }

    override fun isActiveStateLocked(): Boolean {
        return delegate.isActiveStateLocked
    }

    override fun setTabText(paramString: String?) {
        val str = tabText
        delegate.tabText = paramString
        firePropertyChange("tabText", str, paramString)    }

    override fun getTabText(): String {
        return delegate.tabText!!
    }

    override fun setDockbarIcon(paramIcon: Icon?) {
        val localIcon = dockbarIcon
        delegate.dockbarIcon = paramIcon
        firePropertyChange("dockbarIcon", localIcon, paramIcon)    }

    override fun getDockbarIcon(): Icon {
        val localIcon = delegate.dockbarIcon
        return localIcon!! ?: icon!!
    }

    override fun setTabIcon(paramIcon: Icon?) {
        val localIcon = tabIcon
        delegate.tabIcon = paramIcon
        firePropertyChange("tabIcon", localIcon, paramIcon)    }

    override fun getTabIcon(): Icon {
        val localIcon = delegate.tabIcon
        return localIcon ?: icon!!
    }
    override fun setRegionInset(paramString: String?, paramFloat: Float) {
        delegate.setRegionInset(paramString, paramFloat)
    }

    override fun getRegionInset(paramString: String?): Float {
        return delegate.getRegionInset(paramString)
    }

    override fun setSiblingSize(paramString: String?, paramFloat: Float) {
        delegate.setSiblingSize(paramString, paramFloat)
    }

    override fun getSiblingSize(paramString: String?): Float {
        return delegate.getSiblingSize(paramString)
    }

    override fun setTerritoryBlocked(paramString: String?, paramBoolean: Boolean) {
        delegate.setTerritoryBlocked(paramString, paramBoolean)
    }

    override fun isTerritoryBlocked(paramString: String?): Boolean {
        return delegate.isTerritoryBlocked(paramString)
    }

    override fun setDockableAcceptor(paramIDockableAcceptor: IDockableAcceptor?) {
        val localIDockableAcceptor: IDockableAcceptor? = dockableAcceptor
        delegate.dockableAcceptor = paramIDockableAcceptor
        firePropertyChange("dockableAcceptor", localIDockableAcceptor, paramIDockableAcceptor)
    }

    override fun getDockableAcceptor(): IDockableAcceptor {
        return delegate.dockableAcceptor!!
    }

    override fun setDragTheshold(paramFloat: Float) {
        delegate.setDragTheshold(paramFloat)
    }

    override fun getDragThreshold(): Float {
        return delegate.dragThreshold
    }

    override fun setPreviewSize(paramFloat: Float) {
        delegate.previewSize = paramFloat
    }

    override fun getPreviewSize(): Float {
        return delegate.previewSize
    }

    override fun isMouseMotionListenersBlockedWhileDragging(): Boolean {
        return delegate.isMouseMotionListenersBlockedWhileDragging
    }

    override fun setMouseMotionListenersBlockedWhileDragging(p0: Boolean) {
        delegate.isMouseMotionListenersBlockedWhileDragging = p0
    }

    override fun dockableRegistered(paramDockingEvent: DockingEvent?) {
        delegate.dockableRegistered(paramDockingEvent)
    }

    override fun dockableUnregistered(paramDockingEvent: DockingEvent?) {
        delegate.dockableUnregistered(paramDockingEvent)
    }

    override fun dockingCanceled(paramDockingEvent: DockingEvent?) {
        delegate.dockingCanceled(paramDockingEvent)
    }

    override fun dockingCompleted(paramDockingEvent: DockingEvent?) {
        delegate.dockingCompleted(paramDockingEvent)
        if (titlebar != null) titlebar!!.revalidate()
    }

    override fun dragStarted(paramDockingEvent: DockingEvent?) {
        delegate.dragStarted(paramDockingEvent)
    }

    override fun dropStarted(paramDockingEvent: DockingEvent?) {
        delegate.dropStarted(paramDockingEvent)
    }

    override fun undockingStarted(paramDockingEvent: DockingEvent?) {
        delegate.undockingStarted(paramDockingEvent)
    }

    override fun undockingCompleted(paramDockingEvent: DockingEvent?) {
        delegate.undockingCompleted(paramDockingEvent)
        clearButtonRollovers()
    }

    protected inner class ViewLayout : LayoutManager2, Serializable {
        override fun preferredLayoutSize(paramContainer: Container): Dimension {
            val localInsets: Insets = this@DockingView.getInsets()
            val localDimension1: Dimension
            localDimension1 =
                if (contentPane != null) contentPane!!.getPreferredSize() else paramContainer.size
            val localDimension2: Dimension
            localDimension2 =
                if (titlebar != null && titlebar!!.isVisible()) titlebar!!.getPreferredSize() else {
                    Dimension(0, 0)
                }
            return Dimension(
                Math.max(
                    localDimension1.width,
                    localDimension2.width
                ) + localInsets.left + localInsets.right,
                localDimension1.height + localDimension2.height + localInsets.top + localInsets.bottom
            )
        }

        override fun minimumLayoutSize(paramContainer: Container): Dimension {
            val localInsets: Insets = this@DockingView.getInsets()
            val localDimension1: Dimension
            localDimension1 =
                if (contentPane != null) contentPane!!.getMinimumSize() else paramContainer.size
            val localDimension2: Dimension
            localDimension2 =
                if (titlebar != null && titlebar!!.isVisible()) titlebar!!.getMinimumSize() else {
                    Dimension(0, 0)
                }
            return Dimension(
                Math.max(
                    localDimension1.width,
                    localDimension2.width
                ) + localInsets.left + localInsets.right,
                localDimension1.height + localDimension2.height + localInsets.top + localInsets.bottom
            )
        }

        override fun maximumLayoutSize(paramContainer: Container): Dimension {
            val localInsets: Insets = this@DockingView.getInsets()
            val localDimension2: Dimension
            localDimension2 =
                if (titlebar != null && titlebar!!.isVisible()) titlebar!!.getMaximumSize() else Dimension(
                    0,
                    0
                )
            val localDimension1: Dimension
            localDimension1 = if (contentPane != null) {
                contentPane!!.getMaximumSize()
            } else {
                Dimension(2147483647, 2147483647 - localInsets.top - localInsets.bottom - localDimension2.height - 1)
            }
            return Dimension(
                Math.min(
                    localDimension1.width,
                    localDimension2.width
                ) + localInsets.left + localInsets.right,
                localDimension1.height + localDimension2.height + localInsets.top + localInsets.bottom
            )
        }

        override fun layoutContainer(paramContainer: Container) {
            val localRectangle = paramContainer.bounds
            val localInsets: Insets = this@DockingView.getInsets()
            val i = localInsets.left
            var j = localInsets.top
            val k = localRectangle.width - localInsets.right - localInsets.left
            var m = localRectangle.height - localInsets.top - localInsets.bottom
            if (titlebar != null && titlebar!!.isVisible()) {
                val localDimension = titlebar!!.getPreferredSize()
                titlebar!!.setBounds(0, 0, k, localDimension.height)
                j += localDimension.height
                m -= j
            }
            if (contentPane != null) contentPane!!.setBounds(i, j, k, m)
        }

        override fun addLayoutComponent(paramString: String, paramComponent: Component) {}
        override fun removeLayoutComponent(paramComponent: Component) {}
        override fun addLayoutComponent(paramComponent: Component, paramObject: Any) {}
        override fun getLayoutAlignmentX(paramContainer: Container): Float {
            return 0.0f
        }

        override fun getLayoutAlignmentY(paramContainer: Container): Float {
            return 0.0f
        }

        override fun invalidateLayout(paramContainer: Container) {}

//        companion object {
//            private const val serialVersionUID = -785289772450064051L
//        }
    }

    init {
        delegate = DockableComponentWrapper(this, paramString1, paramString3, false)
        contentPane = DockingViewContentPane()
        add(contentPane)
        setTitlebar(DockingViewTitleBar())
        paintContentPaneBackground = true
        setContentPane(createContentPane())
        super.setLayout(createLayout())
        title = paramString2 ?: ""
        addHierarchyListener(object : HierarchyListener {
            override fun hierarchyChanged(paramAnonymousHierarchyEvent: HierarchyEvent) {
                clearButtonRollovers()
            }
        })
        updateUI()
        //        DockingManager.registerDockable(this);
    }
}