@file:Suppress("JAVA_MODULE_DOES_NOT_EXPORT_PACKAGE")
/*
 * Copyright (c) 2007, Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.docking.swing.plaf.basic

import com.macrofocus.docking.swing.plaf.DockingButtonUI
import sun.swing.SwingUtilities2
import java.awt.*
import javax.swing.*
import javax.swing.plaf.ComponentUI
import javax.swing.plaf.basic.BasicButtonListener
import javax.swing.plaf.basic.BasicGraphicsUtils
import javax.swing.plaf.basic.BasicHTML
import javax.swing.text.View

/**
 * Created by IntelliJ IDEA.
 * User: luc
 * Date: Feb 16, 2007
 * Time: 10:54:16 AM
 */
@Suppress("ACCIDENTAL_OVERRIDE")
open class BasicDockingButtonUI : DockingButtonUI() {
    // Visual constants
    // NOTE: This is not used or set any where. Were we allowed to remove
    // fields, this would be removed.
    protected var defaultTextIconGap = 0

    // Amount to offset text, the value of this comes from
    // defaultTextShiftOffset once setTextShiftOffset has been invoked.
    protected var textShiftOffset = 0
        private set

    // Value that is set in shiftOffset once setTextShiftOffset has been
    // invoked. The value of this comes from the defaults table.
    protected var defaultTextShiftOffset = 0
    protected val propertyPrefix: String
        protected get() = Companion.propertyPrefix

    // ********************************
    //          Install PLAF
    // ********************************
    override fun installUI(c: JComponent) {
        installDefaults(c as AbstractButton)
        installListeners(c)
        installKeyboardActions(c)
        BasicHTML.updateRenderer(c, c.text)
    }

    protected open fun installDefaults(b: AbstractButton?) {
        // load shared instance defaults
        val pp = propertyPrefix
        defaultTextShiftOffset = UIManager.getInt(pp + "textShiftOffset")
        //
//        // set the following defaults on the button
//        if (b.isContentAreaFilled()) {
//            LookAndFeel.installProperty(b, "opaque", Boolean.TRUE);
//        } else {
//            LookAndFeel.installProperty(b, "opaque", Boolean.FALSE);
//        }
//
//        if(b.getMargin() == null || (b.getMargin() instanceof UIResource)) {
//            b.setMargin(UIManager.getInsets(pp + "margin"));
//        }
//
        LookAndFeel.installColorsAndFont(
            b, pp + "background",
            pp + "foreground", pp + "font"
        )
        //        LookAndFeel.installBorder(b, pp + "border");
//
//        Object rollover = UIManager.get(pp + "rollover");
//        if (rollover != null) {
//            LookAndFeel.installProperty(b, "rolloverEnabled", rollover);
//        }
//
//        LookAndFeel.installProperty(b, "iconTextGap", new Integer(4));
    }

    protected fun installListeners(b: AbstractButton) {
        val listener = createButtonListener(b)
        if (listener != null) {
            b.addMouseListener(listener)
            b.addMouseMotionListener(listener)
            b.addFocusListener(listener)
            b.addPropertyChangeListener(listener)
            b.addChangeListener(listener)
        }
    }

    protected fun installKeyboardActions(b: AbstractButton) {
        val listener = getButtonListener(b)
        listener?.installKeyboardActions(b)
    }

    // ********************************
    //         Uninstall PLAF
    // ********************************
    override fun uninstallUI(c: JComponent) {
        uninstallKeyboardActions(c as AbstractButton)
        uninstallListeners(c)
        uninstallDefaults(c)
        BasicHTML.updateRenderer(c, "")
    }

    protected fun uninstallKeyboardActions(b: AbstractButton) {
        val listener = getButtonListener(b)
        listener?.uninstallKeyboardActions(b)
    }

    protected fun uninstallListeners(b: AbstractButton) {
        val listener = getButtonListener(b)
        if (listener != null) {
            b.removeMouseListener(listener)
            b.removeMouseMotionListener(listener)
            b.removeFocusListener(listener)
            b.removeChangeListener(listener)
            b.removePropertyChangeListener(listener)
        }
    }

    protected open fun uninstallDefaults(b: AbstractButton?) {
        LookAndFeel.uninstallBorder(b)
    }

    // ********************************
    //        Create Listeners
    // ********************************
    protected fun createButtonListener(b: AbstractButton?): BasicButtonListener {
        return BasicButtonListener(b)
    }

    fun getDefaultTextIconGap(b: AbstractButton?): Int {
        return defaultTextIconGap
    }

    // ********************************
    //          Paint Methods
    // ********************************
    override fun paint(g: Graphics, c: JComponent) {
        val b = c as AbstractButton
        val model = b.model
        val text = layout(
            b, SwingUtilities2.getFontMetrics(b, g),
            b.width, b.height
        )
        clearTextShiftOffset()

        // perform UI specific press action, e.g. Windows L&F shifts text
        if (model.isArmed && model.isPressed) {
            paintButtonPressed(g, b)
        }

        // Paint the Icon
        if (b.icon != null) {
            paintIcon(g, c, iconRect)
        }
        if (text != null && text != "") {
            val v = c.getClientProperty(BasicHTML.propertyKey) as View?
            if (v != null) {
                v.paint(g, textRect)
            } else {
                paintText(g, b, textRect, text)
            }
        }
        if (b.isFocusPainted && b.hasFocus()) {
            // paint UI specific focus
            paintFocus(g, b, viewRect, textRect, iconRect)
        }
    }

    protected fun paintIcon(g: Graphics?, c: JComponent, iconRect: Rectangle) {
        val b = c as AbstractButton
        val model = b.model
        var icon = b.icon
        var tmpIcon: Icon? = null
        if (icon == null) {
            return
        }
        var selectedIcon: Icon? = null

        /* the fallback icon should be based on the selected state */if (model.isSelected) {
            selectedIcon = b.selectedIcon
            if (selectedIcon != null) {
                icon = selectedIcon
            }
        }
        if (!model.isEnabled) {
            if (model.isSelected) {
                tmpIcon = b.disabledSelectedIcon
                if (tmpIcon == null) {
                    tmpIcon = selectedIcon
                }
            }
            if (tmpIcon == null) {
                tmpIcon = b.disabledIcon
            }
        } else if (model.isPressed && model.isArmed) {
            tmpIcon = b.pressedIcon
            if (tmpIcon != null) {
                // revert back to 0 offset
                clearTextShiftOffset()
            }
        } else if (b.isRolloverEnabled && model.isRollover) {
            if (model.isSelected) {
                tmpIcon = b.rolloverSelectedIcon
                if (tmpIcon == null) {
                    tmpIcon = selectedIcon
                }
            }
            if (tmpIcon == null) {
                tmpIcon = b.rolloverIcon
            }
        }
        if (tmpIcon != null) {
            icon = tmpIcon
        }
        if (model.isPressed && model.isArmed) {
            icon.paintIcon(
                c, g, iconRect.x + textShiftOffset,
                iconRect.y + textShiftOffset
            )
        } else {
            icon.paintIcon(c, g, iconRect.x, iconRect.y)
        }
    }

    /**
     * As of Java 2 platform v 1.4 this method should not be used or overriden.
     * Use the paintText method which takes the AbstractButton argument.
     */
    protected fun paintText(g: Graphics, c: JComponent, textRect: Rectangle, text: String?) {
        val b = c as AbstractButton
        val model = b.model
        val fm = SwingUtilities2.getFontMetrics(c, g)
        val mnemonicIndex = b.displayedMnemonicIndex

        /* Draw the Text */if (model.isEnabled) {
            /*** paint the text normally  */
            g.color = b.foreground
            SwingUtilities2.drawStringUnderlineCharAt(
                c, g, text, mnemonicIndex,
                textRect.x + textShiftOffset,
                textRect.y + fm.ascent + textShiftOffset
            )
        } else {
            /*** paint the text disabled  */
            g.color = b.background.brighter()
            SwingUtilities2.drawStringUnderlineCharAt(
                c, g, text, mnemonicIndex,
                textRect.x, textRect.y + fm.ascent
            )
            g.color = b.background.darker()
            SwingUtilities2.drawStringUnderlineCharAt(
                c, g, text, mnemonicIndex,
                textRect.x - 1, textRect.y + fm.ascent - 1
            )
        }
    }

    /**
     * Method which renders the text of the current button.
     *
     *
     *
     * @param g        Graphics context
     * @param b        Current button to render
     * @param textRect Bounding rectangle to render the text.
     * @param text     String to render
     * @since 1.4
     */
    protected fun paintText(g: Graphics, b: AbstractButton, textRect: Rectangle, text: String?) {
        paintText(g, b as JComponent, textRect, text)
    }

    // Method signature defined here overriden in subclasses.
    // Perhaps this class should be abstract?
    protected fun paintFocus(
        g: Graphics?, b: AbstractButton?,
        viewRect: Rectangle?, textRect: Rectangle?, iconRect: Rectangle?
    ) {
    }

    protected fun paintButtonPressed(g: Graphics?, b: AbstractButton?) {}
    protected fun clearTextShiftOffset() {
        textShiftOffset = 0
    }

    protected fun setTextShiftOffset() {
        textShiftOffset = defaultTextShiftOffset
    }

    // ********************************
    //          Layout Methods
    // ********************************
    override fun getMinimumSize(c: JComponent): Dimension {
        val d = getPreferredSize(c)
        val v = c.getClientProperty(BasicHTML.propertyKey) as View?
        if (v != null) {
            d.width -= (v.getPreferredSpan(View.X_AXIS) - v.getMinimumSpan(View.X_AXIS)).toInt()
        }
        return d
    }

    override fun getPreferredSize(c: JComponent): Dimension {
        val b = c as AbstractButton
        return BasicGraphicsUtils.getPreferredButtonSize(b, b.iconTextGap)
    }

    override fun getMaximumSize(c: JComponent): Dimension {
        val d = getPreferredSize(c)
        val v = c.getClientProperty(BasicHTML.propertyKey) as View?
        if (v != null) {
            d.width += (v.getMaximumSpan(View.X_AXIS) - v.getPreferredSpan(View.X_AXIS)).toInt()
        }
        return d
    }

    /**
     * Returns the baseline.
     *
     * @throws NullPointerException     {@inheritDoc}
     * @throws IllegalArgumentException {@inheritDoc}
     * @see JComponent.getBaseline
     * @since 1.6
     */
    override fun getBaseline(c: JComponent, width: Int, height: Int): Int {
        super.getBaseline(c, width, height)
        val b = c as AbstractButton
        val text = b.text
        if (text == null || "" == text) {
            return -1
        }
        val fm = b.getFontMetrics(b.font)
        layout(b, fm, width, height)
        return getBaseline(
            b, textRect.y, fm.ascent,
            textRect.width, textRect.height
        )
    }

    private fun layout(
        b: AbstractButton, fm: FontMetrics,
        width: Int, height: Int
    ): String {
        val i = b.insets
        viewRect.x = i.left
        viewRect.y = i.top
        viewRect.width = width - (i.right + viewRect.x)
        viewRect.height = height - (i.bottom + viewRect.y)
        textRect.height = 0
        textRect.width = textRect.height
        textRect.y = textRect.width
        textRect.x = textRect.y
        iconRect.height = 0
        iconRect.width = iconRect.height
        iconRect.y = iconRect.width
        iconRect.x = iconRect.y

        // layout the text and icon
        return SwingUtilities.layoutCompoundLabel(
            b, fm, b.text, b.icon,
            b.verticalAlignment, b.horizontalAlignment,
            b.verticalTextPosition, b.horizontalTextPosition,
            viewRect, iconRect, textRect,
            if (b.text == null) 0 else b.iconTextGap
        )
    }

    /**
     * Returns the ButtonListener for the passed in Button, or null if one
     * could not be found.
     */
    private fun getButtonListener(b: AbstractButton): BasicButtonListener? {
        val listeners = b.mouseMotionListeners
        if (listeners != null) {
            for (counter in listeners.indices) {
                if (listeners[counter] is BasicButtonListener) {
                    return listeners[counter] as BasicButtonListener
                }
            }
        }
        return null
    }

    companion object {
        // Shared UI object
        private val buttonUI: DockingButtonUI = BasicDockingButtonUI()
        private const val propertyPrefix = "Button" + "."

        // ********************************
        //          Create PLAF
        // ********************************
        @JvmStatic fun createUI(c: JComponent?): ComponentUI {
            return buttonUI
        }

        /* These rectangles/insets are allocated once for all
     * ButtonUI.paint() calls.  Re-using rectangles rather than
     * allocating them in each paint call substantially reduced the time
     * it took paint to run.  Obviously, this method can't be re-entered.
     */
        private val viewRect = Rectangle()
        private val textRect = Rectangle()
        private val iconRect = Rectangle()
        fun getBaseline(
            c: JComponent, y: Int, ascent: Int,
            w: Int, h: Int
        ): Int {
            val view = c.getClientProperty(BasicHTML.propertyKey) as View?
            if (view != null) {
                val baseline = BasicHTML.getHTMLBaseline(view, w, h)
                return if (baseline < 0) {
                    baseline
                } else y + baseline
            }
            return y + ascent
        }
    }
}