/*
 * Copyright (c) 2016 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.docking.swing.splitter

import com.macrofocus.docking.splitter.ThreeComponentsSplitter
import java.awt.*
import java.awt.event.MouseAdapter
import java.awt.event.MouseEvent
import javax.swing.*

/**
 * User: luc
 * Date: Jun 1, 2006
 * Time: 9:43:24 AM
 */
class SwingThreeComponentsSplitter @JvmOverloads constructor(private var myVerticalSplit: Boolean = false) : JPanel(), ThreeComponentsSplitter<JComponent> {
    private var myDividerWidth: Int
    var isHonorMinimumSize = false
        private set
    private var myFirstDivider: Divider?
    private var myLastDivider: Divider?
    private var myFirstComponent: JComponent? = null
    private var myInnerComponent: JComponent? = null
    private var myLastComponent: JComponent? = null
    private var myFirstSize = 350.0
    private var myLastSize = 10.0
    private var myShowDividerControls = false
    override var isRelativeResize = true

    protected inner class Divider(isFirst: Boolean) : JPanel(GridBagLayout()) {
        fun setOrientation(isVerticalSplit: Boolean) {
            removeAll()
            if (!myShowDividerControls) {
                return
            } else {
                val xMask = if (isVerticalSplit) 1 else 0
                val yMask = if (isVerticalSplit) 0 else 1
                val splitGlue =
                    SwingThreeComponentsSplitter::class.java.getResource(if (isVerticalSplit) "splitGlueV.png" else "splitGlueH.png")
                if (splitGlue == null) {
                    System.err.println(if (isVerticalSplit) "splitGlueV.png" else "splitGlueH.png" + " couldn't be found")
                    return
                }
                val glueIcon: Icon = ImageIcon(splitGlue)
                val glueFill = if (isVerticalSplit) 3 else 2
                add(
                    JLabel(glueIcon),
                    GridBagConstraints(
                        0,
                        0,
                        1,
                        1,
                        0.0,
                        0.0,
                        if (isVerticalSplit) 13 else 11,
                        glueFill,
                        Insets(0, 0, 0, 0),
                        0,
                        0
                    )
                )
                val split =
                    SwingThreeComponentsSplitter::class.java.getResource(if (isVerticalSplit) "splitDown.png" else "splitRight.png")
                if (split == null) {
                    System.err.println(if (isVerticalSplit) "splitDown.png" else "splitRight.png" + " couldn't be found")
                    return
                }
                val splitDownlabel = JLabel(ImageIcon(split))
                splitDownlabel.setCursor(Cursor.getPredefinedCursor(12))
                //                splitDownlabel.setToolTipText(isVerticalSplit ? UIBundle.message("splitter.down.tooltip.text", new Object[0]) : UIBundle.message("splitter.right.tooltip.text", new Object[0]));
                splitDownlabel.addMouseListener(object : MouseAdapter() {
                    override fun mouseClicked(e: MouseEvent) {
                        if (myInnerComponent != null) {
                            if (myFirstComponent != null && myFirstComponent!!.isVisible()) {
                                val income: Int =
                                    if (myVerticalSplit) myInnerComponent!!.getHeight() else myInnerComponent!!.getWidth()
                                if (myIsFirst) firstSize = myFirstSize + income else lastSize = myLastSize + income
                            } else {
                                if (myIsFirst) firstSize = getMinSize(myFirstComponent).toDouble() else lastSize =
                                    getMinSize(myLastComponent).toDouble()
                            }
                        } else {
                            val income: Int =
                                if (myVerticalSplit) myLastComponent!!.getHeight() else myLastComponent!!.getWidth()
                            if (myIsFirst) firstSize = myFirstSize + income
                        }
                    }
                })
                add(
                    splitDownlabel,
                    GridBagConstraints(
                        if (isVerticalSplit) 1 else 0,
                        if (isVerticalSplit) 0 else 5,
                        1,
                        1,
                        0.0,
                        0.0,
                        10,
                        0,
                        Insets(0, 0, 0, 0),
                        0,
                        0
                    )
                )
                add(
                    JLabel(glueIcon),
                    GridBagConstraints(2 * xMask, 2 * yMask, 1, 1, 0.0, 0.0, 10, glueFill, Insets(0, 0, 0, 0), 0, 0)
                )
                val splitCenterlabel =
                    JLabel(ImageIcon(SwingThreeComponentsSplitter::class.java.getResource(if (isVerticalSplit) "splitCenterV.png" else "splitCenterH.png")))
                splitCenterlabel.setCursor(Cursor.getPredefinedCursor(12))
                //                splitCenterlabel.setToolTipText(UIBundle.message("splitter.center.tooltip.text", new Object[0]));
                splitCenterlabel.addMouseListener(object : MouseAdapter() {
                    override fun mouseClicked(e: MouseEvent) {
                        center()
                    }
                })
                add(
                    splitCenterlabel,
                    GridBagConstraints(3 * xMask, 3 * yMask, 1, 1, 0.0, 0.0, 10, 0, Insets(0, 0, 0, 0), 0, 0)
                )
                add(
                    JLabel(glueIcon),
                    GridBagConstraints(4 * xMask, 4 * yMask, 1, 1, 0.0, 0.0, 10, glueFill, Insets(0, 0, 0, 0), 0, 0)
                )
                val splitUpLabel =
                    JLabel(ImageIcon(SwingThreeComponentsSplitter::class.java.getResource(if (isVerticalSplit) "splitUp.png" else "splitLeft.png")))
                splitUpLabel.setCursor(Cursor.getPredefinedCursor(12))
                //                splitUpLabel.setToolTipText(isVerticalSplit ? UIBundle.message("splitter.up.tooltip.text", new Object[0]) : UIBundle.message("splitter.left.tooltip.text", new Object[0]));
                splitUpLabel.addMouseListener(object : MouseAdapter() {
                    override fun mouseClicked(e: MouseEvent) {
                        if (myInnerComponent != null) {
                            if (myFirstComponent != null && myFirstComponent!!.isVisible()) {
                                if (myIsFirst) firstSize = getMinSize(myFirstComponent).toDouble() else lastSize =
                                    getMinSize(myLastComponent).toDouble()
                            } else {
                                val income: Int =
                                    if (myVerticalSplit) myInnerComponent!!.getHeight() else myInnerComponent!!.getWidth()
                                if (myIsFirst) firstSize = myFirstSize + income else lastSize = myLastSize + income
                            }
                        } else {
                            if (myIsFirst) firstSize = getMinSize(myFirstComponent).toDouble()
                        }
                    }
                })
                add(
                    splitUpLabel,
                    GridBagConstraints(
                        if (isVerticalSplit) 5 else 0,
                        if (isVerticalSplit) 0 else 1,
                        1,
                        1,
                        0.0,
                        0.0,
                        10,
                        0,
                        Insets(0, 0, 0, 0),
                        0,
                        0
                    )
                )
                add(
                    JLabel(glueIcon),
                    GridBagConstraints(
                        6 * xMask,
                        6 * yMask,
                        1,
                        1,
                        0.0,
                        0.0,
                        if (isVerticalSplit) 17 else 15,
                        glueFill,
                        Insets(0, 0, 0, 0),
                        0,
                        0
                    )
                )
                return
            }
        }

        private fun center() {
            if (myInnerComponent != null) {
                if (myFirstComponent != null && myFirstComponent!!.isVisible()) {
                    val total: Int =
                        myFirstSize.toInt() + if (myVerticalSplit) myInnerComponent!!.getHeight() else myInnerComponent!!.getWidth()
                    if (myIsFirst) firstSize = (total / 2).toDouble() else lastSize = (total / 2).toDouble()
                } else {
                    val total: Int =
                        myLastSize.toInt() + if (myVerticalSplit) myInnerComponent!!.getHeight() else myInnerComponent!!.getWidth()
                    if (myIsFirst) firstSize = (total / 2).toDouble() else lastSize = (total / 2).toDouble()
                }
            } else {
                val total: Int =
                    myFirstSize.toInt() + if (myVerticalSplit) myLastComponent!!.getHeight() else myLastComponent!!.getWidth()
                if (myIsFirst) firstSize = (total / 2).toDouble()
            }
        }

        override fun processMouseMotionEvent(e: MouseEvent) {
            super.processMouseMotionEvent(e)
            if (506 == e.id) {
                myDragging = true
                setCursor(if (vertical) Cursor.getPredefinedCursor(9) else Cursor.getPredefinedCursor(11))
                myPoint = SwingUtilities.convertPoint(this, e.point, this@SwingThreeComponentsSplitter)
                if (vertical) {
                    if (getHeight() > 0) if (myIsFirst) firstSize =
                        Math.max(getMinSize(myFirstComponent), myPoint!!.y).toDouble() else lastSize = Math.max(
                        getMinSize(myLastComponent),
                        this@SwingThreeComponentsSplitter.getHeight() - myPoint!!.y - dividerWidth
                    ).toDouble()
                } else if (getWidth() > 0) {
                    if (myIsFirst) {
                        firstSize = Math.max(getMinSize(myFirstComponent), myPoint!!.x).toDouble()
                    } else {
                        lastSize = Math.max(
                            getMinSize(myLastComponent),
                            this@SwingThreeComponentsSplitter.getWidth() - myPoint!!.x - dividerWidth
                        ).toDouble()
                    }
                }
                doLayout()
            }
        }

        private fun getMinSize(component: JComponent?): Int {
            return if (isHonorMinimumSize && component != null && myFirstComponent != null && myFirstComponent!!.isVisible() && myLastComponent != null && myLastComponent!!.isVisible()) {
                if (vertical) component.getMinimumSize().height else component.getMinimumSize().width
            } else {
                0
            }
        }

        override fun processMouseEvent(e: MouseEvent) {
            super.processMouseEvent(e)
            when (e.id) {
                503 -> {
                }
                504 -> setCursor(if (vertical) Cursor.getPredefinedCursor(9) else Cursor.getPredefinedCursor(11))
                505 -> if (!myDragging) setCursor(Cursor.getPredefinedCursor(0))
                501 -> setCursor(if (vertical) Cursor.getPredefinedCursor(9) else Cursor.getPredefinedCursor(11))
                502 -> {
                    myDragging = false
                    myPoint = null
                }
                500 -> if (e.clickCount == 2) center()
                else -> {
                }
            }
        }

        protected var myDragging = false
        protected var myPoint: Point? = null
        private val myIsFirst: Boolean

        init {
            setFocusable(false)
            enableEvents(48L)
            myIsFirst = isFirst
            setOrientation(myVerticalSplit)
            setOpaque(false)
        }
    }

    override fun setShowDividerControls(showDividerControls: Boolean) {
        myShowDividerControls = showDividerControls
        vertical = myVerticalSplit
    }

    fun setHonorComponentsMinimumSize(honorMinimumSize: Boolean) {
        isHonorMinimumSize = honorMinimumSize
    }

    public override fun isVisible(): Boolean {
        return super.isVisible() && (firstVisible() || innerVisible() || lastVisible())
    }

    private fun lastVisible(): Boolean {
        return myLastComponent != null && myLastComponent!!.isVisible()
    }

    private fun innerVisible(): Boolean {
        return myInnerComponent != null && myInnerComponent!!.isVisible()
    }

    private fun firstVisible(): Boolean {
        return myFirstComponent != null && myFirstComponent!!.isVisible()
    }

    private fun visibleDividersCount(): Int {
        var count = 0
        if (firstDividerVisible()) count++
        if (lastDividerVisible()) count++
        return count
    }

    private fun firstDividerVisible(): Boolean {
        return firstVisible() && innerVisible() || firstVisible() && lastVisible() && !innerVisible()
    }

    private fun lastDividerVisible(): Boolean {
        return innerVisible() && lastVisible()
    }


    override fun getMinimumSize(): Dimension {
         return if (isHonorMinimumSize) {
            val dividerWidth = dividerWidth
            val firstSize = if (myFirstComponent == null) Dimension(0, 0) else myFirstComponent!!.getMinimumSize()
            val lastSize = if (myLastComponent == null) Dimension(0, 0) else myLastComponent!!.getMinimumSize()
            val innerSize = if (myInnerComponent == null) Dimension(0, 0) else myInnerComponent!!.getMinimumSize()
            if (vertical) {
                val width = Math.max(firstSize.width, Math.max(lastSize.width, innerSize.width))
                var height = visibleDividersCount() * dividerWidth
                height += firstSize.height
                height += lastSize.height
                height += innerSize.height
                Dimension(width, height)
            } else {
                val heigth = Math.max(firstSize.height, Math.max(lastSize.height, innerSize.height))
                var width = visibleDividersCount() * dividerWidth
                width += firstSize.width
                width += lastSize.width
                width += innerSize.width
                Dimension(width, heigth)
            }
        } else {
            super.getMinimumSize()
        }
    }

    override fun doLayout() {
        val width: Int = getWidth()
        val height: Int = getHeight()
        if (width > 0 && height > 0) {
            val firstRect = Rectangle()
            val firstDividerRect = Rectangle()
            val lastDividerRect = Rectangle()
            val lastRect = Rectangle()
            val innerRect = Rectangle()
            val componentSize = if (vertical) height else width
            var dividerWidth = dividerWidth
            val dividersCount = visibleDividersCount()
            var firstCompontSize: Double
            var lastComponentSize: Double
            var innerComponentSize: Double
            if (componentSize <= dividersCount * dividerWidth) {
                firstCompontSize = 0.0
                lastComponentSize = 0.0
                innerComponentSize = 0.0
                dividerWidth = componentSize
            } else {
                firstCompontSize = firstSizeIfVisible
                lastComponentSize = lastSizeIfVisible
                val sizeLack =
                    (firstCompontSize + lastComponentSize).toInt() - (componentSize - dividersCount * dividerWidth)
                if (sizeLack > 0) {
                    firstCompontSize -= sizeLack.toDouble()
                    innerComponentSize = 0.0
                } else {
                    innerComponentSize =
                        componentSize - dividersCount * dividerWidth - firstSizeIfVisible - lastSizeIfVisible
                }
                if (!innerVisible()) {
                    if (lastVisible()) {
                        lastComponentSize += innerComponentSize
                    } else {
                        firstCompontSize += innerComponentSize
                    }
                    innerComponentSize = 0.0
                }
            }
            if (vertical) {
                // Vertical
                var space = firstCompontSize.toInt()
                firstRect.setBounds(0, 0, width, firstCompontSize.toInt())
                if (firstDividerVisible()) {
                    firstDividerRect.setBounds(0, space, width, dividerWidth)
                    space += dividerWidth
                }
                if (innerVisible()) {
                    innerRect.setBounds(0, space, width, innerComponentSize.toInt())
                    space += innerComponentSize.toInt()
                }
                if (lastDividerVisible()) {
                    lastDividerRect.setBounds(0, space, width, dividerWidth)
                    space += dividerWidth
                }
                lastRect.setBounds(0, space, width, lastComponentSize.toInt())
            } else {
                // Horizontal
                var space = firstCompontSize.toInt()
                firstRect.setBounds(0, 0, firstCompontSize.toInt(), height)
                if (firstDividerVisible()) {
                    firstDividerRect.setBounds(space, 0, dividerWidth, height)
                    space += dividerWidth
                }
                if (innerVisible()) {
                    innerRect.setBounds(space, 0, innerComponentSize.toInt(), height)
                    space += innerComponentSize.toInt()
                }
                if (lastDividerVisible()) {
                    lastDividerRect.setBounds(space, 0, dividerWidth, height)
                    space += dividerWidth
                }
                lastRect.setBounds(space, 0, lastComponentSize.toInt(), height)
            }
            myFirstDivider!!.setVisible(firstDividerVisible())
            myFirstDivider!!.setBounds(firstDividerRect)
            myFirstDivider!!.doLayout()
            myLastDivider!!.setVisible(lastDividerVisible())
            myLastDivider!!.setBounds(lastDividerRect)
            myLastDivider!!.doLayout()
            if (myFirstComponent != null) {
                myFirstComponent!!.setBounds(firstRect)
                myFirstComponent!!.revalidate()
            }
            if (myInnerComponent != null) {
                myInnerComponent!!.setBounds(innerRect)
                myInnerComponent!!.revalidate()
            }
            if (myLastComponent != null) {
                myLastComponent!!.setBounds(lastRect)
                myLastComponent!!.revalidate()
            }
        }
    }

    var dividerWidth: Int
        get() = myDividerWidth
        set(width) {
            require(width >= 0) { "Wrong divider width: $width" }
            if (myDividerWidth != width) {
                myDividerWidth = width
                doLayout()
                repaint()
            }
        }
    override var vertical: Boolean
        get() = myVerticalSplit
        set(verticalSplit) {
            myVerticalSplit = verticalSplit
            myFirstDivider!!.setOrientation(verticalSplit)
            myLastDivider!!.setOrientation(verticalSplit)
            doLayout()
            repaint()
        }

    override fun setComponents(first: JComponent?, inner: JComponent?, last: JComponent?) {
        firstComponent = first
        innerComponent = inner
        lastComponent = last
    }

    override var firstComponent: JComponent?
        get() = myFirstComponent
        set(component) {
            if (myFirstComponent !== component) {
                if (myFirstComponent != null) remove(myFirstComponent)
                myFirstComponent = component
                if (myFirstComponent != null) {
                    super.add(myFirstComponent)
                    myFirstComponent!!.invalidate()
                }
            }
        }
    override var lastComponent: JComponent?
        get() = myLastComponent
        set(component) {
            if (myLastComponent !== component) {
                if (myLastComponent != null) remove(myLastComponent)
                myLastComponent = component
                if (myLastComponent != null) {
                    super.add(myLastComponent)
                    myLastComponent!!.invalidate()
                }
            }
        }
    override var innerComponent: JComponent?
        get() = myInnerComponent
        set(component) {
            if (myInnerComponent !== component) {
                if (myInnerComponent != null) remove(myInnerComponent)
                myInnerComponent = component
                if (myInnerComponent != null) {
                    super.add(myInnerComponent)
                    myInnerComponent!!.invalidate()
                }
                SwingUtilities.invokeLater(Runnable {
                    if (myInnerComponent != null) {
                        SwingUtilities.updateComponentTreeUI(myInnerComponent)
                    }
                    revalidate()
                    repaint()
                })
            }
        }
    override var firstSize: Double
        get() = myFirstSize
        set(size) {
            if (myFirstSize != size) {
                myFirstSize = size
                doLayout()
                repaint()
            }
        }
    override var lastSize: Double
        get() = myLastSize
        set(size) {
            if (myLastSize != size) {
                myLastSize = size
                doLayout()
                repaint()
            }
        }
    override val firstSizeIfVisible: Double
        get() = if (firstVisible()) myFirstSize else 0.0
    override val lastSizeIfVisible: Double
        get() = if (lastVisible()) myLastSize else 0.0

    override fun setBounds(x: Int, y: Int, width: Int, height: Int) {
        if (isRelativeResize) {
            val currentHeight: Int = getHeight()
            val currentWidth: Int = getWidth()
            var firstSize = myFirstSize.toInt()
            if (currentWidth > 0 && currentHeight > 0) {
                firstSize = if (myVerticalSplit) {
                    firstSize * height / currentHeight
                } else {
                    firstSize * width / currentWidth
                }
            }
            var lastSize = myLastSize.toInt()
            if (currentWidth > 0 && currentHeight > 0) {
                lastSize = if (myVerticalSplit) {
                    lastSize * height / currentHeight
                } else {
                    lastSize * width / currentWidth
                }
            }
            super.setBounds(x, y, width, height)
            firstSize = firstSize.toDouble().toInt()
            lastSize = lastSize.toDouble().toInt()
        } else {
            super.setBounds(x, y, width, height)
        }
    }

    override val nativeComponent: JComponent
        get() = this

    fun dispose() {
        myFirstDivider = null
        myLastDivider = null
        myFirstComponent = null
        myInnerComponent = null
        myLastComponent = null
    }

    override fun setWidgetHidden(widget: JComponent, hidden: Boolean) {}

    init {
        myFirstDivider = Divider(true)
        myLastDivider = Divider(false)
        myDividerWidth = 8
        setOpaque(false)
        super.add(myFirstDivider)
        super.add(myLastDivider)
    }
}