/*
 * Copyright (c) 2016 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.docking.swing

import com.macrofocus.common.helper.WeakActionListeners
import com.macrofocus.docking.*
import java.awt.*
import java.awt.event.*
import java.util.ArrayList
import javax.swing.*

/**
 * User: luc
 * Date: May 31, 2006
 * Time: 10:02:25 PM
 */
class SwingDockingBar(private val container: DockingContainer, orientation: DockingAnchor) : JPanel(), DockingBar<JComponent> {
    override val dockables: MutableList<Dockable<JComponent>>
    private val dockingButtons: MutableMap<Dockable<*>, DockingButton>
    private val dockingStates: MutableMap<Dockable<*>, DockingState>
    private val orientation: DockingAnchor
    private var enterTimer: Timer?
    private var insideTimerAction: InsideTimerAction?
    private val actionListeners: WeakActionListeners = WeakActionListeners()

    override fun attach(dockable: Dockable<JComponent>) {
        dockable.dockingBar = this
        val first = dockables.isEmpty()
        dockables.add(dockable)
        val button = DockingButton(orientation)
        button.text = dockable.shortTitle
        if (dockable is IconDockable<*>) {
            button.icon = (dockable as IconDockable<*>).icon
        }
        button.toolTipText = dockable.description
        val popupMenu = JPopupMenu()
        popupMenu.add(JMenuItem(object : AbstractAction("Float") {
            override fun actionPerformed(e: ActionEvent) {
                setFloating(dockable, true)
            }
        }))
        button.componentPopupMenu = popupMenu
        dockingButtons[dockable] = button
        val state = DockingState()
        dockingStates[dockable] = state
        button.addActionListener(actionListeners.create(object : ActionListener {
            override fun actionPerformed(e: ActionEvent) {
                setActive(dockable, button.isSelected)
            }
        }, button))
        button.addMouseListener(object : MouseAdapter() {
            override fun mouseEntered(e: MouseEvent) {
                insideTimerAction!!.setDockable(dockable)
                enterTimer!!.start()
            }

            override fun mouseExited(e: MouseEvent) {
//                setActive(dockable, false);
                enterTimer!!.stop()
            }
        })
        container.layout = OverlayLayout(container)
        if (first) {
            setActive(dockable, true)
        }
        update()
    }

    override fun detach(dockable: Dockable<JComponent>) {
        setActive(dockable, false)
        dockables.remove(dockable)
        dockingButtons.remove(dockable)
        dockingStates.remove(dockable)
        update()
    }

    override fun removeAll() {
        for (dockable in dockables) {
            setActive(dockable, false)
            dockingButtons.remove(dockable)
            dockingStates.remove(dockable)
        }
        dockables.clear()
        update()
    }

    override fun isActive(dockable: Dockable<JComponent>): Boolean {
        val state: DockingState = dockingStates[dockable]!!
        return state.isActive
    }

    override fun setActive(dockable: Dockable<JComponent>, active: Boolean) {
        val state: DockingState = dockingStates[dockable]!!
        if (!state.isFloating) {
            for (d in dockables) {
                val s: DockingState = dockingStates[d]!!
                if (s.isActive) {
                    s.isActive = d === dockable && active
                    val button = dockingButtons[d]!!
                    button.setSelected(d === dockable && active)
                    container.remove(d.component)
                }
            }
            if (active) {
                state.isActive = true
                container.add(dockable.component)
                container.validate()
                container.isVisible = true
                val button = dockingButtons[dockable]!!
                button.setSelected(true)
                container.repaint()
            } else {
                container.isVisible = false
                container.revalidate()
                container.repaint()
            }
        } else {
            SwingUtilities.windowForComponent(dockable.container).toFront()
            val button = dockingButtons[dockable]!!
            button.setSelected(false)
        }
    }

    override fun setFloating(dockable: Dockable<JComponent>, floating: Boolean) {
        val state: DockingState = dockingStates[dockable]!!
        if (floating) {
            if (!state.isFloating) {
                val component: JComponent = dockable.component!!
                val p: Point?
                p = if (component.isShowing) {
                    component.locationOnScreen
                } else {
                    null
                }
                val size = component.size
                setActive(dockable, false)
                val frame = JFrame()
                frame.setAlwaysOnTop(true)
                if (p != null) {
                    frame.setLocation(p)
                } else {
                    frame.setLocationRelativeTo(this)
                }
                frame.setSize(size)
                val title: String? = dockable.title
                //                final Icon icon = dockable.getIcon();
                frame.setTitle(title)
                //                if (icon != null) {
                // ToDo: reimplement
                //frame.setIconImage(icon.getImage());
//                }
                val container: JComponent = dockable.detachContainer()!!
                frame.getContentPane().add(container)
                frame.validate()
                frame.setVisible(true)
                frame.addWindowListener(object : WindowAdapter() {
                    override fun windowClosing(e: WindowEvent) {
                        setFloating(dockable, false)
                    }
                })
                state.setType(DockingType.FLOATING)
            }
        } else {
            if (state.isFloating) {
                val component: JComponent = dockable.container!!
                component.parent.remove(component)
                dockable.attachContainer()
                state.setType(DockingType.DOCKED)
                setActive(dockable, true)
            }
        }
    }

    override fun setSliding(dockable: Dockable<JComponent>, sliding: Boolean) {
        val state: DockingState = dockingStates[dockable]!!
        if (sliding) {
            if (!state.isSliding) {
                val component: JComponent = dockable.component!!
                //                Point p = component.getLocationOnScreen();
//                Dimension size = component.getSize();
//                setActive(dockable, false);
                val frame = JFrame()
                //                frame.setLocation(p);
                frame.setSize(50, 50)
                frame.setUndecorated(true)
                val title: String? = dockable.title
                //                final Icon icon = dockable.getIcon();
                frame.setTitle(title)
                //                if (icon != null) {
                // ToDo: reimplement
//                    frame.setIconImage(icon.getImage());
//                }
                val container: JComponent = dockable.detachContainer()!!
                frame.getContentPane().add(container)
                frame.validate()
                frame.setVisible(true)
                state.setType(DockingType.SLIDING)
            }
        } else {
            if (state.isSliding) {
                val component: JComponent = dockable.container!!
                component.getParent().remove(component)
                dockable.attachContainer()
                state.setType(DockingType.DOCKED)
                //                setActive(dockable, true);
            }
        }
    }

    private fun update() {
        super.removeAll()
        //        setLayout(new GridBagLayout());
        if (orientation == DockingAnchor.TOP || orientation == DockingAnchor.BOTTOM) {
//            int gap = 22;
//            add(Box.createHorizontalStrut(gap));
            var i = 1
            for (dockable in dockables) {
                val dockingButton = dockingButtons[dockable]
                add(dockingButton, GridBagConstraints(i, 0, 1, 1, 0.0, 1.0, 17, 3, Insets(0, 0, 0, 0), 0, 0))
                i++
            }
            add(Box.createGlue(), GridBagConstraints(i, 0, 1, 1, 1.0, 1.0, 17, 0, Insets(0, 0, 0, 0), 0, 0))
        } else if (orientation == DockingAnchor.LEFT || orientation == DockingAnchor.RIGHT) {
            var i = 0
            for (dockable in dockables) {
                val dockingButton = dockingButtons[dockable]
                add(dockingButton, GridBagConstraints(0, i, 1, 1, 1.0, 0.0, 10, 2, Insets(0, 0, 0, 0), 0, 0))
                i++
            }
            val gridbagconstraints = GridBagConstraints()
            gridbagconstraints.gridy = dockables.size
            gridbagconstraints.weighty = 1.0
            gridbagconstraints.anchor = 11
            add(Box.createGlue(), gridbagconstraints)
        } else {
            throw UnknownError()
        }
        //        if(getComponentCount() > 2) {
//            setVisible(true);
//        } else {
//            setVisible(false);
//        }
    }

    override fun dispose() {
        if (enterTimer != null) {
            enterTimer!!.stop()
            enterTimer = null
        }
        insideTimerAction = null
        dockables.clear()
        for (dockingButton in dockingButtons.values) {
//            ComponentCleaner.cleanComponent(dockingButton)
        }
        dockingButtons.clear()
        dockingStates.clear()
//        ComponentCleaner.cleanComponent(container)
    }

    override val nativeComponent: JComponent
        get() = this

    private class InsideTimerAction  //        private Dockable dockable;
        : ActionListener {
        fun setDockable(dockable: Dockable<*>?) {
//            this.dockable = dockable;
        }

        override fun actionPerformed(e: ActionEvent) {
//            setSliding(dockable, true);
        }
    }

    init {
        this.orientation = orientation
        dockables = ArrayList<Dockable<JComponent>>()
        dockingButtons = HashMap<Dockable<*>, DockingButton>()
        dockingStates = HashMap<Dockable<*>, DockingState>()
        insideTimerAction = InsideTimerAction()
        enterTimer = Timer(750, insideTimerAction)
        enterTimer!!.isRepeats = false
        setLayout(GridBagLayout())
        setOpaque(false)
    }
}