/*
 * Copyright (c) 2016 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.docking.swing.splitter

import java.awt.*
import java.awt.event.*
import javax.swing.*

/**
 * User: luc
 * Date: Jun 1, 2006
 * Time: 9:43:24 AM
 */
class TwoComponentsSplitter @JvmOverloads constructor(private var myVerticalSplit: Boolean = false) : JPanel() {
    protected inner class Divider(isFirst: Boolean) : JPanel(GridBagLayout()) {
        fun setOrientation(isVerticalSplit: Boolean) {
            removeAll()
            if (!myShowDividerControls) {
                return
            } else {
                val xMask = if (isVerticalSplit) 1 else 0
                val yMask = if (isVerticalSplit) 0 else 1
                val glueIcon: Icon =
                    ImageIcon(javaClass.getResource(if (isVerticalSplit) "splitGlueV.png" else "splitGlueH.png"))
                val glueFill = if (isVerticalSplit) 3 else 2
                add(
                    JLabel(glueIcon),
                    GridBagConstraints(
                        0,
                        0,
                        1,
                        1,
                        0.0,
                        0.0,
                        if (isVerticalSplit) 13 else 11,
                        glueFill,
                        Insets(0, 0, 0, 0),
                        0,
                        0
                    )
                )
                val splitDownlabel =
                    JLabel(ImageIcon(javaClass.getResource(if (isVerticalSplit) "splitDown.png" else "splitRight.png")))
                splitDownlabel.cursor = Cursor.getPredefinedCursor(12)
                //                splitDownlabel.setToolTipText(isVerticalSplit ? UIBundle.message("splitter.down.tooltip.text", new Object[0]) : UIBundle.message("splitter.right.tooltip.text", new Object[0]));
                splitDownlabel.addMouseListener(object : MouseAdapter() {
                    override fun mouseClicked(e: MouseEvent) {
                        val income = if (myVerticalSplit) lastComponent!!.height else lastComponent!!.width
                        if (myIsFirst) firstSize = myFirstSize + income
                    }
                })
                add(
                    splitDownlabel,
                    GridBagConstraints(
                        if (isVerticalSplit) 1 else 0,
                        if (isVerticalSplit) 0 else 5,
                        1,
                        1,
                        0.0,
                        0.0,
                        10,
                        0,
                        Insets(0, 0, 0, 0),
                        0,
                        0
                    )
                )
                add(
                    JLabel(glueIcon),
                    GridBagConstraints(2 * xMask, 2 * yMask, 1, 1, 0.0, 0.0, 10, glueFill, Insets(0, 0, 0, 0), 0, 0)
                )
                val splitCenterlabel =
                    JLabel(ImageIcon(javaClass.getResource(if (isVerticalSplit) "splitCenterV.png" else "splitCenterH.png")))
                splitCenterlabel.cursor = Cursor.getPredefinedCursor(12)
                //                splitCenterlabel.setToolTipText(UIBundle.message("splitter.center.tooltip.text", new Object[0]));
                splitCenterlabel.addMouseListener(object : MouseAdapter() {
                    override fun mouseClicked(e: MouseEvent) {
                        center()
                    }
                })
                add(
                    splitCenterlabel,
                    GridBagConstraints(3 * xMask, 3 * yMask, 1, 1, 0.0, 0.0, 10, 0, Insets(0, 0, 0, 0), 0, 0)
                )
                add(
                    JLabel(glueIcon),
                    GridBagConstraints(4 * xMask, 4 * yMask, 1, 1, 0.0, 0.0, 10, glueFill, Insets(0, 0, 0, 0), 0, 0)
                )
                val splitUpLabel =
                    JLabel(ImageIcon(javaClass.getResource(if (isVerticalSplit) "splitUp.png" else "splitLeft.png")))
                splitUpLabel.cursor = Cursor.getPredefinedCursor(12)
                //                splitUpLabel.setToolTipText(isVerticalSplit ? UIBundle.message("splitter.up.tooltip.text", new Object[0]) : UIBundle.message("splitter.left.tooltip.text", new Object[0]));
                splitUpLabel.addMouseListener(object : MouseAdapter() {
                    override fun mouseClicked(e: MouseEvent) {
//                        if(myInnerComponent != null)
                        if (myIsFirst) firstSize = getMinSize(firstComponent)
                    }
                })
                add(
                    splitUpLabel,
                    GridBagConstraints(
                        if (isVerticalSplit) 5 else 0,
                        if (isVerticalSplit) 0 else 1,
                        1,
                        1,
                        0.0,
                        0.0,
                        10,
                        0,
                        Insets(0, 0, 0, 0),
                        0,
                        0
                    )
                )
                add(
                    JLabel(glueIcon),
                    GridBagConstraints(
                        6 * xMask,
                        6 * yMask,
                        1,
                        1,
                        0.0,
                        0.0,
                        if (isVerticalSplit) 17 else 15,
                        glueFill,
                        Insets(0, 0, 0, 0),
                        0,
                        0
                    )
                )
                return
            }
        }

        private fun center() {
            val total = myFirstSize + if (myVerticalSplit) lastComponent!!.height else lastComponent!!.width
            if (myIsFirst) firstSize = total / 2
        }

        override fun processMouseMotionEvent(e: MouseEvent) {
            super.processMouseMotionEvent(e)
            if (506 == e.id) {
                myDragging = true
                cursor = if (orientation) Cursor.getPredefinedCursor(9) else Cursor.getPredefinedCursor(11)
                myPoint = SwingUtilities.convertPoint(this, e.point, this@TwoComponentsSplitter)
                if (orientation) {
                    if (height > 0) if (myIsFirst) firstSize = Math.max(getMinSize(firstComponent), myPoint!!.y)
                } else if (width > 0) if (myIsFirst) firstSize = Math.max(getMinSize(firstComponent), myPoint!!.x)
                doLayout()
            }
        }

        private fun getMinSize(component: JComponent?): Int {
            return if (isHonorMinimumSize && component != null && firstComponent != null && firstComponent!!.isVisible && lastComponent != null && lastComponent!!.isVisible) {
                if (orientation) component.minimumSize.height else component.minimumSize.width
            } else {
                0
            }
        }

        override fun processMouseEvent(e: MouseEvent) {
            super.processMouseEvent(e)
            when (e.id) {
                503 -> {
                }
                504 -> cursor = if (orientation) Cursor.getPredefinedCursor(9) else Cursor.getPredefinedCursor(11)
                505 -> if (!myDragging) cursor = Cursor.getPredefinedCursor(0)
                501 -> cursor = if (orientation) Cursor.getPredefinedCursor(9) else Cursor.getPredefinedCursor(11)
                502 -> {
                    myDragging = false
                    myPoint = null
                }
                500 -> if (e.clickCount == 2) center()
                else -> {
                }
            }
        }

        protected var myDragging = false
        protected var myPoint: Point? = null
        private val myIsFirst: Boolean

        init {
            isFocusable = false
            enableEvents(48L)
            myIsFirst = isFirst
            setOrientation(myVerticalSplit)
            isOpaque = false
        }
    }

    fun setShowDividerControls(showDividerControls: Boolean) {
        myShowDividerControls = showDividerControls
        orientation = myVerticalSplit
    }

    fun setHonorComponentsMinimumSize(honorMinimumSize: Boolean) {
        isHonorMinimumSize = honorMinimumSize
    }

    override fun isVisible(): Boolean {
        return super.isVisible() && (firstVisible() || lastVisible())
    }

    private fun lastVisible(): Boolean {
        return lastComponent != null && lastComponent!!.isVisible
    }

    private fun firstVisible(): Boolean {
        return firstComponent != null && firstComponent!!.isVisible
    }

    private fun visibleDividersCount(): Int {
        var count = 0
        if (firstDividerVisible()) count++
        return count
    }

    private fun firstDividerVisible(): Boolean {
        return firstVisible() && lastVisible()
    }

    override fun getMinimumSize(): Dimension {
        return if (isHonorMinimumSize) {
            val dividerWidth = dividerWidth
            val firstSize = if (firstComponent == null) Dimension(0, 0) else firstComponent!!.minimumSize
            val lastSize = if (lastComponent == null) Dimension(0, 0) else lastComponent!!.minimumSize
            if (orientation) {
                val width = Math.max(firstSize.width, lastSize.width)
                var height = visibleDividersCount() * dividerWidth
                height += firstSize.height
                height += lastSize.height
                Dimension(width, height)
            } else {
                val heigth = Math.max(firstSize.height, lastSize.height)
                var width = visibleDividersCount() * dividerWidth
                width += firstSize.width
                width += lastSize.width
                Dimension(width, heigth)
            }
        } else {
            super.getMinimumSize()
        }
    }

    override fun doLayout() {
        val width = width
        val height = height
        val firstRect = Rectangle()
        val firstDividerRect = Rectangle()
        val lastRect = Rectangle()
        val componentSize = if (orientation) height else width
        var dividerWidth = dividerWidth
        val dividersCount = visibleDividersCount()
        var firstCompontSize: Int
        val lastComponentSize: Int
        if (componentSize <= dividersCount * dividerWidth) {
            firstCompontSize = 0
            lastComponentSize = 0
            dividerWidth = componentSize
        } else {
            if (firstDividerVisible()) {
                firstCompontSize = firstSize
                lastComponentSize = componentSize - dividersCount * dividerWidth - firstCompontSize
            } else {
                if (firstComponent!!.isVisible) {
                    firstCompontSize = componentSize
                    lastComponentSize = 0
                } else {
                    firstCompontSize = 0
                    lastComponentSize = componentSize
                }
            }
            val sizeLack = firstCompontSize - (componentSize - dividersCount * dividerWidth)
            if (sizeLack > 0) {
                firstCompontSize -= sizeLack
            }
        }
        if (orientation) {
            var space = firstCompontSize
            firstRect.setBounds(0, 0, width, firstCompontSize)
            if (firstDividerVisible()) {
                firstDividerRect.setBounds(0, space, width, dividerWidth)
                space += dividerWidth
            }
            lastRect.setBounds(0, space, width, lastComponentSize)
        } else {
            var space = firstCompontSize
            firstRect.setBounds(0, 0, firstCompontSize, height)
            if (firstDividerVisible()) {
                firstDividerRect.setBounds(space, 0, dividerWidth, height)
                space += dividerWidth
            }
            lastRect.setBounds(space, 0, lastComponentSize, height)
        }
        myDivider.isVisible = firstDividerVisible()
        myDivider.bounds = firstDividerRect
        myDivider.doLayout()
        if (firstComponent != null) {
            firstComponent!!.bounds = firstRect
            firstComponent!!.revalidate()
        }
        if (lastComponent != null) {
            lastComponent!!.bounds = lastRect
            lastComponent!!.revalidate()
        }
    }

    var dividerWidth: Int
        get() = myDividerWidth
        set(width) {
            require(width > 0) { "Wrong divider width: $width" }
            if (myDividerWidth != width) {
                myDividerWidth = width
                doLayout()
                repaint()
            }
        }
    var orientation: Boolean
        get() = myVerticalSplit
        set(verticalSplit) {
            myVerticalSplit = verticalSplit
            myDivider.setOrientation(verticalSplit)
            doLayout()
            repaint()
        }

    fun setFirstComponent(component: JComponent) {
        if (firstComponent !== component) {
            if (firstComponent != null) remove(firstComponent)
            firstComponent = component
            if (firstComponent != null) {
                super.add(firstComponent)
                firstComponent!!.invalidate()
            }
        }
    }

    fun setLastComponent(component: JComponent) {
        if (lastComponent !== component) {
            if (lastComponent != null) remove(lastComponent)
            lastComponent = component
            if (lastComponent != null) {
                super.add(lastComponent)
                lastComponent!!.invalidate()
            }
        }
    }

    var firstSize: Int
        get() = if (firstVisible()) myFirstSize else 0
        set(size) {
            if (myFirstSize != size) {
                myFirstSize = size
                doLayout()
                repaint()
            }
        }

    override fun setBounds(x: Int, y: Int, width: Int, height: Int) {
        var firstSize = myFirstSize
        if (getWidth() > 0 && getHeight() > 0) {
            firstSize = if (myVerticalSplit) {
                firstSize * height / getHeight()
            } else {
                firstSize * width / getWidth()
            }
        }
        super.setBounds(x, y, width, height)
        firstSize = firstSize
    }

    private var myDividerWidth: Int
    var isHonorMinimumSize = false
        private set
    private val myDivider: Divider
    var firstComponent: JComponent? = null
        private set
    var lastComponent: JComponent? = null
        private set
    private var myFirstSize = 10
    private var myShowDividerControls = false

    init {
        myDivider = Divider(true)
        myDividerWidth = 7
        isOpaque = false
        super.add(myDivider)
    }
}