/*
 * Copyright (c) 2012 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.slider

import com.macrofocus.common.interval.*
import com.macrofocus.common.math.Histogram
import kotlin.math.abs
import kotlin.math.min

class BoundedIntervalRangeSliderAdapter(boundedRangeModel: MutableBoundedInterval, minExtentFactor: Double, maxExtentFactor: Double) :
    com.macrofocus.slider.AbstractRangeSliderModel() {
    private var boundedRangeModel: MutableBoundedInterval
    private val minExtentFactor: Double
    private val maxExtentFactor: Double
    override var lowerMinimumValue = 0.0
        get() = if (lowerMinimumEnabled) {
            field
        } else {
            minimum
        }
        set(lowerMinimumValue) {
            field = lowerMinimumValue
            lowerMinimumEnabled = true
        }
    override var lowerMaximumValue = 0.0
        get() = if (lowerMaximumEnabled) {
            field
        } else {
            maximum
        }
        set(lowerMaximumValue) {
            field = lowerMaximumValue
            lowerMaximumEnabled = true
        }
    override var upperMinimumValue = 0.0
        get() = if (upperMinimumEnabled) {
            field
        } else {
            minimum
        }
        set(upperMinimumValue) {
            field = upperMinimumValue
            upperMinimumEnabled = true
        }
    override var upperMaximumValue = 0.0
        get() = if (upperMaximumEnabled) {
            field
        } else {
            maximum
        }
        set(upperMaximumValue) {
            field = upperMaximumValue
            upperMaximumEnabled = true
        }
    private var lowerMinimumEnabled = false
    private var lowerMaximumEnabled = false
    private var upperMinimumEnabled = false
    private var upperMaximumEnabled = false
    private val sliderListeners: ArrayList<RangeSliderListener>
    override var histogram: Histogram? = null
    private val listener: IntervalListener = object : IntervalListener {
        override fun intervalChanged(event: IntervalEvent) {
            notifySliderChanged(
                com.macrofocus.slider.RangeSliderEvent(
                    self,
                    this,
                    event.newStart,
                    event.newEnd,
                    event.oldStart,
                    event.oldEnd
                )
            )
        }
    }
    private val boundedListener: BoundedIntervalListener = object : BoundedIntervalListener {
        override fun boundedIntervalChanged(event: BoundedIntervalEvent) {
            val eventRange: com.macrofocus.slider.RangeSliderEvent = com.macrofocus.slider.RangeSliderEvent(
                self, this,
                boundedRangeModel.minimum, boundedRangeModel.maximum,
                boundedRangeModel.minimum, boundedRangeModel.maximum
            )
            notifySliderScaleChanged(eventRange)
        }
    }
    private var resetStrategy: ResetStrategy = object : ResetStrategy {
        override fun reset() {
            setValues(
                this,
                boundedRangeModel.minimum,
                min(boundedRangeModel.minimum + boundedRangeModel.maximumExtent, boundedRangeModel.maximum)
            )
        }
    }

    constructor(boundedRangeModel: MutableBoundedInterval) : this(boundedRangeModel, 0.0, 1.0) {}

    private val self: com.macrofocus.slider.RangeSliderModel
        private get() = this

    override fun setValues(source: Any, rangeMin: Double, rangeMax: Double) {
        boundedRangeModel.setValue(rangeMin, rangeMax - rangeMin)
    }

    override val currentExtent: Double
        get() = boundedRangeModel.extent
    override val currentMinimum: Double
        get() = boundedRangeModel.start
    override val currentMaximum: Double
        get() = boundedRangeModel.end
    override val minimum: Double
        get() = boundedRangeModel.minimum
    override val maximum: Double
        get() = boundedRangeModel.maximum

    override fun setMinMax(source: Any, min: Double, max: Double) {
        boundedRangeModel.setMinMax(min, max)
    }

//    fun setMinimum(source: Any?, min: Double) {
//        boundedRangeModel.setMinimum(min)
//    }

//    fun setMaximum(source: Any?, max: Double) {
//        boundedRangeModel.setMaximum(max)
//    }

    override val extent: Double
        get() = abs(maximum - minimum)

    fun setLowerMinimumEnabled(lowerMinimumEnabled: Boolean) {
        this.lowerMinimumEnabled = lowerMinimumEnabled
    }

    fun setLowerMaximumEnabled(lowerMaximumEnabled: Boolean) {
        this.lowerMaximumEnabled = lowerMaximumEnabled
    }

    fun setUpperMinimumEnabled(upperMinimumEnabled: Boolean) {
        this.upperMinimumEnabled = upperMinimumEnabled
    }

    fun setUpperMaximumEnabled(upperMaximumEnabled: Boolean) {
        this.upperMaximumEnabled = upperMaximumEnabled
    }

    override fun reset(source: Any) {
        resetStrategy.reset()
    }

    override val maximumExtent: Double
        get() = if (maxExtentFactor == 0.0) {
            if (boundedRangeModel.maximumExtent == 0.0) {
                Double.MAX_VALUE
            } else {
                abs(boundedRangeModel.maximumExtent)
            }
        } else {
            abs(boundedRangeModel.maximum - boundedRangeModel.minimum) / maxExtentFactor
        }
    override val minimumExtent: Double
        get() = if (minExtentFactor == 0.0) {
            if (boundedRangeModel.minimumExtent == 0.0) {
                0.0
            } else {
                boundedRangeModel.minimumExtent
            }
        } else {
            abs(boundedRangeModel.maximum - boundedRangeModel.minimum) / minExtentFactor
        }

    fun setBoundedRangeModel(boundedRangeModel: MutableBoundedInterval) {
        if (this.boundedRangeModel != null) {
            this.boundedRangeModel.removeIntervalListener(listener)
        }
        this.boundedRangeModel = boundedRangeModel
        if (this.boundedRangeModel != null) {
            val eventRange: com.macrofocus.slider.RangeSliderEvent = com.macrofocus.slider.RangeSliderEvent(
                this, this,
                boundedRangeModel.minimum, boundedRangeModel.maximum,
                this.boundedRangeModel.minimum, this.boundedRangeModel.maximum
            )
            this.boundedRangeModel.addIntervalListener(listener)
            notifySliderScaleChanged(eventRange)
        } else {
            throw RuntimeException("Debug: Null range model")
        }
    }

    override fun addSliderListener(rangeSliderListener: com.macrofocus.slider.RangeSliderListener) {
        sliderListeners.add(rangeSliderListener)
    }

    override fun removeSliderListener(rangeSliderListener: com.macrofocus.slider.RangeSliderListener) {
        sliderListeners.remove(rangeSliderListener)
    }

    private fun notifySliderChanged(eventRange: com.macrofocus.slider.RangeSliderEvent) {
        for (sliderListener in sliderListeners) {
            sliderListener.sliderChanged(eventRange)
        }
    }

    private fun notifySliderScaleChanged(eventRange: com.macrofocus.slider.RangeSliderEvent) {
        for (sliderListener in sliderListeners) {
            sliderListener.sliderScaleChanged(eventRange)
        }
    }

    override fun toString(): String {
        return boundedRangeModel.toString()
    }

    override val isInvertedScale: Boolean
        get() = boundedRangeModel.minimum > boundedRangeModel.maximum

    fun setResetStrategy(resetStrategy: ResetStrategy) {
        this.resetStrategy = resetStrategy
    }

    interface ResetStrategy {
        fun reset()
    }

    init {
        this.boundedRangeModel = boundedRangeModel
        this.minExtentFactor = minExtentFactor
        this.maxExtentFactor = maxExtentFactor
        boundedRangeModel.addIntervalListener(listener)
        boundedRangeModel.addBoundedIntervalListener(boundedListener)
        sliderListeners = ArrayList()
    }
}