package com.macrofocus.plot.guide

import com.macrofocus.common.format.FormatFactory
import kotlin.math.min

/**
 * A collection of tick units, used by the [DateAxis] and
 * [NumberAxis] classes.
 */
internal class TickUnits(formatFactory: FormatFactory) : TickUnitSource {
    private val formatFactory: FormatFactory

    /** Storage for the tick units.  */
    private val tickUnits: MutableList<TickUnit>

    /**
     * Adds a tick unit to the collection.  The tick units are maintained in
     * ascending order.
     *
     * @param unit the tick unit to add (`null` not permitted).
     */
    fun add(unit: TickUnit) {
        tickUnits.add(unit)
        tickUnits.sort()
    }

    /**
     * Returns a tick unit that is larger than the supplied unit.
     *
     * @param unit the unit.
     *
     * @return A tick unit that is larger than the supplied unit.
     */
    override fun getLargerTickUnit(unit: TickUnit): TickUnit {
        var index: Int = tickUnits.binarySearch(unit)
        index += if (index >= 0) 1 else -index
        return tickUnits.get(
            min(
                index,
                tickUnits.size - 1
            )
        )
    }

    /**
     * Returns the tick unit in the collection that is greater than or equal
     * to (in size) the specified unit.
     *
     * @param unit the unit.
     *
     * @return A unit from the collection.
     */
    override fun getCeilingTickUnit(unit: TickUnit): TickUnit {
        var index: Int = tickUnits.binarySearch(unit)
        return if (index >= 0) {
            tickUnits.get(index)
        } else {
            index = -(index + 1)
            tickUnits.get(
                min(
                    index,
                    tickUnits.size - 1
                )
            )
        }
    }

    /**
     * Returns the tick unit in the collection that is greater than or equal
     * to the specified size.
     *
     * @param size the size.
     *
     * @return A unit from the collection.
     */
    override fun getCeilingTickUnit(size: Double): TickUnit {
        return getCeilingTickUnit(
            NumberTickUnit(
                size,
                formatFactory.createDecimalFormat()
            )
        )
    }

    /** Constructs a new collection of tick units.  */
    init {
        this.formatFactory = formatFactory
        tickUnits = ArrayList()
    }
}