package com.macrofocus.plot.guide

import com.macrofocus.common.format.CPFormat
import com.macrofocus.common.format.FormatFactory
import com.macrofocus.common.geom.Dimension
import com.macrofocus.common.transform.OneDScreenTransform
import com.macrofocus.common.transform.ScreenTransformEvent
import com.macrofocus.common.transform.ScreenTransformListener
import org.mkui.canvas.CPCanvas
import org.mkui.color.MkColor
import org.mkui.color.MkColorFactory
import org.mkui.color.colorOf
import org.mkui.component.CPComponent
import org.mkui.font.CPFontFactory
import org.mkui.font.CPFontMetrics
import org.mkui.font.Weight
import org.mkui.geom.Point2D
import org.mkui.geom.Rectangle
import org.mkui.geom.Rectangle2D
import org.mkui.graphics.AbstractIDrawing
import org.mkui.graphics.CPHeadless
import org.mkui.graphics.IGraphics
import kotlin.math.max
import kotlin.math.min

abstract class AbstractAxisGuide(private var type: Guide.Type) : AbstractGuide(),AxisGuide  {
    private var canvas = CPCanvas()
    init {
        canvas.addLayer(object : AbstractIDrawing() {
            override fun draw(g: IGraphics, point: Point2D?, width: Double, height: Double, clipBounds: Rectangle) {
                paintComponent(g)
            }
        })
    }

    override val component: CPComponent
        get() = canvas

    private var labelColor: MkColor = MkColorFactory.instance.black
    private var gridLinesColor: MkColor = colorOf(199, 199, 196)
    private var paintTicks = true
    private val coordinateAxisListener: ScreenTransformListener = object : ScreenTransformListener {
        override fun transformChanged(event: ScreenTransformEvent) {
            refresh()
        }
    }

    private fun refresh() {
//        repaint();
//            invalidate();
//        revalidate();
//        if (panel != null) {
//            panel.valideAxis();
//        }
        revalidate()
        repaint()
    }

    private var coordinateAxis: OneDScreenTransform? = null
    private var axis: ValueAxis? = null
    private var format: CPFormat<Any?>? = null

    constructor(type: Guide.Type, coordinateAxis: OneDScreenTransform?) : this(type) {
        setCoordinateAxis(coordinateAxis)
    }

    constructor(coordinateAxis: OneDScreenTransform?) : this(Guide.Type.Bottom) {
        setCoordinateAxis(coordinateAxis)
    }

//    fun AbstractGWTAxisGuide(type: Guide.Type?) {
//        this.type = type
//        addComponentListener(new ComponentListener() {
//            @Override
//            public void componentResized(ComponentEvent e) {
//                // ToDo: Figure out why this is needed! Otherwise, the space used by for the vertical axis isn't computed propertly.
//                refresh();
//            }
//
//            @Override
//            public void componentMoved(ComponentEvent e) {
//            }
//
//            @Override
//            public void componentShown(ComponentEvent e) {
//                refresh();
//            }
//
//            @Override
//            public void componentHidden(ComponentEvent e) {
//            }
//        });
//    }

    constructor() : this(Guide.Type.Bottom)

    override fun setCoordinateAxis(coordinateAxis: OneDScreenTransform?) {
        if (this.coordinateAxis != null) {
            this.coordinateAxis!!.removeScreenTransformListener(coordinateAxisListener)
        }
        this.coordinateAxis = coordinateAxis
        if (this.coordinateAxis != null) {
            this.coordinateAxis!!.addScreenTransformListener(coordinateAxisListener)
        }
        refresh()
    }

    override fun computeSpace(g2: IGraphics, dataArea: Rectangle): Double {
        val space = AxisSpace()
        when (type) {
            Guide.Type.Top -> {
                getAxis().reserveSpace(g2, dataArea, RectangleEdge.Top, space)
                return space.bottom
            }

            Guide.Type.Bottom -> {
                getAxis().reserveSpace(g2, dataArea, RectangleEdge.Bottom, space)
                return space.bottom + 4
            }

            Guide.Type.Left -> {
                getAxis().reserveSpace(g2, dataArea, RectangleEdge.Left, space)
                return space.left
            }

            Guide.Type.Right -> {
                getAxis().reserveSpace(g2, dataArea, RectangleEdge.Right, space)
                return space.left
            }

            else -> throw IllegalArgumentException()
        }
    }

    private fun getAxis(): ValueAxis {
        if (axis == null) {
            axis = NumberAxis(CPFontFactory.instance, CPHeadless, FormatFactory.instance)
            axis!!.tickLabelFont = CPFontFactory.instance.createFont("Roboto, Helvetica, Arial, sans-serif", Weight.NORMAL, 18)
            if (format != null) {
                axis!!.setFormatOverride(format)
            }
        }
        return axis!!
    }

    override fun drawGrid(g2: IGraphics, d: Dimension) {
        gridRenderer.drawGrid(g2, type!!, coordinateAxis!!, getAxis(), gridLineRenderer, d)
    }

    override fun paintComponent(g: IGraphics) {
        val d: Dimension? = size

        if (coordinateAxis != null) {
            val min: Double = min(coordinateAxis!!.worldMin, coordinateAxis!!.worldMax)
            val max: Double = max(coordinateAxis!!.worldMin, coordinateAxis!!.worldMax)
            if (min <= max) {
                val dataArea: Rectangle2D =
                    Rectangle2D.Double(0.0, 0.0, d!!.width.toDouble(), d.height.toDouble())
                getAxis().setRange(min, max)
                val ticks: List<ValueTick?> = when (type) {
                    Guide.Type.Top -> getAxis().refreshTicks(g, dataArea, RectangleEdge.Top)
                    Guide.Type.Bottom -> getAxis().refreshTicks(g, dataArea, RectangleEdge.Bottom)
                    Guide.Type.Left -> getAxis().refreshTicks(g, dataArea, RectangleEdge.Left)
                    Guide.Type.Right -> getAxis().refreshTicks(g, dataArea, RectangleEdge.Right)
                    else -> throw IllegalArgumentException()
                }

                // Draw vertical group starting lines
                for (tick in ticks) {
                    val position: Int = coordinateAxis!!.worldToScreen(tick!!.value)

                    if (paintTicks) {
                        g.setColor(gridLinesColor!!)
                        when (type) {
                            Guide.Type.Top -> g.drawLine(position.toDouble(), d.height - 5.0, position.toDouble(), d.height)
                            Guide.Type.Bottom -> g.drawLine(position, 0, position, 5)
                            Guide.Type.Left -> g.drawLine(d.width - 5.0, position.toDouble(), d.width, position.toDouble())
                            Guide.Type.Right -> g.drawLine(0, position, 5, position)
                        }
                    }

                    g.setColor(labelColor)

                    val text = tick!!.text
                    val fontMetrics: CPFontMetrics = g.getFontMetrics()!!
                    val width = g.getStringWidth(text).toInt()
                    when (type) {
                        Guide.Type.Top -> g.drawString(text, (position - (width shr 1)).toFloat(), fontMetrics.getHeight())
                        Guide.Type.Bottom -> g.drawString(text, (position - (width shr 1)).toFloat(), (d.height - 2).toFloat())
                        Guide.Type.Left -> g.drawString(
                            text,
                            (d.width - width - 7).toFloat(),
                            (position + (fontMetrics.getAscent() shr 1)).toFloat(),
                        )

                        Guide.Type.Right -> g.drawString(text, 7f, (position + (fontMetrics.getAscent() shr 1)).toFloat())
                    }
                }
            }
        }
    }


    //    @Override
    //    public void setFont(Font font) {
    //        super.setFont(font);
    //
    //        getAxis().setTickLabelFont(new SwingFont(font));
    //    }
    override fun setFormat(format: CPFormat<Any?>?) {
        this.format = format
        if (axis != null) {
            axis!!.setFormatOverride(this.format)
        }
        refresh()
    }

    override fun setAxis(axis: ValueAxis) {
        this.axis = axis
        refresh()
    }

    fun setGridLinesColor(gridLinesColor: MkColor) {
        this.gridLinesColor = gridLinesColor
        repaint()
    }

    fun setLabelColor(labelColor: MkColor) {
        this.labelColor = labelColor
        repaint()
    }

    fun setPaintTicks(paintTicks: Boolean) {
        this.paintTicks = paintTicks
        repaint()
    }

    override fun toString(): String {
        return "AbstractAxisGuide{" +
                "coordinateAxis=" + coordinateAxis +
                ", axis=" + axis +
                '}'
    }

    override fun invalidate() {
    }

    override fun revalidate() {
    }

    override fun repaint() {
        canvas.redraw()
    }

    override val size: Dimension?
        get() = Dimension(canvas.getWidth().toInt(), canvas.getHeight().toInt())
    override val width: Int
        get() = canvas.getWidth().toInt()
    override val height: Int
        get() = canvas.getHeight().toInt()

    override fun setSize(width: Int, height: Int) {
    }

    override val graphics: IGraphics?
        get() = canvas.igraphics
    override val preferredSize: Dimension?
        get() = TODO("Not yet implemented")
}