package com.macrofocus.plot.guide

import com.macrofocus.common.date.CPTimeZone
import com.macrofocus.common.format.CPFormat
import com.macrofocus.common.format.FormatFactory
import com.macrofocus.common.locale.CPLocale
import com.macrofocus.plot.datetime.CPCalendar
import kotlinx.datetime.Instant
import kotlinx.datetime.LocalDateTime
import kotlinx.datetime.format.MonthNames
import kotlinx.datetime.format.Padding
import kotlinx.datetime.format.char
import org.mkui.font.CPFontFactory
import org.mkui.geom.Rectangle2D
import org.mkui.graphics.CPHeadless
import org.mkui.graphics.IGraphics
import org.mkui.graphics.IHeadless
import kotlin.math.PI
import kotlin.math.abs
import kotlin.math.max

/**
 * The base class for axes that display dates.  You will find it easier to
 * understand how this axis works if you bear in mind that it really
 * displays/measures integer (or long) data, where the integers are
 * milliseconds since midnight, 1-Jan-1970.  When displaying tick labels, the
 * millisecond values are converted back to dates using a
 * `DateFormat` instance.
 */
class DateAxis(
    fontFactory: CPFontFactory = CPFontFactory.instance,
    headless: IHeadless = CPHeadless,
    label: String? = null,
    zone: CPTimeZone? = null,
    locale: CPLocale? = null,
    formatFactory: FormatFactory = FormatFactory.instance
) : ValueAxis(fontFactory, headless, label, createStandardDateTickUnits(zone, locale, formatFactory)) {
    /**
     * Tick marks can be displayed at the start or the middle of the time
     * period.
     */
    private val tickMarkPosition = DateTickMarkPosition.Start

    /** The time zone for the axis.  */
    private val timeZone: CPTimeZone?

    /** The locale for the axis (`null` is not permitted).  */
    private val locale: CPLocale?

    /** Our underlying timeline.  */
    private val timeline: Timeline

    /** The current tick unit.  */
    private var tickUnit: DateTickUnit?

    /** The override date format.  */
    private var dateFormatOverride: CPFormat<Any?>?

    /**
     * Creates a date axis. A timeline is specified for the axis. This allows
     * special transformations to occur between a domain of values and the
     * values included in the axis.
     */
    constructor(
        fontFactory: CPFontFactory,
        headless: IHeadless,
        formatFactory: FormatFactory,
    ) : this(fontFactory, headless, null, null, null, formatFactory) {
    }

    /**
     * Sets the tick unit attribute.
     *
     * @param unit the new tick unit.
     */
    fun setTickUnit(unit: DateTickUnit?) {
        tickUnit = unit
    }

    /**
     * Returns the date format override.  If this is non-null, then it will be
     * used to format the dates on the axis.
     *
     * @return The formatter (possibly `null`).
     */
    fun getDateFormatOverride(): CPFormat<Any?>? {
        return dateFormatOverride
    }

    /**
     * Sets the date format override.  If this is non-null, then it will be
     * used to format the dates on the axis.
     *
     * @param formatter the date formatter (`null` permitted).
     */
    override fun setFormatOverride(formatter: CPFormat<Any?>?) {
        dateFormatOverride = formatter
    }

    /**
     * Sets the upper and lower bounds for the axis and sends an
     * to all registered listeners.  As a side-effect,
     * the auto-range flag is set to false.
     *
     * @param range the new range (`null` not permitted).
     */
    override fun setRange(range: Range) {
        setRange(range, true, true)
    }

    /**
     * Returns the earliest date visible on the axis.
     *
     * @return The date.
     *
     * @see .getMaximumDate
     */
    private val minimumDate: Instant
        private get() {
            val result: Instant
            val range: Range = getRange()!!
            if (range is DateRange) {
                val r: DateRange = range as DateRange
                result = r.getLowerDate()
            } else {
                result = Instant.fromEpochMilliseconds(range.lowerBound as Long)
            }
            return result
        }

    /**
     * Sets the range for the axis, if requested, sends an
     * to all registered listeners.  As a side-effect,
     * the auto-range flag is set to `false` (optional).
     *
     * @param range            the range (`null` not permitted).
     * @param turnOffAutoRange a flag that controls whether or not the auto
     * range is turned off.
     * @param notify           a flag that controls whether or not listeners are
     * notified.
     */
    override fun setRange(
        range: Range, turnOffAutoRange: Boolean,
        notify: Boolean
    ) {
        var range: Range? = range
        // usually the range will be a DateRange, but if it isn't do a
        // conversion...
        if (range !is DateRange) {
            range = DateRange(range!!)
        }
        super.setRange(range, turnOffAutoRange, notify)
    }

    /**
     * Returns the latest date visible on the axis.
     *
     * @return The date.
     *
     * @see .getMinimumDate
     */
    private val maximumDate: Instant
        private get() {
            val result: Instant
            val range: Range = getRange()!!
            if (range is DateRange) {
                val r: DateRange = range as DateRange
                result = r.getUpperDate()
            } else {
                result = Instant.fromEpochMilliseconds(range.upperBound as Long)
            }
            return result
        }

    /**
     * Sets the axis range and sends an to all
     * registered listeners.
     *
     * @param lower the lower bound for the axis.
     * @param upper the upper bound for the axis.
     */
    override fun setRange(lower: Double, upper: Double) {
        require(lower < upper)
        setRange(DateRange(lower, upper))
    }

    /**
     * Returns `true` if the axis hides this value, and
     * `false` otherwise.
     *
     * @return A value.
     */
    private val isHiddenValue: Boolean
        private get() = !timeline.containsDomainValue()

    /**
     * Calculates the value of the lowest visible tick on the axis.
     *
     * @param unit date unit to use.
     *
     * @return The value of the lowest visible tick on the axis.
     */
    private fun calculateLowestVisibleTickValue(unit: DateTickUnit?): Instant {
        return nextStandardDate(minimumDate, unit)
    }

    /**
     * Returns the previous "standard" date, for a given date and tick unit.
     *
     * @param date the reference date.
     * @param unit the tick unit.
     *
     * @return The previous "standard" date.
     */
    private fun previousStandardDate(date: Instant, unit: DateTickUnit?): Instant? {
        val calendar: CPCalendar = CPCalendar()
        calendar.time = date
        val count: Int = unit!!.count
        val current: Int = calendar.get(unit!!.calendarField)
        val value = count * current / count
        val years: Int
        val months: Int
        val days: Int
        val hours: Int
        val minutes: Int
        val seconds: Int
        val milliseconds: Int
        return when (unit.unit) {
            DateTickUnit.MILLISECOND -> {
                years = calendar.get(CPCalendar.YEAR)
                months = calendar.get(CPCalendar.MONTH)
                days = calendar.get(CPCalendar.DATE)
                hours = calendar.get(CPCalendar.HOUR_OF_DAY)
                minutes = calendar.get(CPCalendar.MINUTE)
                seconds = calendar.get(CPCalendar.SECOND)
                calendar.setAll(years, months, days, hours, minutes, seconds)
                calendar.set(CPCalendar.MILLISECOND, value)
                var mm: Instant = calendar.time!!
                if (mm.toEpochMilliseconds() >= date.toEpochMilliseconds()) {
                    calendar.set(CPCalendar.MILLISECOND, value - 1)
                    mm = calendar.time!!
                }
                mm
            }
            DateTickUnit.SECOND -> {
                years = calendar.get(CPCalendar.YEAR)
                months = calendar.get(CPCalendar.MONTH)
                days = calendar.get(CPCalendar.DATE)
                hours = calendar.get(CPCalendar.HOUR_OF_DAY)
                minutes = calendar.get(CPCalendar.MINUTE)
                milliseconds = if (tickMarkPosition == DateTickMarkPosition.Start) {
                    0
                } else if (tickMarkPosition == DateTickMarkPosition.Middle) 500 else 999
                calendar.set(CPCalendar.MILLISECOND, milliseconds)
                calendar.setAll(years, months, days, hours, minutes, value)
                var dd: Instant = calendar.time!!
                if (dd.toEpochMilliseconds() >= date.toEpochMilliseconds()) {
                    calendar.set(CPCalendar.SECOND, value - 1)
                    dd = calendar.time!!
                }
                dd
            }
            DateTickUnit.MINUTE -> {
                years = calendar.get(CPCalendar.YEAR)
                months = calendar.get(CPCalendar.MONTH)
                days = calendar.get(CPCalendar.DATE)
                hours = calendar.get(CPCalendar.HOUR_OF_DAY)
                seconds = if (tickMarkPosition == DateTickMarkPosition.Start) {
                    0
                } else if (tickMarkPosition == DateTickMarkPosition.Middle) 30 else 59
                calendar.clear(CPCalendar.MILLISECOND)
                calendar.setAll(years, months, days, hours, value, seconds)
                var d0: Instant = calendar.time!!
                if (d0.toEpochMilliseconds() >= date.toEpochMilliseconds()) {
                    calendar.set(CPCalendar.MINUTE, value - 1)
                    d0 = calendar.time!!
                }
                d0
            }
            DateTickUnit.HOUR -> {
                years = calendar.get(CPCalendar.YEAR)
                months = calendar.get(CPCalendar.MONTH)
                days = calendar.get(CPCalendar.DATE)
                if (tickMarkPosition == DateTickMarkPosition.Start) {
                    minutes = 0
                    seconds = 0
                } else if (tickMarkPosition == DateTickMarkPosition.Middle) {
                    minutes = 30
                    seconds = 0
                } else {
                    minutes = 59
                    seconds = 59
                }
                calendar.clear(CPCalendar.MILLISECOND)
                calendar.setAll(years, months, days, value, minutes, seconds)
                var d1: Instant = calendar.time!!
                if (d1.toEpochMilliseconds() >= date.toEpochMilliseconds()) {
                    calendar.set(CPCalendar.HOUR_OF_DAY, value - 1)
                    d1 = calendar.time!!
                }
                d1
            }
            DateTickUnit.DAY -> {
                years = calendar.get(CPCalendar.YEAR)
                months = calendar.get(CPCalendar.MONTH)
                if (tickMarkPosition == DateTickMarkPosition.Start) {
                    hours = 0
                    minutes = 0
                    seconds = 0
                } else if (tickMarkPosition == DateTickMarkPosition.Middle) {
                    hours = 12
                    minutes = 0
                    seconds = 0
                } else {
                    hours = 23
                    minutes = 59
                    seconds = 59
                }
                calendar.clear(CPCalendar.MILLISECOND)
                calendar.setAll(years, months, value, hours, 0, 0)
                // long result = calendar.getTimeInMillis();
                // won't work with JDK 1.3
                var d2: Instant = calendar.time!!
                if (d2.toEpochMilliseconds() >= date.toEpochMilliseconds()) {
                    calendar.set(CPCalendar.DATE, value - 1)
                    d2 = calendar.time!!
                }
                d2
            }
            DateTickUnit.MONTH -> {
                years = calendar.get(CPCalendar.YEAR)
                calendar.clear(CPCalendar.MILLISECOND)
                calendar.setAll(years, value, 1, 0, 0, 0)
                var month: Month = Month(
                    calendar.time, timeZone,
                    locale
                )
                var standardDate: Instant? = calculateDateForPosition(
                    month, tickMarkPosition
                )
                val millis: Long = standardDate!!.toEpochMilliseconds()
                if (millis >= date.toEpochMilliseconds()) {
                    month = month.previous() as Month
                    // need to peg the month in case the time zone isn't the
                    // default - see bug 2078057
                    month!!.peg(CPCalendar())
                    standardDate = calculateDateForPosition(
                        month, tickMarkPosition
                    )
                }
                standardDate
            }
            DateTickUnit.YEAR -> {
                if (tickMarkPosition == DateTickMarkPosition.Start) {
                    months = 0
                    days = 1
                } else if (tickMarkPosition == DateTickMarkPosition.Middle) {
                    months = 6
                    days = 1
                } else {
                    months = 11
                    days = 31
                }
                calendar.clear(CPCalendar.MILLISECOND)
                calendar.setAll(value, months, days, 0, 0, 0)
                var d3: Instant = calendar.time!!
                if (d3.toEpochMilliseconds() >= date.toEpochMilliseconds()) {
                    calendar.set(CPCalendar.YEAR, value - 1)
                    d3 = calendar.time!!
                }
                d3
            }
            else -> null
        }
    }

    /**
     * Returns the first "standard" date (based on the specified field and
     * units).
     *
     * @param date the reference date.
     * @param unit the date tick unit.
     *
     * @return The next "standard" date.
     */
    private fun nextStandardDate(date: Instant, unit: DateTickUnit?): Instant {
        val previous: Instant? = previousStandardDate(date, unit)
        val calendar: CPCalendar = CPCalendar()
        calendar.time = previous
        calendar.add(unit!!.calendarField, unit.multiple)
        return calendar.time!!
    }

    /**
     * Translates the data value to the display coordinates (Java 2D User Space)
     * of the chart.
     *
     * @param value the date to be plotted.
     * @param area  the rectangle (in Java2D space) where the data is to be
     * plotted.
     * @param edge  the axis location.
     *
     * @return The coordinate corresponding to the supplied data value.
     */
    protected override fun valueToJava2D(
        value: Double, area: Rectangle2D,
        edge: RectangleEdge
    ): Double {
        var value = value
        value = timeline.toTimelineValue(value.toLong()).toDouble()
        val range: DateRange = getRange() as DateRange
        val axisMin: Double = timeline.toTimelineValue(range.lowerMillis).toDouble()
        val axisMax: Double = timeline.toTimelineValue(range.upperMillis).toDouble()
        var result = 0.0
        if (RectangleEdge.Companion.isTopOrBottom(edge)) {
            val minX: Double = area.x
            val maxX: Double = area.maxX
            result = if (isInverted)
                maxX + (value - axisMin) / (axisMax - axisMin) * minX - maxX else minX+(value-axisMin) / (axisMax-axisMin) * maxX - minX
        } else if (RectangleEdge.Companion.isLeftOrRight(edge)) {
            val minY: Double = area.minY
            val maxY: Double = area.maxY
            result = if (isInverted) minY + (value - axisMin) / (axisMax - axisMin) * maxY - minY else maxY-(value-axisMin) / (axisMax-axisMin) * maxY - minY
        }
        return result
    }

    /**
     * Selects an appropriate tick value for the axis.  The strategy is to
     * display as many ticks as possible (selected from an array of 'standard'
     * tick units) without the labels overlapping.
     *
     * @param g2       the graphics device.
     * @param dataArea the area defined by the axes.
     * @param edge     the axis location.
     */
    private fun selectAutoTickUnit(
        g2: IGraphics,
        dataArea: Rectangle2D,
        edge: RectangleEdge
    ) {
        if (RectangleEdge.Companion.isTopOrBottom(edge)) {
            selectHorizontalAutoTickUnit(g2, dataArea, edge)
        } else if (RectangleEdge.Companion.isLeftOrRight(edge)) {
            selectVerticalAutoTickUnit(g2, dataArea, edge)
        }
    }

    /**
     * Selects an appropriate tick size for the axis.  The strategy is to
     * display as many ticks as possible (selected from a collection of
     * 'standard' tick units) without the labels overlapping.
     *
     * @param g2       the graphics device.
     * @param dataArea the area defined by the axes.
     * @param edge     the axis location.
     */
    private fun selectHorizontalAutoTickUnit(
        g2: IGraphics,
        dataArea: Rectangle2D, edge: RectangleEdge
    ) {
        val shift: Long = 0
        val zero = valueToJava2D(shift + 0.0, dataArea, edge)
        var tickLabelWidth = estimateMaximumTickLabelWidth(
            g2,
            tickUnit
        )

        // start with the current tick unit...
        val tickUnits: TickUnitSource = getStandardTickUnits()
        val unit1: TickUnit = tickUnits.getCeilingTickUnit(tickUnit as DateTickUnit)
        val x1 = valueToJava2D(shift + unit1.size, dataArea, edge)
        val unit1Width: Double = abs(x1 - zero)

        // then extrapolate...
        val guess: Double = tickLabelWidth / unit1Width * unit1.size
        var unit2: DateTickUnit = tickUnits.getCeilingTickUnit(guess) as DateTickUnit
        val x2 = valueToJava2D(shift + unit2.size, dataArea, edge)
        val unit2Width: Double = abs(x2 - zero)
        tickLabelWidth = estimateMaximumTickLabelWidth(g2, unit2)
        if (tickLabelWidth > unit2Width) {
            unit2 = tickUnits.getLargerTickUnit(unit2) as DateTickUnit
        }
        tickUnit = unit2
    }

    /**
     * Selects an appropriate tick size for the axis.  The strategy is to
     * display as many ticks as possible (selected from a collection of
     * 'standard' tick units) without the labels overlapping.
     *
     * @param g2       the graphics device.
     * @param dataArea the area in which the plot should be drawn.
     * @param edge     the axis location.
     */
    private fun selectVerticalAutoTickUnit(
        g2: IGraphics,
        dataArea: Rectangle2D,
        edge: RectangleEdge
    ) {

        // start with the current tick unit...
        val tickUnits: TickUnitSource = getStandardTickUnits()
        val zero = valueToJava2D(0.0, dataArea, edge)

        // start with a unit that is at least 1/10th of the axis length
        val estimate1: Double = getRange()!!.length / 10.0
        val candidate1: DateTickUnit = tickUnits.getCeilingTickUnit(estimate1) as DateTickUnit
        val labelHeight1 = estimateMaximumTickLabelHeight(g2, candidate1)
        val y1 = valueToJava2D(candidate1.size, dataArea, edge)
        val candidate1UnitHeight: Double = abs(y1 - zero)

        // now extrapolate based on label height and unit height...
        val estimate2: Double = labelHeight1 / candidate1UnitHeight * candidate1.size
        val candidate2: DateTickUnit = tickUnits.getCeilingTickUnit(estimate2) as DateTickUnit
        val labelHeight2 = estimateMaximumTickLabelHeight(g2, candidate2)
        val y2 = valueToJava2D(candidate2.size, dataArea, edge)
        val unit2Height: Double = abs(y2 - zero)

        // make final selection...
        val finalUnit: DateTickUnit
        finalUnit = if (labelHeight2 < unit2Height) candidate2 else (tickUnits.getLargerTickUnit(candidate2) as DateTickUnit)!!
        tickUnit = finalUnit
    }

    /**
     * Estimates the maximum width of the tick labels, assuming the specified
     * tick unit is used.
     *
     *
     * Rather than computing the string bounds of every tick on the axis, we
     * just look at two values: the lower bound and the upper bound for the
     * axis.  These two values will usually be representative.
     *
     * @param g2   the graphics device.
     * @param unit the tick unit to use for calculation.
     *
     * @return The estimated maximum width of the tick labels.
     */
    private fun estimateMaximumTickLabelWidth(
        g2: IGraphics,
        unit: DateTickUnit?
    ): Double {
        val tickLabelInsets: RectangleInsets = tickLabelInsets
        var result: Double = tickLabelInsets.left + tickLabelInsets.right
        if (isVerticalTickLabels) {
            // all tick labels have the same width (equal to the height of
            // the font)...
            result += headless.getStringHeight(g2, tickLabelFont, "Dummy")
        } else {
            // look at lower and upper bounds...
            val range: DateRange = getRange() as DateRange
            val lower: Instant = range.getLowerDate()
            val upper: Instant = range.getUpperDate()
            val lowerStr: String
            val upperStr: String
            val formatter: CPFormat<Any?>? = dateFormatOverride
            if (formatter != null) {
                lowerStr = formatter.format(lower)!!
                upperStr = formatter.format(upper)!!
            } else {
                lowerStr = unit!!.dateToString(lower)
                upperStr = unit!!.dateToString(upper)
            }
            val w1: Double = headless.getStringWidth(g2, tickLabelFont, lowerStr).toDouble()
            val w2: Double = headless.getStringWidth(g2, tickLabelFont, upperStr).toDouble()
            result += max(w1, w2)
        }
        return result
    }

    /**
     * Estimates the maximum width of the tick labels, assuming the specified
     * tick unit is used.
     *
     *
     * Rather than computing the string bounds of every tick on the axis, we
     * just look at two values: the lower bound and the upper bound for the
     * axis.  These two values will usually be representative.
     *
     * @param g2   the graphics device.
     * @param unit the tick unit to use for calculation.
     *
     * @return The estimated maximum width of the tick labels.
     */
    private fun estimateMaximumTickLabelHeight(
        g2: IGraphics,
        unit: DateTickUnit
    ): Double {
        val tickLabelInsets: RectangleInsets = tickLabelInsets
        var result: Double = tickLabelInsets.top + tickLabelInsets.bottom
        if (isVerticalTickLabels) {
            // look at lower and upper bounds...
            val range: DateRange = getRange() as DateRange
            val lower: Instant = range.getLowerDate()
            val upper: Instant = range.getUpperDate()
            val lowerStr: String
            val upperStr: String
            val formatter: CPFormat<Any?>? = dateFormatOverride
            if (formatter != null) {
                lowerStr = formatter.format(lower)!!
                upperStr = formatter.format(upper)!!
            } else {
                lowerStr = unit.dateToString(lower)
                upperStr = unit.dateToString(upper)
            }
            val w1: Double = headless.getStringWidth(g2, tickLabelFont, lowerStr).toDouble()
            val w2: Double = headless.getStringWidth(g2, tickLabelFont, upperStr).toDouble()
            result += max(w1, w2)
        } else {
            // all tick labels have the same width (equal to the height of
            // the font)...
            result += headless.getStringHeight(g2, tickLabelFont, "Dummy")
        }
        return result
    }

    /**
     * Calculates the positions of the tick labels for the axis, storing the
     * results in the tick label list (ready for drawing).
     *
     * @param g2       the graphics device.
     * @param dataArea the area in which the plot should be drawn.
     * @param edge     the location of the axis.
     *
     * @return A list of ticks.
     */
    override fun refreshTicks(
        g2: IGraphics,
        dataArea: Rectangle2D,
        edge: RectangleEdge
    ): List<ValueTick> {
        var result: List<ValueTick>? = null
        if (RectangleEdge.Companion.isTopOrBottom(edge)) {
            result = refreshTicksHorizontal(g2, dataArea, edge)
        } else if (RectangleEdge.Companion.isLeftOrRight(edge)) {
            result = refreshTicksVertical(g2, dataArea, edge)
        }
        return result!!
    }

    /**
     * Corrects the given tick date for the position setting.
     *
     * @param time     the tick date/time.
     * @param unit     the tick unit.
     * @param position the tick position.
     *
     * @return The adjusted time.
     */
    private fun correctTickDateForPosition(
        time: Instant?, unit: DateTickUnit?,
        position: DateTickMarkPosition
    ): Instant? {
        var result: Instant? = time
        when (unit!!.unit) {
            DateTickUnit.Companion.MILLISECOND, DateTickUnit.Companion.SECOND, DateTickUnit.Companion.MINUTE, DateTickUnit.Companion.HOUR, DateTickUnit.Companion.DAY -> {
            }
            DateTickUnit.Companion.MONTH -> result = calculateDateForPosition(
                Month(
                    time,
                    timeZone, locale
                ), position
            )
            DateTickUnit.Companion.YEAR -> result = calculateDateForPosition(
                Year(
                    time,
                    timeZone, locale
                ), position
            )
            else -> {
            }
        }
        return result
    }

    /**
     * Recalculates the ticks for the date axis.
     *
     * @param g2       the graphics device.
     * @param dataArea the area in which the data is to be drawn.
     * @param edge     the location of the axis.
     *
     * @return A list of ticks.
     */
    private fun refreshTicksHorizontal(
        g2: IGraphics,
        dataArea: Rectangle2D, edge: RectangleEdge
    ): List<ValueTick> {
        val result: MutableList<ValueTick> = ArrayList<ValueTick>()
        g2.setFont(tickLabelFont)
        if (isAutoTickUnitSelection) {
            selectAutoTickUnit(g2, dataArea, edge)
        }
        val unit: DateTickUnit? = tickUnit
        var tickDate: Instant? = calculateLowestVisibleTickValue(unit)
        val upperDate: Instant = maximumDate
        while (tickDate!! < upperDate) {
            // could add a flag to make the following correction optional...
            tickDate = correctTickDateForPosition(
                tickDate, unit,
                tickMarkPosition
            )
            val lowestTickTime: Long = tickDate!!.toEpochMilliseconds()
            val distance: Long = (unit!!.addToDate(tickDate, timeZone).toEpochMilliseconds()
                    - lowestTickTime)
            var minorTickSpaces: Int = minorTickCount
            if (minorTickSpaces <= 0) {
                minorTickSpaces = unit.minorTickCount
            }
            for (minorTick in 1 until minorTickSpaces) {
                val minorTickTime = lowestTickTime - distance * minorTick / minorTickSpaces
                if (minorTickTime > 0 && getRange()!!.contains(minorTickTime.toDouble())
                    && !isHiddenValue
                ) {
                    result.add(
                        DateTick(
                            ValueTick.TickType.Minor,
                            Instant.fromEpochMilliseconds(minorTickTime), "", Tick.TextAnchor.TopCenter,
                            Tick.TextAnchor.Center, 0.0
                        )
                    )
                }
            }
            if (isHiddenValue) {
                tickDate = unit.rollDate(tickDate, timeZone)
            } else {
                // work out the value, label and position
                val tickLabel: String
                val formatter: CPFormat<Any?>? = dateFormatOverride
                tickLabel = if (formatter != null) formatter.format(tickDate)!! else tickUnit!!.dateToString(tickDate)
                val anchor: Tick.TextAnchor
                val rotationAnchor: Tick.TextAnchor
                var angle = 0.0
                if (isVerticalTickLabels) {
                    anchor = Tick.TextAnchor.CenterRight
                    rotationAnchor = Tick.TextAnchor.CenterRight
                    angle = if (edge == RectangleEdge.Top) PI / 2.0 else -PI / 2.0
                } else {
                    if (edge == RectangleEdge.Top) {
                        anchor = Tick.TextAnchor.BottomCenter
                        rotationAnchor = Tick.TextAnchor.BottomCenter
                    } else {
                        anchor = Tick.TextAnchor.TopCenter
                        rotationAnchor = Tick.TextAnchor.TopCenter
                    }
                }
                val tick: ValueTick = DateTick(
                    tickDate, tickLabel, anchor,
                    rotationAnchor, angle
                )
                result.add(tick)
                val currentTickTime: Long = tickDate.toEpochMilliseconds()
                tickDate = unit.addToDate(tickDate, timeZone)
                val nextTickTime: Long = tickDate.toEpochMilliseconds()
                for (minorTick in 1 until minorTickSpaces) {
                    val minorTickTime = (currentTickTime
                            + (nextTickTime - currentTickTime)
                            * minorTick / minorTickSpaces)
                    if (getRange()!!.contains(minorTickTime.toDouble())
                        && !isHiddenValue
                    ) {
                        result.add(
                            DateTick(
                                ValueTick.TickType.Minor,
                                Instant.fromEpochMilliseconds(minorTickTime), "",
                                Tick.TextAnchor.TopCenter, Tick.TextAnchor.Center,
                                0.0
                            )
                        )
                    }
                }
            }
        }
        return result
    }

    /**
     * Recalculates the ticks for the date axis.
     *
     * @param g2       the graphics device.
     * @param dataArea the area in which the plot should be drawn.
     * @param edge     the location of the axis.
     *
     * @return A list of ticks.
     */
    private fun refreshTicksVertical(
        g2: IGraphics,
        dataArea: Rectangle2D, edge: RectangleEdge
    ): List<ValueTick> {
        val result: MutableList<ValueTick> = ArrayList<ValueTick>()
        g2.setFont(tickLabelFont)
        if (isAutoTickUnitSelection) {
            selectAutoTickUnit(g2, dataArea, edge)
        }
        val unit: DateTickUnit? = tickUnit
        var tickDate: Instant? = calculateLowestVisibleTickValue(unit)
        val upperDate: Instant = maximumDate
        while (tickDate!! < upperDate) {

            // could add a flag to make the following correction optional...
            tickDate = correctTickDateForPosition(
                tickDate, unit,
                tickMarkPosition
            )
            val lowestTickTime: Long = tickDate!!.toEpochMilliseconds()
            val distance: Long = (unit!!.addToDate(tickDate, timeZone).toEpochMilliseconds()
                    - lowestTickTime)
            var minorTickSpaces: Int = minorTickCount
            if (minorTickSpaces <= 0) {
                minorTickSpaces = unit.minorTickCount
            }
            for (minorTick in 1 until minorTickSpaces) {
                val minorTickTime = lowestTickTime - distance * minorTick / minorTickSpaces
                if (minorTickTime > 0 && getRange()!!.contains(minorTickTime.toDouble())
                    && !isHiddenValue
                ) {
                    result.add(
                        DateTick(
                            ValueTick.TickType.Minor,
                            Instant.fromEpochMilliseconds(minorTickTime), "", Tick.TextAnchor.TopCenter,
                            Tick.TextAnchor.Center, 0.0
                        )
                    )
                }
            }
            if (isHiddenValue) {
                tickDate = unit.rollDate(tickDate, timeZone)
            } else {
                // work out the value, label and position
                val tickLabel: String
                val formatter: CPFormat<Any?>? = dateFormatOverride
                tickLabel = if (formatter != null) formatter!!.format(tickDate)!! else tickUnit!!.dateToString(tickDate)
                val anchor: Tick.TextAnchor
                val rotationAnchor: Tick.TextAnchor
                var angle = 0.0
                if (isVerticalTickLabels) {
                    anchor = Tick.TextAnchor.BottomCenter
                    rotationAnchor = Tick.TextAnchor.BottomCenter
                    angle = if (edge == RectangleEdge.Left) -PI / 2.0 else PI / 2.0
                } else {
                    if (edge == RectangleEdge.Left) {
                        anchor = Tick.TextAnchor.CenterRight
                        rotationAnchor = Tick.TextAnchor.CenterRight
                    } else {
                        anchor = Tick.TextAnchor.CenterLeft
                        rotationAnchor = Tick.TextAnchor.CenterLeft
                    }
                }
                val tick: ValueTick = DateTick(
                    tickDate, tickLabel, anchor,
                    rotationAnchor, angle
                )
                result.add(tick)
                val currentTickTime: Long = tickDate!!.toEpochMilliseconds()
                tickDate = unit!!.addToDate(tickDate, timeZone)
                val nextTickTime: Long = tickDate.toEpochMilliseconds()
                for (minorTick in 1 until minorTickSpaces) {
                    val minorTickTime = (currentTickTime
                            + (nextTickTime - currentTickTime)
                            * minorTick / minorTickSpaces)
                    if (getRange()!!.contains(minorTickTime.toDouble())
                        && !isHiddenValue
                    ) {
                        result.add(
                            DateTick(
                                ValueTick.TickType.Minor,
                                Instant.fromEpochMilliseconds(minorTickTime), "",
                                Tick.TextAnchor.TopCenter, Tick.TextAnchor.Center,
                                0.0
                            )
                        )
                    }
                }
            }
        }
        return result
    }

    /**
     * A timeline that includes all milliseconds (as defined by
     * `Instant`) in the real time line.
     */
    private class DefaultTimeline : Timeline {
        /**
         * Converts a millisecond into a timeline value.
         *
         * @param millisecond the millisecond.
         *
         * @return The timeline value.
         */
        override fun toTimelineValue(millisecond: Long): Long {
            return millisecond
        }

        /**
         * Returns `true` if the timeline includes the specified
         * domain value.
         *
         * @return `true`.
         */
        override fun containsDomainValue(): Boolean {
            return true
        }
    }

    /**
     * Used to indicate the required position of tick marks on a date axis relative
     * to the underlying time period.
     */
    internal enum class DateTickMarkPosition {
        /** Start of period.  */
        Start,

        /** Middle of period.  */
        Middle,

        /** End of period.  */
        End
    }

    companion object {
        /** The default axis range.  */
        private val DEFAULT_DATE_RANGE: DateRange = DateRange()

        /** The default date tick unit.  */
        private var DEFAULT_DATE_TICK_UNIT: DateTickUnit? = null

        /** A static default timeline shared by all standard DateAxis  */
        private val DEFAULT_TIMELINE: Timeline = DefaultTimeline()

        /**
         * Returns a collection of standard date tick units.  This collection will
         * be used by default, but you are free to create your own collection if
         * you want to (see the
         * method inherited
         * from the [ValueAxis] class).
         *
         * @param zone   the time zone (`null` not permitted).
         * @param locale the locale (`null` not permitted).
         *
         * @return A collection of standard date tick units.
         */
        private fun createStandardDateTickUnits(
            zone: CPTimeZone?,
            locale: CPLocale?, formatFactory: FormatFactory
        ): TickUnitSource {
            val units: TickUnits = TickUnits(formatFactory)

            // date formatters
//            val f1: CPFormat<Any?> = formatFactory.createDateFormat("HH:mm:ss.SSS", locale, zone)
//            val f2: CPFormat<Any?> = formatFactory.createDateFormat("HH:mm:ss", locale, zone)
//            val f3: CPFormat<Any?> = formatFactory.createDateFormat("HH:mm", locale, zone)
//            val f4: CPFormat<Any?> = formatFactory.createDateFormat("d-MMM, HH:mm", locale, zone)
//            val f5: CPFormat<Any?> = formatFactory.createDateFormat("d-MMM", locale, zone)
//            val f6: CPFormat<Any?> = formatFactory.createDateFormat("MMM-yyyy", locale, zone)
//            val f7: CPFormat<Any?> = formatFactory.createDateFormat("yyyy", locale, zone)

            // "HH:mm:ss.SSS"
            val f1: CPFormat<Any?> = DateTimeFormat(LocalDateTime.Format {
                hour()
                char(':')
                minute()
                char(':')
                second()
                char('.')
                secondFraction(3)
            })
            // "HH:mm:ss"
            val f2: CPFormat<Any?> = DateTimeFormat(LocalDateTime.Format {
                hour()
                char(':')
                minute()
                char(':')
                second()
            })
            // "HH:mm"
            val f3: CPFormat<Any?> = DateTimeFormat(LocalDateTime.Format {
                hour()
                char(':')
                minute()
            })
            // "d-MMM, HH:mm"
            val f4: CPFormat<Any?> = DateTimeFormat(LocalDateTime.Format {
                dayOfMonth(padding = Padding.NONE)  // dd
                char('-')                          // .
                monthName(MonthNames.ENGLISH_ABBREVIATED)
                char(',')
                char(' ')
                hour()
                char(':')
                minute()
            })
            // "d-MMM"
            val f5: CPFormat<Any?> = DateTimeFormat(LocalDateTime.Format {
                dayOfMonth(padding = Padding.NONE)  // dd
                char('-')                          // .
                monthName(MonthNames.ENGLISH_ABBREVIATED)
            })
            // "MMM-yyyy"
            val f6: CPFormat<Any?> = DateTimeFormat(LocalDateTime.Format {
                monthName(MonthNames.ENGLISH_ABBREVIATED)
                char('-')
                year()
            })
            // "yyyy"
            val f7: CPFormat<Any?> = DateTimeFormat(LocalDateTime.Format {
                year()
            })

            // milliseconds
            units.add(DateTickUnit(DateTickUnitType.Companion.MILLISECOND, f1))
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.MILLISECOND, 5,
                    DateTickUnitType.Companion.MILLISECOND, 1, f1
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.MILLISECOND, 10,
                    DateTickUnitType.Companion.MILLISECOND, 1, f1
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.MILLISECOND, 25,
                    DateTickUnitType.Companion.MILLISECOND, 5, f1
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.MILLISECOND, 50,
                    DateTickUnitType.Companion.MILLISECOND, 10, f1
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.MILLISECOND, 100,
                    DateTickUnitType.Companion.MILLISECOND, 10, f1
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.MILLISECOND, 250,
                    DateTickUnitType.Companion.MILLISECOND, 10, f1
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.MILLISECOND, 500,
                    DateTickUnitType.Companion.MILLISECOND, 50, f1
                )
            )

            // seconds
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.SECOND, 1,
                    DateTickUnitType.Companion.MILLISECOND, 50, f2
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.SECOND, 5,
                    DateTickUnitType.Companion.SECOND, 1, f2
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.SECOND, 10,
                    DateTickUnitType.Companion.SECOND, 1, f2
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.SECOND, 30,
                    DateTickUnitType.Companion.SECOND, 5, f2
                )
            )

            // minutes
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.MINUTE, 1,
                    DateTickUnitType.Companion.SECOND, 5, f3
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.MINUTE, 2,
                    DateTickUnitType.Companion.SECOND, 10, f3
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.MINUTE, 5,
                    DateTickUnitType.Companion.MINUTE, 1, f3
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.MINUTE, 10,
                    DateTickUnitType.Companion.MINUTE, 1, f3
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.MINUTE, 15,
                    DateTickUnitType.Companion.MINUTE, 5, f3
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.MINUTE, 20,
                    DateTickUnitType.Companion.MINUTE, 5, f3
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.MINUTE, 30,
                    DateTickUnitType.Companion.MINUTE, 5, f3
                )
            )

            // hours
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.HOUR, 1,
                    DateTickUnitType.Companion.MINUTE, 5, f3
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.HOUR, 2,
                    DateTickUnitType.Companion.MINUTE, 10, f3
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.HOUR, 4,
                    DateTickUnitType.Companion.MINUTE, 30, f3
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.HOUR, 6,
                    DateTickUnitType.Companion.HOUR, 1, f3
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.HOUR, 12,
                    DateTickUnitType.Companion.HOUR, 1, f4
                )
            )

            // days
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.DAY, 1,
                    DateTickUnitType.Companion.HOUR, 1, f5
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.DAY, 2,
                    DateTickUnitType.Companion.HOUR, 1, f5
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.DAY, 7,
                    DateTickUnitType.Companion.DAY, 1, f5
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.DAY, 15,
                    DateTickUnitType.Companion.DAY, 1, f5
                )
            )

            // months
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.MONTH, 1,
                    DateTickUnitType.Companion.DAY, 1, f6
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.MONTH, 2,
                    DateTickUnitType.Companion.DAY, 1, f6
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.MONTH, 3,
                    DateTickUnitType.Companion.MONTH, 1, f6
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.MONTH, 4,
                    DateTickUnitType.Companion.MONTH, 1, f6
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.MONTH, 6,
                    DateTickUnitType.Companion.MONTH, 1, f6
                )
            )

            // years
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.YEAR, 1,
                    DateTickUnitType.Companion.MONTH, 1, f7
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.YEAR, 2,
                    DateTickUnitType.Companion.MONTH, 3, f7
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.YEAR, 5,
                    DateTickUnitType.Companion.YEAR, 1, f7
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.YEAR, 10,
                    DateTickUnitType.Companion.YEAR, 1, f7
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.YEAR, 25,
                    DateTickUnitType.Companion.YEAR, 5, f7
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.YEAR, 50,
                    DateTickUnitType.Companion.YEAR, 10, f7
                )
            )
            units.add(
                DateTickUnit(
                    DateTickUnitType.Companion.YEAR, 100,
                    DateTickUnitType.Companion.YEAR, 20, f7
                )
            )
            return units
        }

        /**
         * Returns a [Date] corresponding to the specified position
         * within a [RegularTimePeriod].
         *
         * @param period   the period.
         * @param position the position (`null` not permitted).
         *
         * @return A date.
         */
        private fun calculateDateForPosition(
            period: RegularTimePeriod?,
            position: DateTickMarkPosition?
        ): Instant? {
            require(position != null)
            var result: Instant? = null
            if (position == DateTickMarkPosition.Start) {
                result = Instant.fromEpochMilliseconds(period!!.firstMillisecond)
            } else if (position == DateTickMarkPosition.Middle) {
                result = Instant.fromEpochMilliseconds(period!!.middleMillisecond)
            } else if (position == DateTickMarkPosition.End) {
                result = Instant.fromEpochMilliseconds(period!!.lastMillisecond)
            }
            return result
        }
    }

    /**
     * Creates a date axis. A timeline is specified for the axis. This allows
     * special transformations to occur between a domain of values and the
     * values included in the axis.
     *
     * @param label  the axis label (`null` permitted).
     * @param zone   the time zone.
     * @param locale the locale (`null` not permitted).
     */
    init {
        if (DEFAULT_DATE_TICK_UNIT == null) {
            DEFAULT_DATE_TICK_UNIT = DateTickUnit(
                DateTickUnitType.Companion.DAY,
                DateTimeFormat(LocalDateTime.Format {
                    dayOfMonth(Padding.NONE)
                    char('.')
                    monthNumber(Padding.NONE)
                    char('.')
                    year()
                })
            )
        }
        tickUnit = DEFAULT_DATE_TICK_UNIT
        setRange(DEFAULT_DATE_RANGE, false, false)
        dateFormatOverride = null
        timeZone = zone
        this.locale = locale
        timeline = DEFAULT_TIMELINE
    }
}