package com.macrofocus.plot.guide

import org.mkui.geom.Rectangle2D

/**
 * Represents the insets for a rectangle, specified in absolute or relative
 * terms. This class is immutable.
 */
class RectangleInsets(
    top: Double, left: Double,
    bottom: Double, right: Double
) {
    /** Absolute or relative units.  */
    private val unitType: UnitType
    /**
     * Returns the top insets.
     *
     * @return The top insets.
     */
    /** The top insets.  */
    val top: Double
    /**
     * Returns the left insets.
     *
     * @return The left insets.
     */
    /** The left insets.  */
    val left: Double
    /**
     * Returns the bottom insets.
     *
     * @return The bottom insets.
     */
    /** The bottom insets.  */
    val bottom: Double
    /**
     * Returns the right insets.
     *
     * @return The right insets.
     */
    /** The right insets.  */
    val right: Double

    /**
     * Creates an outset rectangle.
     *
     * @param base the base rectangle (`null` not permitted).
     *
     * @return An outset rectangle.
     */
    fun createOutsetRectangle(base: Rectangle2D): Rectangle2D {
        val topMargin: Double
        val bottomMargin: Double
        topMargin = calculateTopOutset(base.height)
        bottomMargin = calculateBottomOutset(base.height)
        val leftMargin: Double
        val rightMargin: Double
        leftMargin = calculateLeftOutset(base.width)
        rightMargin = calculateRightOutset(base.width)
        return Rectangle2D.Double(
            base.x - leftMargin,
            base.y - topMargin,
            base.width + leftMargin + rightMargin,
            base.height + topMargin + bottomMargin
        )
    }

    /**
     * Returns the top margin.
     *
     * @param height the height of the base rectangle.
     *
     * @return The top margin (in Java2D units).
     */
    private fun calculateTopOutset(height: Double): Double {
        var result = top
        if (unitType == UnitType.Relative) {
            result = height / (1 - top - bottom) * top
        }
        return result
    }

    /**
     * Returns the bottom margin.
     *
     * @param height the height of the base rectangle.
     *
     * @return The bottom margin (in Java2D units).
     */
    private fun calculateBottomOutset(height: Double): Double {
        var result = bottom
        if (unitType == UnitType.Relative) {
            result = height / (1 - top - bottom) * bottom
        }
        return result
    }

    /**
     * Returns the left margin.
     *
     * @param width the width of the base rectangle.
     *
     * @return The left margin (in Java2D units).
     */
    private fun calculateLeftOutset(width: Double): Double {
        var result = left
        if (unitType == UnitType.Relative) {
            result = width / (1 - left - right) * left
        }
        return result
    }

    /**
     * Returns the right margin.
     *
     * @param width the width of the base rectangle.
     *
     * @return The right margin (in Java2D units).
     */
    private fun calculateRightOutset(width: Double): Double {
        var result = right
        if (unitType == UnitType.Relative) {
            result = width / (1 - left - right) * right
        }
        return result
    }

    /** Used to indicate absolute or relative units.  */
    internal enum class UnitType {
        /** Absolute.  */
        Absolute,

        /** Relative.  */
        Relative
    }

    /**
     * Creates a new instance.
     *
     * @param top    the top insets.
     * @param left   the left insets.
     * @param bottom the bottom insets.
     * @param right  the right insets.
     */
    init {
        unitType = UnitType.Absolute
        this.top = top
        this.bottom = bottom
        this.left = left
        this.right = right
    }
}