package com.macrofocus.plot.guide

import com.macrofocus.plot.datetime.CPCalendar.Companion.APRIL
import com.macrofocus.plot.datetime.CPCalendar.Companion.AUGUST
import com.macrofocus.plot.datetime.CPCalendar.Companion.DECEMBER
import com.macrofocus.plot.datetime.CPCalendar.Companion.FEBRUARY
import com.macrofocus.plot.datetime.CPCalendar.Companion.JANUARY
import com.macrofocus.plot.datetime.CPCalendar.Companion.JULY
import com.macrofocus.plot.datetime.CPCalendar.Companion.JUNE
import com.macrofocus.plot.datetime.CPCalendar.Companion.MARCH
import com.macrofocus.plot.datetime.CPCalendar.Companion.MAY
import com.macrofocus.plot.datetime.CPCalendar.Companion.NOVEMBER
import com.macrofocus.plot.datetime.CPCalendar.Companion.OCTOBER
import com.macrofocus.plot.datetime.CPCalendar.Companion.SEPTEMBER

/**
 * An abstract class that defines our requirements for manipulating dates,
 * without tying down a particular implementation.
 *
 *
 * Requirement 1 : match at least what Excel does for dates;
 * Requirement 2 : the date represented by the class is immutable;
 *
 *
 * Why not just use java.util.Date?  We will, when it makes sense.  At times,
 * java.util.Date can be *too* precise - it represents an instant in time,
 * accurate to 1/1000th of a second (with the date itself depending on the
 * time-zone).  Sometimes we just want to represent a particular day (e.g. 21
 * January 2015) without concerning ourselves about the time of day, or the
 * time-zone, or anything else.  That's what we've defined SerialDate for.
 *
 *
 * You can call getInstance() to get a concrete subclass of SerialDate,
 * without worrying about the exact implementation.
 */
internal object SerialDate : com.macrofocus.plot.guide.MonthConstants {
    /** Date format symbols.  */
    private val DATE_FORMAT_SYMBOLS =
        arrayOf("January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December")

    /** The number of days in each month in non leap years.  */
    private val LAST_DAY_OF_MONTH = intArrayOf(0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31)

    /**
     * Returns a string representing the supplied month.
     *
     *
     * The string returned is the long or short form of the month name taken
     * from the default locale.
     *
     * @param month the month.
     *
     * @return a string representing the supplied month.
     */
    fun monthCodeToString(month: Int): String {

        // check arguments...
        require(isValidMonthCode(month))
        val months: Array<String>
        months = DATE_FORMAT_SYMBOLS
        return months[month - 1]
    }

    /**
     * Returns true if the supplied integer code represents a valid month.
     *
     * @param code the code being checked for validity.
     *
     * @return `true` if the supplied integer code represents a
     * valid month.
     */
    private fun isValidMonthCode(code: Int): Boolean {
        return when (code) {
            JANUARY, FEBRUARY, MARCH, APRIL, MAY, JUNE, JULY, AUGUST, SEPTEMBER, OCTOBER, NOVEMBER, DECEMBER -> true
            else -> false
        }
    }

    /**
     * Returns the number of the last day of the month, taking into account
     * leap years.
     *
     * @param month the month.
     * @param yyyy  the year (in the range 1900 to 9999).
     *
     * @return the number of the last day of the month.
     */
    fun lastDayOfMonth(month: Int, yyyy: Int): Int {
        val result = LAST_DAY_OF_MONTH[month]
        return if ((month - 1) != FEBRUARY) {
            result
        } else if (isLeapYear(yyyy)) result + 1 else result
    }

    /**
     * Determines whether or not the specified year is a leap year.
     *
     * @param yyyy the year (in the range 1900 to 9999).
     *
     * @return `true` if the specified year is a leap year.
     */
    private fun isLeapYear(yyyy: Int): Boolean {
        return if (yyyy % 4 != 0) {
            false
        } else if (yyyy % 400 == 0) {
            true
        } else yyyy % 100 != 0
    }
}