/*
 * Copyright (c) 2005 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.slider

import com.macrofocus.common.math.Histogram
import kotlin.math.max
import kotlin.math.min

class SimpleRangeSliderModel : com.macrofocus.slider.AbstractRangeSliderModel {
    override var currentMinimum: Double
        private set
    override var currentMaximum: Double
        private set
    override var currentExtent: Double
        private set
    override var minimum: Double
        private set
    override var maximum: Double
        private set
    override var minimumExtent: Double
        private set
    override var maximumExtent: Double
        private set
    private var sliderListeners: MutableList<com.macrofocus.slider.RangeSliderListener>
    override var histogram: Histogram? = null

    constructor(min: Double, max: Double) {
        minimum = min
        maximum = max
        currentMinimum = min
        currentMaximum = max
        currentExtent = max(currentMinimum, currentMaximum) - min(currentMinimum, currentMaximum)
        minimumExtent = 0.0
        maximumExtent = max(min, max) - min(min, max)
        sliderListeners = ArrayList<com.macrofocus.slider.RangeSliderListener>()
    }

    constructor(min: Double, max: Double, minExtent: Double, maxExtent: Double) {
        minimum = min
        maximum = max
        currentMinimum = min
        currentMaximum = max
        currentExtent = max(currentMinimum, currentMaximum) - min(currentMinimum, currentMaximum)
        minimumExtent = minExtent
        maximumExtent = maxExtent
        sliderListeners = ArrayList<com.macrofocus.slider.RangeSliderListener>()
    }

    override fun setValues(source: Any, rangeMin: Double, rangeMax: Double) {
        if (currentMinimum != rangeMin || currentMaximum != rangeMax) {
            val previousRangeMin = currentMinimum
            val previousRangeMax = currentMaximum
            currentMinimum = rangeMin
            currentMaximum = rangeMax
            currentExtent = max(rangeMin, rangeMax) - min(rangeMin, rangeMax)
            val eventRange: com.macrofocus.slider.RangeSliderEvent =
                com.macrofocus.slider.RangeSliderEvent(this, source, currentMinimum, currentMaximum, previousRangeMin, previousRangeMax)
            notifySliderChanged(eventRange)
        }
    }

    override fun setMinMax(source: Any, min: Double, max: Double) {
        if (minimum != min || maximum != max) {
            minimum = min
            maximum = max
            val eventRange: RangeSliderEvent =
                RangeSliderEvent(this, source, currentMinimum, currentMaximum, currentMinimum, currentMaximum)
            notifySliderScaleChanged(eventRange)
        }
    }

    override val extent: Double
        get() = max(minimum, maximum) - min(minimum, maximum)
    override val upperMinimumValue: Double
        get() = minimum
    override val upperMaximumValue: Double
        get() = maximum
    override val lowerMinimumValue: Double
        get() = minimum
    override val lowerMaximumValue: Double
        get() = maximum

    override fun reset(source: Any) {
        val previousRangeMin = currentMinimum
        val previousRangeMax = currentMaximum
        if (!isInvertedScale) {
            var tmpRangeMax = maximum
            var tmpRangeMin = minimum
            if (tmpRangeMax - tmpRangeMin > maximumExtent) {
                tmpRangeMax = currentMinimum + maximumExtent
                tmpRangeMin = currentMinimum
                if (tmpRangeMax > maximum) {
                    tmpRangeMin = tmpRangeMin - (tmpRangeMax - maximum)
                    tmpRangeMax = maximum
                }
            }
            currentMaximum = tmpRangeMax
            currentMinimum = tmpRangeMin
            if (currentMinimum != previousRangeMin || currentMaximum != previousRangeMax) {
                notifySliderChanged(com.macrofocus.slider.RangeSliderEvent(this, source, currentMinimum, currentMaximum, previousRangeMin, previousRangeMax))
            }
        } else {
            var tmpRangeMax = maximum
            var tmpRangeMin = minimum
            if (tmpRangeMin - tmpRangeMax > maximumExtent) {
                tmpRangeMax = currentMinimum
                tmpRangeMin = currentMinimum + maximumExtent
                if (tmpRangeMin > minimum) {
                    tmpRangeMax = tmpRangeMax - (tmpRangeMin - minimum)
                    tmpRangeMin = minimum
                }
            }
            currentMaximum = tmpRangeMax
            currentMinimum = tmpRangeMin
            if (currentMinimum != previousRangeMin || currentMaximum != previousRangeMax) {
                notifySliderChanged(com.macrofocus.slider.RangeSliderEvent(this, source, currentMinimum, currentMaximum, previousRangeMin, previousRangeMax))
            }
        }
    }

    override fun addSliderListener(rangeSliderListener: RangeSliderListener) {
        sliderListeners.add(rangeSliderListener)
    }

    override fun removeSliderListener(rangeSliderListener: RangeSliderListener) {
        sliderListeners.remove(rangeSliderListener)
    }

    private fun notifySliderChanged(eventRange: RangeSliderEvent) {
        for (listener in sliderListeners) {
            listener.sliderChanged(eventRange)
        }
    }

    private fun notifySliderScaleChanged(eventRange: RangeSliderEvent) {
        for (listener in sliderListeners) {
            listener.sliderScaleChanged(eventRange)
        }
    }

    override fun toString(): String {
        return this::class.simpleName
            .toString() + "[rangeMin=" + currentMinimum + ",rangeMax=" + currentMaximum + ",amount=" + currentExtent + ",min=" + minimum + ",max=" + maximum + "]"
    }

    override val isInvertedScale: Boolean
        get() = minimum > maximum
}