@file:OptIn(ExperimentalTime::class)

package com.macrofocus.plot.guide

import kotlin.time.Instant
import kotlinx.datetime.LocalDateTime
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toInstant
import kotlin.time.ExperimentalTime

/**
 * A range specified in terms of two `java.util.Date` objects.
 * Instances of this class are immutable.
 */
internal class DateRange : Range {
    /**
     * Returns the lower bound of the range in milliseconds.
     *
     * @return The lower bound.
     *
     * @see .getLowerDate
     */
    /** The lower bound for the range.  */
    val lowerMillis: Long
    /**
     * Returns the upper bound of the range in milliseconds.
     *
     * @return The upper bound.
     *
     * @see .getUpperDate
     */
    /** The upper bound for the range.  */
    val upperMillis: Long

    /** Default constructor.  */
    constructor() : this(LocalDateTime(2024, 1, 1, 0, 0).toInstant(TimeZone.UTC).toEpochMilliseconds().toDouble(), LocalDateTime(2025, 1, 1, 0, 0).toInstant(TimeZone.UTC).toEpochMilliseconds().toDouble()) {}

    /**
     * Constructs a new range.
     *
     * @param lower the lower bound (`null` not permitted).
     * @param upper the upper bound (`null` not permitted).
     */
    private constructor(lower: Instant, upper: Instant) : super(lower.toEpochMilliseconds().toDouble(), upper.toEpochMilliseconds().toDouble()) {
        lowerMillis = lower.toEpochMilliseconds()
        upperMillis = upper.toEpochMilliseconds()
    }

    /**
     * Constructs a new range that is based on another [Range].  The
     * other range does not have to be a .  If it is not, the
     * upper and lower bounds are evaluated as milliseconds since midnight
     * GMT, 1-Jan-1970.
     *
     * @param other the other range (`null` not permitted).
     */
    constructor(other: Range) : this(other.lowerBound, other.upperBound) {}

    /**
     * Constructs a new range using two values that will be interpreted as
     * "milliseconds since midnight GMT, 1-Jan-1970".
     *
     * @param lower the lower (oldest) date.
     * @param upper the upper (most recent) date.
     */
    constructor(lower: Double, upper: Double) : super(lower, upper) {
        lowerMillis = lower.toLong()
        upperMillis = upper.toLong()
    }

    /**
     * Returns the lower (earlier) date for the range.
     *
     * @return The lower date for the range.
     *
     * @see .getLowerMillis
     */
    fun getLowerDate(): Instant {
        return Instant.fromEpochMilliseconds(lowerMillis)
    }

    /**
     * Returns the upper (later) date for the range.
     *
     * @return The upper date for the range.
     *
     * @see .getUpperMillis
     */
    fun getUpperDate(): Instant {
        return Instant.fromEpochMilliseconds(upperMillis)
    }
}