@file:OptIn(ExperimentalTime::class)

package com.macrofocus.plot.guide

import com.macrofocus.common.date.CPTimeZone
import com.macrofocus.common.format.CPFormat
import com.macrofocus.plot.datetime.CPCalendar
import kotlin.time.ExperimentalTime
import kotlin.time.Instant

/**
 * A tick unit for use by subclasses of [DateAxis].  Instances of this
 * class are immutable.
 */
class DateTickUnit(
    unitType: DateTickUnitType, multiple: Int,
    rollUnitType: DateTickUnitType, rollMultiple: Int,
    formatter: CPFormat<Any?>?
) : TickUnit(getMillisecondCount(unitType, multiple).toDouble()) {
    /** The units.  */
    private val unitType: DateTickUnitType
    /**
     * Returns the unit count.
     *
     * @return The unit count.
     *
     */
    val count: Int
        get() {
            return multiple
        }

    /**
     * Returns the unit multiple.
     *
     * @return The unit multiple (always > 0).
     */
    /** The unit count.  */
    @get:Deprecated("Use {@link #getMultiple()}.")
    val multiple: Int
        get() = field

    /** The roll unit type.  */
    private val rollUnitType: DateTickUnitType

    /** The roll count.  */
    private val rollCount: Int

    /** The date formatter.  */
    private val formatter: CPFormat<Any?>?
    /**
     * Returns the date unit.  This will be one of the constants
     * `YEAR`, `MONTH`, `DAY`,
     * `HOUR`, `MINUTE`, `SECOND` or
     * `MILLISECOND`, defined by this class.  Note that these
     * constants do NOT correspond to those defined in Java's
     * `Calendar` class.
     *
     * @return The date unit.
     *
     */
    /**
     * The unit.
     *
     */
    @get:Deprecated("Use the getUnitType() method.")
    @Deprecated("Use the unitType field.")
    val unit: Int

    /**
     * Creates a new date tick unit.
     *
     * @param unitType  the unit type (`null` not permitted).
     * @param formatter the date formatter (`null` not permitted).
     */
    constructor(
        unitType: DateTickUnitType,
        formatter: CPFormat<Any?>?
    ) : this(unitType, 1, unitType, 1, formatter) {
    }

    /**
     * Formats a value.
     *
     * @param milliseconds date in milliseconds since 01-01-1970.
     *
     * @return The formatted date.
     */
    override fun valueToString(milliseconds: Double): String {
        return formatter!!.format(Instant.fromEpochMilliseconds(milliseconds.toLong()))!!
    }

    /**
     * Formats a date using the tick unit's formatter.
     *
     * @param date the date.
     *
     * @return The formatted date.
     */
    fun dateToString(date:Instant?): String {
        return formatter!!.format(date)!!
    }

    /**
     * Calculates a new date by adding this unit to the base date.
     *
     * @param base the base date.
     * @param zone the time zone for the date calculation.
     *
     * @return A new date one unit after the base date.
     */
    fun addToDate(base: Instant?, zone: CPTimeZone?): Instant {
        // as far as I know, the Locale for the calendar only affects week
        // number calculations, and since DateTickUnit doesn't do week
        // arithmetic, the default locale (whatever it is) should be fine
        // here...
        val calendar: CPCalendar = CPCalendar()
        calendar.time = base
        calendar.add(unitType.calendarField, multiple)
        return calendar.time!!
    }

    /**
     * Rolls the date forward by the amount specified by the roll unit and
     * count.
     *
     * @param base the base date.
     * @param zone the time zone.
     *
     * @return The rolled date.
     */
    fun rollDate(base: Instant?, zone: CPTimeZone?): Instant {
        // as far as I know, the Locale for the calendar only affects week
        // number calculations, and since DateTickUnit doesn't do week
        // arithmetic, the default locale (whatever it is) should be fine
        // here...
        val calendar: CPCalendar = CPCalendar()
        calendar.time = base
        calendar.add(rollUnitType.calendarField, rollCount)
        return calendar.time!!
    }

    /**
     * Returns a field code that can be used with the `Calendar`
     * class.
     *
     * @return The field code.
     */
    val calendarField: Int
        get() = unitType.calendarField

    companion object {
        /**
         * A constant for years.
         *
         */
        @Deprecated("As of version 1.0.13, use {@link DateTickUnitType} instead.")
        val YEAR = 0

        /**
         * A constant for months.
         *
         */
        @Deprecated("As of version 1.0.13, use {@link DateTickUnitType} instead.")
        val MONTH = 1

        /**
         * A constant for days.
         *
         */
        @Deprecated("As of version 1.0.13, use {@link DateTickUnitType} instead.")
        val DAY = 2

        /**
         * A constant for hours.
         *
         */
        @Deprecated("As of version 1.0.13, use {@link DateTickUnitType} instead.")
        val HOUR = 3

        /**
         * A constant for minutes.
         *
         */
        @Deprecated("As of version 1.0.13, use {@link DateTickUnitType} instead.")
        val MINUTE = 4

        /**
         * A constant for seconds.
         *
         */
        @Deprecated("As of version 1.0.13, use {@link DateTickUnitType} instead.")
        val SECOND = 5

        /**
         * A constant for milliseconds.
         *
         */
        @Deprecated("As of version 1.0.13, use {@link DateTickUnitType} instead.")
        val MILLISECOND = 6

        /**
         * Converts a unit type to the corresponding deprecated integer constant.
         *
         * @param unitType the unit type (`null` not permitted).
         *
         * @return The int code.
         */
        private fun unitTypeToInt(unitType: DateTickUnitType?): Int {
            require(unitType != null)
            return if (unitType == DateTickUnitType.Companion.YEAR) {
                YEAR
            } else if (unitType == DateTickUnitType.Companion.MONTH) {
                MONTH
            } else if (unitType == DateTickUnitType.Companion.DAY) {
                DAY
            } else if (unitType == DateTickUnitType.Companion.HOUR) {
                HOUR
            } else if (unitType == DateTickUnitType.Companion.MINUTE) {
                MINUTE
            } else if (unitType == DateTickUnitType.Companion.SECOND) {
                SECOND
            } else if (unitType == DateTickUnitType.Companion.MILLISECOND) {
                MILLISECOND
            } else {
                -1
            }
        }

        /**
         * Returns the (approximate) number of milliseconds for the given unit and
         * unit count.
         *
         *
         * This value is an approximation some of the time (e.g. months are
         * assumed to have 31 days) but this shouldn't matter.
         *
         * @param unit  the unit.
         * @param count the unit count.
         *
         * @return The number of milliseconds.
         */
        private fun getMillisecondCount(unit: DateTickUnitType, count: Int): Long {
            return if (unit == DateTickUnitType.Companion.YEAR) {
                365L * 24L * 60L * 60L * 1000L * count
            } else if (unit == DateTickUnitType.Companion.MONTH) {
                31L * 24L * 60L * 60L * 1000L * count
            } else if (unit == DateTickUnitType.Companion.DAY) {
                24L * 60L * 60L * 1000L * count
            } else if (unit == DateTickUnitType.Companion.HOUR) {
                60L * 60L * 1000L * count
            } else if (unit == DateTickUnitType.Companion.MINUTE) {
                60L * 1000L * count
            } else if (unit == DateTickUnitType.Companion.SECOND) {
                1000L * count
            } else if (unit == DateTickUnitType.Companion.MILLISECOND) {
                count.toLong()
            } else {
                -1
            }
        }
    }

    /**
     * Creates a new unit.
     *
     * @param unitType     the unit.
     * @param multiple     the multiple.
     * @param rollUnitType the roll unit.
     * @param rollMultiple the roll multiple.
     * @param formatter    the date formatter (`null` not permitted).
     */
    init {
        require(formatter != null)
        require(multiple > 0)
        require(rollMultiple > 0)
        this.unitType = unitType
        this.multiple = multiple
        this.rollUnitType = rollUnitType
        rollCount = rollMultiple
        this.formatter = formatter

        // populate deprecated fields
        unit = unitTypeToInt(unitType)
    }
}