@file:OptIn(ExperimentalTime::class)

package com.macrofocus.plot.guide

import com.macrofocus.common.date.CPTimeZone
import com.macrofocus.common.locale.CPLocale
import com.macrofocus.plot.datetime.CPCalendar
import kotlin.time.ExperimentalTime
import kotlin.time.Instant


/**
 * Represents a single month.  This class is immutable, which is a requirement
 * for all [RegularTimePeriod] subclasses.
 */
internal class Month : RegularTimePeriod {
    /** The month (1-12).  */
    private val month: Int

    /** The year in which the month falls.  */
    private val year: Int
    /**
     * Returns the first millisecond of the month.  This will be determined
     * relative to the time zone specified in the constructor, or in the
     * calendar instance passed in the most recent call to the
     * [.peg] method.
     *
     * @return The first millisecond of the month.
     *
     * @see .getLastMillisecond
     */
    /** The first millisecond.  */
    override var firstMillisecond: Long = 0
        private set
    /**
     * Returns the last millisecond of the month.  This will be
     * determined relative to the time zone specified in the constructor, or
     * in the calendar instance passed in the most recent call to the
     * [.peg] method.
     *
     * @return The last millisecond of the month.
     *
     * @see .getFirstMillisecond
     */
    /** The last millisecond.  */
    override var lastMillisecond: Long = 0
        private set

    /**
     * Constructs a new month instance.
     *
     * @param month the month (in the range 1 to 12).
     * @param year  the year.
     */
    private constructor(month: Int, year: Int) {
        require(month >= 1 && month <= 12)
        this.month = month
        this.year = year
        peg(CPCalendar())
    }

    /**
     * Creates a new `Month` instance, based on the specified time,
     * zone and locale.
     *
     * @param time   the current time.
     * @param zone   the time zone.
     * @param locale the locale.
     */
    constructor(time: Instant?, zone: CPTimeZone?, locale: CPLocale?) {
        val calendar: CPCalendar = CPCalendar()
        calendar.time = time
        month = calendar.get(CPCalendar.MONTH) + 1
        year = calendar.get(CPCalendar.YEAR)
        peg(calendar)
    }

    /**
     * Recalculates the start date/time and end date/time for this time period
     * relative to the supplied calendar (which incorporates a time zone).
     *
     * @param calendar the calendar (`null` not permitted).
     */
    fun peg(calendar: CPCalendar) {
        firstMillisecond = getFirstMillisecond(calendar)
        lastMillisecond = getLastMillisecond(calendar)
    }

    /**
     * Returns the last millisecond of the month, evaluated using the supplied
     * calendar (which determines the time zone).
     *
     * @param calendar the calendar (`null` not permitted).
     *
     * @return The last millisecond of the month.
     *
     * @throws NullPointerException if `calendar` is
     * `null`.
     */
    private fun getLastMillisecond(calendar: CPCalendar): Long {
        val eom: Int = SerialDate.lastDayOfMonth(month, year)
        println("getLastMillisecond $year $month $eom")
        calendar.setAll(year, month - 1, eom, 23, 59, 59)
        calendar.set(CPCalendar.MILLISECOND, 999)
        // in the following line, we'd rather call calendar.getTimeInMillis()
        // to avoid object creation, but that isn't supported in Java 1.3.1
        return calendar.time!!.toEpochMilliseconds()
    }

    /**
     * Returns the first millisecond of the month, evaluated using the supplied
     * calendar (which determines the time zone).
     *
     * @param calendar the calendar (`null` not permitted).
     *
     * @return The first millisecond of the month.
     *
     * @throws NullPointerException if `calendar` is
     * `null`.
     */
    private fun getFirstMillisecond(calendar: CPCalendar): Long {
        calendar.setAll(year, month - 1, 1, 0, 0, 0)
        calendar.set(CPCalendar.MILLISECOND, 0)
        // in the following line, we'd rather call calendar.getTimeInMillis()
        // to avoid object creation, but that isn't supported in Java 1.3.1
        return calendar.time!!.toEpochMilliseconds()
    }

    /**
     * Returns a string representing the month (e.g. "January 2002").
     *
     *
     * To do: look at internationalisation.
     *
     * @return A string representing the month.
     */
    override fun toString(): String {
        return SerialDate.monthCodeToString(month) + ' ' + year
    }

    /**
     * Returns the month preceding this one.  Note that the returned
     * is "pegged" using the default time-zone, irrespective of
     * the time-zone used to peg of the current month (which is not recorded
     * anywhere).  See the [.peg] method.
     *
     * @return The month preceding this one.
     */
    fun previous(): RegularTimePeriod? {
        val result: Month?
        result = if (month == MonthConstants.Companion.JANUARY) {
            if (year > 1900) Month(MonthConstants.Companion.DECEMBER, year - 1) else null
        } else {
            Month(month - 1, year)
        }
        return result
    }

    override fun hashCode(): Int {
        var result = 17
        result = 37 * result + month
        result = 37 * result + year
        return result
    }

    override fun equals(obj: Any?): Boolean {
        if (obj === this) {
            return true
        }
        if (obj !is Month) {
            return false
        }
        val that = obj
        return if (month != that.month) {
            false
        } else year == that.year
    }

    override operator fun compareTo(o1: Any?): Int {
        var result: Int

        // CASE 1 : Comparing to another Month object
        // --------------------------------------------
        if (o1 is Month) {
            val m = o1
            result = year - m.year
            if (result == 0) {
                result = month - m.month
            }
        } else result = if (o1 is RegularTimePeriod) 0 else 1
        return result
    }
}