@file:OptIn(ExperimentalTime::class)

package com.macrofocus.plot.guide

import kotlin.time.ExperimentalTime
import kotlin.time.Instant

/**
 * An abstract class representing a unit of time.  Convenient methods are
 * provided for calculating the next and previous time periods.  Conversion
 * methods are defined that return the first and last milliseconds of the time
 * period.  The results from these methods are timezone dependent.
 *
 *
 * This class is immutable, and all subclasses should be immutable also.
 */
internal abstract class RegularTimePeriod : TimePeriod, Comparable<Any?>, MonthConstants {
    /**
     * Returns the millisecond closest to the middle of the time period.
     *
     * @return The middle millisecond.
     */
    val middleMillisecond: Long
        get() {
            val m1 = firstMillisecond
            val m2 = lastMillisecond
            return m1 + (m2 - m1) / 2
        }

    /**
     * Returns the last millisecond of the time period.  This will be
     * determined relative to the time zone specified in the constructor, or
     * in the calendar instance passed in the most recent call to the
     * method.
     *
     * @return The last millisecond of the time period.
     *
     * @see .getFirstMillisecond
     */
    abstract val lastMillisecond: Long

    /**
     * Returns the first millisecond of the time period.  This will be
     * determined relative to the time zone specified in the constructor, or
     * in the calendar instance passed in the most recent call to the
     * method.
     *
     * @return The first millisecond of the time period.
     *
     * @see .getLastMillisecond
     */
    abstract val firstMillisecond: Long

    /**
     * Returns a string representation of the time period.
     *
     * @return The string.
     */
    override fun toString(): String {
        return start.toString()
    }

    /**
     * Returns the date/time that marks the start of the time period.  This
     * method returns a new `Date` instance every time it is called.
     *
     * @return The start date/time.
     *
     * @see .getFirstMillisecond
     */
    private val start: Instant
        private get() = Instant.fromEpochMilliseconds(firstMillisecond)
}