/*
 * Copyright (c) 2016 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.slider

import com.macrofocus.common.crossplatform.CPHelper
import com.macrofocus.common.interval.IntervalEvent
import com.macrofocus.common.interval.IntervalListener
import com.macrofocus.common.interval.SimpleBoundedInterval
import com.macrofocus.common.properties.EnumProperties
import com.macrofocus.common.properties.MutableProperties
import com.macrofocus.common.properties.MutableProperty
import com.macrofocus.common.timer.CPTimer
import com.macrofocus.common.timer.CPTimerListener
import com.macrofocus.common.transform.SimpleOneDScreenTransform
import org.mkui.canvas.CPCanvas
import org.mkui.canvas.MouseEvent
import org.mkui.canvas.MouseListener
import org.mkui.canvas.MouseMotionListener
import org.mkui.color.MkColor
import org.mkui.color.colorOf
import org.mkui.geom.Point
import org.mkui.geom.Point2D
import org.mkui.geom.Rectangle
import org.mkui.graphics.AbstractIDrawing
import org.mkui.graphics.IGraphics
import org.mkui.graphics.colortheme.ColorTheme
import kotlin.math.*

/**
 * Created by luc on 25/05/16.
 */
abstract class AbstractRangeSliderView() : CPCanvas(),RangeSliderView {
    protected val properties: MutableProperties<RangeSliderView.PropertyType> = EnumProperties<RangeSliderView.PropertyType>(RangeSliderView.PropertyType.values())
    private val showValues = true
    private var mouseReleased = false

    private enum class Knob {
        Start, End, Thumb
    }

    private var model: RangeSliderModel? = null
    private var interval: SimpleBoundedInterval? = null
    private var fullInterval: SimpleBoundedInterval? = null
    private var drawing: SliderDrawing? = null
    private var arrowColor: MkColor
    private val updateModelTimer: CPTimer
    private val rangeSliderListener: RangeSliderListener = object : RangeSliderListener {
        override fun sliderChanged(e: RangeSliderEvent) {
            interval?.setValue(model!!.currentMinimum, model!!.currentExtent)
        }

        override fun sliderScaleChanged(e: RangeSliderEvent) {}
    }

    override fun setModel(model: RangeSliderModel?) {
        if (this.model !== model) {
            if (this.model != null) {
                this.model!!.removeSliderListener(rangeSliderListener)
            }
            this.model = model
            interval = SimpleBoundedInterval(model!!.currentMinimum, model.currentExtent)
            fullInterval = SimpleBoundedInterval(model.minimum, model.extent)
            val transform = SimpleOneDScreenTransform(fullInterval!!, 0)
            drawing = SliderDrawing(interval!!, fullInterval!!, transform)
            addLayer(drawing!!)
            val l: SliderListener = SliderListener(interval!!, transform)
            addMouseListener(l)
            addMouseMotionListener(l)
            model.addSliderListener(rangeSliderListener)
        }
    }

    override val orientation: MutableProperty<RangeSliderView.Orientation>
        get() = properties.getProperty(RangeSliderView.PropertyType.Orientation) as MutableProperty<RangeSliderView.Orientation>
    override val continuousUpdating: MutableProperty<Boolean>
        get() = properties.getProperty(RangeSliderView.PropertyType.ContinuousUpdating) as MutableProperty<Boolean>
    val colorTheme: MutableProperty<ColorTheme>
        get() = properties.getProperty(RangeSliderView.PropertyType.ColorTheme) as MutableProperty<ColorTheme>
    val height: Int
        get() = getHeight().toInt()

    private inner class SliderDrawing(interval: SimpleBoundedInterval, fullInterval: SimpleBoundedInterval, transform: SimpleOneDScreenTransform) :
        AbstractIDrawing() {
        private val interval: SimpleBoundedInterval
        private val fullInterval: SimpleBoundedInterval
        private val transform: SimpleOneDScreenTransform
        override fun draw(g: IGraphics, point: Point2D?, width: Double, height: Double, clipBounds: Rectangle) {
            val w: Double
            val h: Double
            if (orientation.value === RangeSliderView.Orientation.Horizontal) {
                w = width
                h = height
            } else {
                w = height
                h = width
                val w2 = width.toInt() / 2
                val h2 = height.toInt() / 2
                g.translate(w2, h2)
                g.rotate(-PI / 2)
                g.translate(-h2, -w2)
            }
            transform.screenSize = w.toInt()
            g.setLineWidth(1.5)
            //            g.setColor(GWTFactory.getInstance().createColor(220, 220, 220));
//            g.setColor(fullRangeColor);
            val centerY = (h / 2).toInt()
            val fullX1: Int = transform.worldToScreen(fullInterval.start)
            val fullX2: Int = transform.worldToScreen(fullInterval.end)
            //            System.err.println(fullX1 + ", " + fullX2 + ": " + centerY);
//            g.drawLine(fullX1, centerY, fullX2, centerY);

//            g.setLineWidth(1.5f);
//            g.setColor(currentRangecolor);
            val x1: Int = transform.worldToScreen(interval.start)
            val x2: Int = transform.worldToScreen(interval.end)
            //            g.drawLine(x1, centerY, x2, centerY);
//
//            g.setLineWidth(6);
            g.setColor(arrowColor)
            drawArrow(g, Point(fullX1 - 1, centerY), Point(x1, centerY))
            drawArrow(g, Point(fullX2 + 1, centerY), Point(x2, centerY))
        }

        public override fun notifyIDrawingChanged() {
            super.notifyIDrawingChanged()
        }

        init {
            this.interval = interval
            this.fullInterval = fullInterval
            this.transform = transform
            transform.setScreenMargins(16)
            interval.addIntervalListener(object : IntervalListener {
                override fun intervalChanged(event: IntervalEvent) {
                    notifyIDrawingChanged()
                }
            })
        }
    }

    fun drawArrow(g: IGraphics, from: Point2D, to: Point2D) {
        val drawPath = false
        val triangle = true
        val headlen = 10 // length of head in pixels
        val a: Double = PI / 3.5 // inclination of the arrow
        val angle: Double = atan2(to.y - from.y, to.x - from.x)
        g.beginPath()
        if (drawPath) {
            g.moveTo(from.x, from.y)
            g.lineTo(to.x, to.y)
        }
        //        context.moveTo(to.x, to.y);
//        context.lineTo(to.x-headlen*Math.cos(angle - a),to.y-headlen*Math.sin(angle - a));
//        context.moveTo(to.x, to.y);
//        context.lineTo(to.x-headlen*Math.cos(angle + a),to.y-headlen*Math.sin(angle + a));
        g.moveTo(to.x - headlen * cos(angle - a), to.y - headlen * sin(angle - a))
        g.lineTo(to.x, to.y)
        g.lineTo(to.x - headlen * cos(angle + a), to.y - headlen * sin(angle + a))
        if (!triangle) {
            g.moveTo(to.x, to.y)
        }
        g.closePath()
        if (!triangle) {
            g.stroke()
        } else {
            g.fill()
        }
    }

    private inner class SliderListener(interval: SimpleBoundedInterval, transform: SimpleOneDScreenTransform) : MouseListener,
        MouseMotionListener {
        private val interval: SimpleBoundedInterval
        private val transform: SimpleOneDScreenTransform
        var last: Int? = null
        var lastStart: Double? = null
        var lastEnd: Double? = null
        private var knob: Knob? = null
        override fun mouseClicked(e: MouseEvent) {
            if (e.clickCount > 1) {
                interval.reset()
            }
        }

        override fun mousePressed(e: MouseEvent) {
            val x: Int = if (orientation.value === RangeSliderView.Orientation.Horizontal) e.x else height - e.y
            last = x
            val x1: Int = transform.worldToScreen(interval.start)
            val x2: Int = transform.worldToScreen(interval.end)
            if (abs(x - x1) <= 8) {
                knob = Knob.Start
            } else if (abs(x - x2) <= 8) {
                knob = Knob.End
            } else if (!interval.isFullRange && x > x1 && x < x2) {
                knob = Knob.Thumb
                lastStart = interval.start
                lastEnd = interval.end
            } else {
                knob = null
            }
            if (knob != null) {
                e.stopPropagation()
            }
        }

        override fun mouseReleased(e: MouseEvent) {
            mouseReleased = true
            if (knob != null && last != null) {
                updateInterval(e)
            }
            last = null
            lastStart = null
            lastEnd = null
            knob = null
        }

        override fun mouseEntered(e: MouseEvent) {
//            footerPanel.setVisible(true);
            arrowColor = if (colorTheme.value != null) colorTheme.value.probing else colorOf(255, 200, 0)
            drawing!!.notifyIDrawingChanged()
        }

        override fun mouseExited(e: MouseEvent) {
//            if (knob != null) {
//                mouseReleased = true;
//                updateInterval(e);
//            }

//            footerPanel.setVisible(false);
            arrowColor = colorOf(127, 127, 127)
            drawing!!.notifyIDrawingChanged()

//            last = null;
        }

        override fun mouseDragged(e: MouseEvent) {
            if (knob != null && last != null) {
                updateInterval(e)
                e.stopPropagation()
            }
        }

        private fun updateInterval(event: MouseEvent) {
            val x: Int = if (orientation.value === RangeSliderView.Orientation.Horizontal) event.x else height - event.y
            when (knob) {
                Knob.Start -> {
                    val x1: Int = transform.worldToScreen(fullInterval!!.start)
                    val start: Double =
                        min(interval.end, max(interval.minimum, transform.screenToWorld(x1 + (x - last!!))))
                    if (start >= interval.minimum && start <= interval.end) {
                        interval.start = start
                    }
                }
                Knob.End -> {
                    val x2: Int = transform.worldToScreen(fullInterval!!.end)
                    val end: Double =
                        max(interval.start, min(interval.maximum, transform.screenToWorld(x2 + (x - last!!))))
                    if (end <= interval.maximum && end >= interval.start) {
                        interval.end = end
                    }
                }
                Knob.Thumb -> {
                    val x1: Int = transform.worldToScreen(lastStart!!)
                    val x2: Int = transform.worldToScreen(lastEnd!!)
                    var s: Double = transform.screenToWorld(x1 + (x - last!!))
                    var e: Double = transform.screenToWorld(x2 + (x - last!!))
                    if (s < interval.minimum) {
                        s = interval.minimum
                        e = interval.minimum + interval.extent
                    }
                    if (e > interval.maximum) {
                        e = interval.maximum
                        s = interval.maximum - interval.extent
                    }
                    val start: Double = min(interval.end, max(interval.minimum, s))
                    val end: Double = max(interval.start, min(interval.maximum, e))
                    if (start >= interval.minimum && start <= interval.end) {
                        interval.start = start
                    }
                    if (end <= interval.maximum && end >= interval.start) {
                        interval.end = end
                    }
                }
                else -> {}
            }
            drawing!!.notifyIDrawingChanged()
            if (continuousUpdating.value || mouseReleased) {
                scheduleUpdateModel()
            }
            mouseReleased = false
        }

        override fun mouseMoved(e: MouseEvent) {}

        init {
            this.interval = interval
            this.transform = transform
        }
    }

    private fun scheduleUpdateModel() {
        updateModelTimer.restart()
    }

    private fun updateModel() {
        model!!.setValues(this, interval!!.start, interval!!.end)
    }

    init {
        updateModelTimer = CPHelper.instance.createTimer("RangeSliderUpdater", 40, true, object : CPTimerListener {
            override fun timerTriggered() {
                updateModel()
            }
        })
        properties.createProperty(RangeSliderView.PropertyType.Orientation, RangeSliderView.Orientation.Vertical)
        properties.createProperty(RangeSliderView.PropertyType.ContinuousUpdating, true)
        properties.createProperty(RangeSliderView.PropertyType.ColorTheme, null as ColorTheme?)
        arrowColor = colorOf(127, 127, 127)
    }
}