/*
 * Copyright (c) 2017 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.plot.guide

import org.mkui.font.CPFontFactory
import org.mkui.font.MkFont
import org.mkui.geom.AffineTransform
import org.mkui.geom.Rectangle2D
import org.mkui.geom.Shape
import org.mkui.graphics.IGraphics
import org.mkui.graphics.IHeadless
import kotlin.math.PI

abstract class AbstractAxis<Color, Font> protected constructor(
    fontFactory: CPFontFactory, headless: IHeadless,
    /** The label for the axis.  */
    private val label: String?
) : Axis<Color, Font> {
    /** A flag indicating whether or not the axis is visible.  */
    override val isVisible: Boolean = DEFAULT_AXIS_VISIBLE

    /** The font for displaying the axis label.  */
    private val labelFont: MkFont?

    /** The insets for the axis label.  */
    private val labelInsets: RectangleInsets

    /** The label angle.  */
    private val labelAngle: Double

    /**
     * A flag that indicates whether or not tick labels are visible for the
     * axis.
     */
    override val isTickLabelsVisible: Boolean = DEFAULT_TICK_LABELS_VISIBLE

    /** The font used to display the tick labels (if showing). */
    override var tickLabelFont: MkFont? = null

    /** The blank space around each tick label.  */
    override val tickLabelInsets: RectangleInsets = DEFAULT_TICK_LABEL_INSETS
    protected val headless: IHeadless

    /**
     * Calculates the positions of the ticks for the axis, storing the results
     * in the tick list (ready for drawing).
     *
     * @param g2       the graphics device.
     * @param dataArea the area inside the axes.
     * @param edge     the edge on which the axis is located.
     *
     * @return The list of ticks.
     */
    abstract fun refreshTicks(
        g2: IGraphics,
        dataArea: Rectangle2D, edge: RectangleEdge
    ): List<ValueTick>

    /**
     * Returns a rectangle that encloses the axis label.  This is typically
     * used for layout purposes (it gives the maximum dimensions of the label).
     *
     * @param g2   the graphics device.
     * @param edge the edge of the plot area along which the axis is measuring.
     *
     * @return The enclosing rectangle.
     */
    fun getLabelEnclosure(g2: IGraphics?, edge: RectangleEdge): Rectangle2D {
        val axisLabel = label
        if (axisLabel != null && !axisLabel.isEmpty()) {
//            final FontMetrics fm = g2.getFontMetrics(labelFont.getNativeFont());
//            Rectangle2D bounds = TextUtilities.getTextBounds(axisLabel, g2, fm);
            val b: Rectangle2D = headless.getStringBounds(g2, labelFont, axisLabel)
            var bounds: Rectangle2D = Rectangle2D.Double(b.x, b.y, b.width, b.height)
            val insets: RectangleInsets = labelInsets
            bounds = insets.createOutsetRectangle(bounds)
            var angle = labelAngle
            if (edge == RectangleEdge.Left || edge == RectangleEdge.Right) {
                angle -= PI / 2.0
            }
            val x: Double = bounds.centerX
            val y: Double = bounds.centerY
            val transformer: AffineTransform = AffineTransform.getRotateInstance(angle, x, y)
            val labelBounds: Shape? = transformer.createTransformedShape(bounds)
            return labelBounds!!.bounds2D
        } else {
            return Rectangle2D.Double(0.0,0.0,0.0, 0.0)
        }
    }

    companion object {
        /** The default axis visibility.  */
        private const val DEFAULT_AXIS_VISIBLE = true

        /** The default axis label insets.  */
        private val DEFAULT_AXIS_LABEL_INSETS: RectangleInsets = RectangleInsets(3.0, 3.0, 3.0, 3.0)

        /** The default tick labels visibility.  */
        private const val DEFAULT_TICK_LABELS_VISIBLE = true

        /** The default tick label insets.  */
        private val DEFAULT_TICK_LABEL_INSETS: RectangleInsets = RectangleInsets(2.0, 4.0, 2.0, 4.0)
    }

    /**
     * Constructs an axis, using default values where necessary.
     *
     * @param label the axis label (`null` permitted).
     */
    init {
        labelFont = fontFactory.createFont("SansSerif", 0, 12)
        labelInsets = DEFAULT_AXIS_LABEL_INSETS
        labelAngle = 0.0
        tickLabelFont = fontFactory.createFont("SansSerif", 0, 10)
        this.headless = headless
    }
}