package com.macrofocus.plot.guide

/** The base class used to represent labelled ticks along an axis.  */
abstract class Tick internal constructor(
    text: String, textAnchor: TextAnchor?, rotationAnchor: TextAnchor?,
    angle: Double
) {
    /**
     * Returns the text version of the tick value.
     *
     * @return A string (possibly `null`;
     */
    /** A text version of the tick value.  */
    val text: String

    /** The text anchor for the tick label.  */
    private val textAnchor: TextAnchor?

    /** The rotation anchor for the tick label.  */
    private val rotationAnchor: TextAnchor?

    /** The rotation angle.  */
    private val angle: Double

    /**
     * Tests this tick for equality with an arbitrary object.
     *
     * @param obj the object (`null` permitted).
     *
     * @return A boolean.
     */
    override fun equals(obj: Any?): Boolean {
        if (this === obj) {
            return true
        }
        if (obj is Tick) {
            val t = obj
            if (!com.macrofocus.plot.guide.ObjectUtilities.equal(text, t.text)) {
                return false
            }
            if (!com.macrofocus.plot.guide.ObjectUtilities.equal(textAnchor, t.textAnchor)) {
                return false
            }
            return if (!com.macrofocus.plot.guide.ObjectUtilities.equal(rotationAnchor, t.rotationAnchor)) {
                false
            } else angle == t.angle
        }
        return false
    }

    /**
     * Returns a string representation of the tick.
     *
     * @return A string.
     */
    override fun toString(): String {
        return text
    }

    /**
     * Used to indicate the position of an anchor point for a text string.  This is
     * frequently used to align a string to a fixed point in some coordinate space.
     */
    enum class TextAnchor {
        /** Top/center.  */
        TopCenter,

        /** Middle/left.  */
        CenterLeft,

        /** Middle/center.  */
        Center,

        /** Middle/right.  */
        CenterRight,

        /** Bottom/center.  */
        BottomCenter
    }

    /**
     * Creates a new tick.
     *
     * @param text           the formatted version of the tick value.
     * @param textAnchor     the text anchor (`null` not permitted).
     * @param rotationAnchor the rotation anchor (`null` not
     * permitted).
     * @param angle          the angle.
     */
    init {
        this.text = text
        this.textAnchor = textAnchor
        this.rotationAnchor = rotationAnchor
        this.angle = angle
    }
}