package com.macrofocus.plot.guide

import com.macrofocus.common.math.doubleToLongBits

/**
 * Base class representing a tick unit.  This determines the spacing of the
 * tick marks on an axis.
 *
 *
 * This class (and any subclasses) should be immutable, the reason being that
 * ORDERED collections of tick units are maintained and if one instance can be
 * changed, it may destroy the order of the collection that it belongs to.
 * In addition, if the implementations are immutable, they can belong to
 * multiple collections.
 *
 * @see ValueAxis
 */
abstract class TickUnit : Comparable<Any?> {
    /**
     * Returns the size of the tick unit.
     *
     * @return The size of the tick unit.
     */
    /** The size of the tick unit.  */
    val size: Double
    /**
     * Returns the minor tick count.
     *
     * @return The minor tick count.
     */
    /** The number of minor ticks.  */
    var minorTickCount = 0
        private set

    /**
     * Constructs a new tick unit.
     *
     * @param size the tick unit size.
     */
    constructor(size: Double) {
        this.size = size
    }

    /**
     * Constructs a new tick unit.
     *
     * @param size           the tick unit size.
     * @param minorTickCount the minor tick count.
     */
    constructor(size: Double, minorTickCount: Int) {
        this.size = size
        this.minorTickCount = minorTickCount
    }

    /**
     * Converts the supplied value to a string.
     *
     *
     * Subclasses may implement special formatting by overriding this method.
     *
     * @param value the data value.
     *
     * @return Value as string.
     */
    open fun valueToString(value: Double): String {
        return value.toString()
    }

    /**
     * Compares this tick unit to an arbitrary object.
     *
     * @param object the object to compare against.
     *
     * @return `1` if the size of the other object is less than this,
     * `0` if both have the same size and `-1` this
     * size is less than the others.
     */
    override fun compareTo(`object`: Any?): Int {
        return if (`object` is TickUnit) {
            val other = `object`
            if (size > other.size) {
                1
            } else if (size < other.size) -1 else 0
        } else {
            -1
        }
    }

    /**
     * Returns a hash code for this instance.
     *
     * @return A hash code.
     */
    override fun hashCode(): Int {
        val temp = if (size != +0.0) doubleToLongBits(size) else 0L
        return (temp xor temp ushr 32).toInt()
    }

    /**
     * Tests this unit for equality with another object.
     *
     * @param obj the object.
     *
     * @return `true` or `false`.
     */
    override fun equals(obj: Any?): Boolean {
        if (obj === this) {
            return true
        }
        if (obj !is TickUnit) {
            return false
        }
        val that = obj
        return if (size != that.size) {
            false
        } else minorTickCount == that.minorTickCount
    }
}