package com.macrofocus.plot.guide

import com.macrofocus.common.format.CPFormat
import org.mkui.font.CPFontFactory
import org.mkui.geom.Polygon
import org.mkui.geom.Rectangle2D
import org.mkui.graphics.IGraphics
import org.mkui.graphics.IHeadless
import kotlin.math.abs

/**
 * The base class for axes that display value data, where values are measured
 * using the `double` primitive.  The two key subclasses are
 * [DateAxis] and [NumberAxis].
 */
abstract class ValueAxis protected constructor(
    fontFactory: CPFontFactory,
    headless: IHeadless,
    label: String?,
    standardTickUnits: TickUnitSource
) : AbstractAxis(fontFactory, headless, label) {
    /**
     * Returns a flag that controls the direction of values on the axis.
     *
     *
     * For a regular axis, values increase from left to right (for a horizontal
     * axis) and bottom to top (for a vertical axis).  When the axis is
     * 'inverted', the values increase in the opposite direction.
     *
     * @return The flag.
     */
    /** A flag that affects the orientation of the values on the axis.  */
    val isInverted: Boolean

    /** The standard tick units for the axis.  */
    private val standardTickUnits: TickUnitSource
    /**
     * Returns the number of minor tick marks to display.
     *
     * @return The number of minor tick marks to display.
     */
    /**
     * The number of minor ticks per major tick unit.  This is an override
     * field, if the value is > 0 it is used, otherwise the axis refers to the
     * minorTickCount in the current tickUnit.
     */
    val minorTickCount: Int
    /**
     * Returns `true` if the tick labels should be rotated (to
     * vertical), and `false` otherwise.
     *
     * @return `true` or `false`.
     */
    /** A flag indicating whether or not tick labels are rotated to vertical.  */
    val isVerticalTickLabels: Boolean

    /** The axis range.  */
    private var range: Range?
    /**
     * Returns a flag indicating whether or not the tick unit is automatically
     * selected from a range of standard tick units.
     *
     * @return A flag indicating whether or not the tick unit is automatically
     * selected.
     */
    /**
     * Flag that indicates whether or not the tick unit is selected
     * automatically.
     */
    var isAutoTickUnitSelection: Boolean
        private set

    /**
     * Returns the space required to draw the axis.
     *
     * @param g2       the graphics device.
     * @param plotArea the area within which the plot should be drawn.
     * @param edge     the axis location.
     * @param space    the space already reserved (for other axes).
     */
    fun reserveSpace(
        g2: IGraphics,
        plotArea: Rectangle2D,
        edge: RectangleEdge, space: AxisSpace?
    ) {

        // create a new space object if one wasn't supplied...
        var space: AxisSpace? = space
        if (space == null) {
            space = AxisSpace()
        }

        // if the axis is not visible, no additional space is required...
        if (!isVisible) {
            return
        }

        // calculate the max size of the tick labels (if visible)...
        var tickLabelHeight = 0.0
        var tickLabelWidth = 0.0
        if (isTickLabelsVisible) {
//            g2.setFont(tickLabelFont);
            val ticks: List<ValueTick> = refreshTicks(g2, plotArea, edge)
            if (RectangleEdge.isTopOrBottom(edge)) {
                tickLabelHeight = findMaximumTickLabelHeight(
                    ticks, g2,
                    isVerticalTickLabels
                )
            } else if (RectangleEdge.isLeftOrRight(edge)) {
                tickLabelWidth = findMaximumTickLabelWidth(
                    ticks, g2,
                    isVerticalTickLabels
                )
            }
        }

        // get the axis label size and update the space object...
        val labelEnclosure: Rectangle2D = getLabelEnclosure(g2, edge)
        if (RectangleEdge.isTopOrBottom(edge)) {
            val labelHeight: Double = labelEnclosure.height
            space.add(labelHeight + tickLabelHeight, edge)
        } else if (RectangleEdge.Companion.isLeftOrRight(edge)) {
            val labelWidth: Double = labelEnclosure.width
            space.add(labelWidth + tickLabelWidth, edge)
        }
    }

    /**
     * A utility method for determining the height of the tallest tick label.
     *
     * @param ticks    the ticks.
     * @param g2       the graphics device.
     * @param vertical a flag that indicates whether or not the tick labels
     * are 'vertical'.
     *
     * @return The height of the tallest tick label.
     */
    private fun findMaximumTickLabelHeight(
        ticks: Iterable<Tick>,
        g2: IGraphics,
        vertical: Boolean
    ): Double {
        val insets: com.macrofocus.plot.guide.RectangleInsets = tickLabelInsets
        var maxHeight = 0.0
        if (vertical) {
            for (tick1 in ticks) {
                val labelWidth: Float = headless.getStringWidth(g2, tickLabelFont, tick1.text)
                if ((labelWidth + insets.top
                            + insets.bottom) > maxHeight
                ) {
                    maxHeight = (labelWidth
                            + insets.top + insets.bottom)
                }
            }
        } else {
            maxHeight = (headless.getStringHeight(g2, tickLabelFont, "ABCxyz")
                    + insets.top + insets.bottom)
        }
        return maxHeight
    }

    /**
     * A utility method for determining the width of the widest tick label.
     *
     * @param ticks    the ticks.
     * @param g2       the graphics device.
     * @param vertical a flag that indicates whether or not the tick labels
     * are 'vertical'.
     *
     * @return The width of the tallest tick label.
     */
    private fun findMaximumTickLabelWidth(
        ticks: Iterable<Tick>,
        g2: IGraphics,
        vertical: Boolean
    ): Double {
        val insets: RectangleInsets = tickLabelInsets
        var maxWidth = 0.0
        if (vertical) {
            maxWidth = (headless.getStringHeight(g2, tickLabelFont, "ABCxyz")
                    + insets.top + insets.bottom)
        } else {
            for (tick1 in ticks) {
                val labelWidth: Float = headless.getStringWidth(g2, tickLabelFont, tick1.text)
                if (labelWidth + insets.left + insets.right > maxWidth) {
                    maxWidth = (labelWidth
                            + insets.left + insets.right)
                }
            }
        }
        return maxWidth
    }

    /**
     * Returns the range for the axis.
     *
     * @return The axis range (never `null`).
     *
     * @see .setRange
     */
    fun getRange(): Range? {
        return range
    }

    /**
     * Sets the range attribute and sends an  to all
     * registered listeners.  As a side-effect, the auto-range flag is set to
     * `false`.
     *
     * @param range the range (`null` not permitted).
     *
     * @see .getRange
     */
    open fun setRange(range: Range) {
        // defer argument checking
        setRange(range, true, true)
    }

    /**
     * Sets the range for the axis, if requested, sends an
     * to all registered listeners.  As a side-effect,
     * the auto-range flag is set to `false` (optional).
     *
     * @param range            the range (`null` not permitted).
     * @param turnOffAutoRange a flag that controls whether or not the auto
     * range is turned off.
     * @param notify           a flag that controls whether or not listeners are
     * notified.
     *
     * @see .getRange
     */
    open fun setRange(
        range: Range, turnOffAutoRange: Boolean,
        notify: Boolean
    ) {
        this.range = range
    }

    /**
     * Sets the axis range and sends an  to all
     * registered listeners.  As a side-effect, the auto-range flag is set to
     * `false`.
     *
     * @param lower the lower axis limit.
     * @param upper the upper axis limit.
     *
     * @see .getRange
     * @see .setRange
     */
    open fun setRange(lower: Double, upper: Double) {
        setRange(Range(lower, upper))
    }

    /**
     * Sets a flag indicating whether or not the tick unit is automatically
     * selected from a range of standard tick units.
     *
     * @see .isAutoTickUnitSelection
     */
    fun setAutoTickUnitSelection() {
        if (isAutoTickUnitSelection) {
            isAutoTickUnitSelection = false
        }
    }

    /**
     * Returns the source for obtaining standard tick units for the axis.
     *
     * @return The source (possibly `null`).
     */
    fun getStandardTickUnits(): TickUnitSource {
        return standardTickUnits
    }

    /**
     * Converts a length in data coordinates into the corresponding length in
     * Java2D coordinates.
     *
     * @param length the length.
     * @param area   the plot area.
     * @param edge   the edge along which the axis lies.
     *
     * @return The length in Java2D coordinates.
     */
    fun lengthToJava2D(
        length: Double, area: Rectangle2D,
        edge: RectangleEdge
    ): Double {
        val zero = valueToJava2D(0.0, area, edge)
        val l = valueToJava2D(length, area, edge)
        return abs(l - zero)
    }

    /**
     * Converts a data value to a coordinate in Java2D space, assuming that the
     * axis runs along one edge of the specified dataArea.
     *
     *
     * Note that it is possible for the coordinate to fall outside the area.
     *
     * @param value the data value.
     * @param area  the area for plotting the data.
     * @param edge  the edge along which the axis lies.
     *
     * @return The Java2D coordinate.
     */
    protected abstract fun valueToJava2D(
        value: Double, area: Rectangle2D,
        edge: RectangleEdge
    ): Double

    abstract fun setFormatOverride(format: CPFormat<Any?>?)

    companion object {
        /** The maximum tick count.  */
        const val MAXIMUM_TICK_COUNT = 500

        /** The default axis range.  */
        private val DEFAULT_RANGE: Range = Range(0.0, 1.0)

        /** The default inverted flag setting.  */
        private const val DEFAULT_INVERTED = false

        /** The default auto-tick-unit-selection value.  */
        private const val DEFAULT_AUTO_TICK_UNIT_SELECTION = true
    }

    /**
     * Constructs a value axis.
     *
     * @param label             the axis label (`null` permitted).
     * @param standardTickUnits the source for standard tick units
     * (`null` permitted).
     */
    init {
        range = DEFAULT_RANGE
        isInverted = DEFAULT_INVERTED
        isAutoTickUnitSelection = DEFAULT_AUTO_TICK_UNIT_SELECTION
        this.standardTickUnits = standardTickUnits
        val p1: Polygon = Polygon()
        p1.addPoint(0, 0)
        p1.addPoint(-2, 2)
        p1.addPoint(2, 2)
        val p2: Polygon = Polygon()
        p2.addPoint(0, 0)
        p2.addPoint(-2, -2)
        p2.addPoint(2, -2)
        val p3: Polygon = Polygon()
        p3.addPoint(0, 0)
        p3.addPoint(-2, -2)
        p3.addPoint(-2, 2)
        val p4: Polygon = Polygon()
        p4.addPoint(0, 0)
        p4.addPoint(2, -2)
        p4.addPoint(2, 2)
        isVerticalTickLabels = false
        minorTickCount = 0
    }
}