package com.macrofocus.plot.guide

import com.macrofocus.common.date.CPTimeZone
import com.macrofocus.common.locale.CPLocale
import com.macrofocus.plot.datetime.CPCalendar
import kotlinx.datetime.Instant

/**
 * Represents a year in the range -9999 to 9999.  This class is immutable,
 * which is a requirement for all [RegularTimePeriod] subclasses.
 */
internal class Year(time: Instant?, zone: CPTimeZone?, locale: CPLocale?) :
    com.macrofocus.plot.guide.RegularTimePeriod() {
    /** The year.  */
    private val year: Int
    /**
     * Returns the first millisecond of the year.  This will be determined
     * relative to the time zone specified in the constructor, or in the
     * calendar instance passed in the most recent call to the
     * [.peg] method.
     *
     * @return The first millisecond of the year.
     *
     * @see .getLastMillisecond
     */
    /** The first millisecond.  */
    override var firstMillisecond: Long = 0
        private set
    /**
     * Returns the last millisecond of the year.  This will be
     * determined relative to the time zone specified in the constructor, or
     * in the calendar instance passed in the most recent call to the
     * [.peg] method.
     *
     * @return The last millisecond of the year.
     *
     * @see .getFirstMillisecond
     */
    /** The last millisecond.  */
    override var lastMillisecond: Long = 0
        private set

    /**
     * Recalculates the start date/time and end date/time for this time period
     * relative to the supplied calendar (which incorporates a time zone).
     *
     * @param calendar the calendar (`null` not permitted).
     */
    private fun peg(calendar: CPCalendar) {
        firstMillisecond = getFirstMillisecond(calendar)
        lastMillisecond = getLastMillisecond(calendar)
    }

    /**
     * Returns the last millisecond of the year, evaluated using the supplied
     * calendar (which determines the time zone).
     *
     * @param calendar the calendar (`null` not permitted).
     *
     * @return The last millisecond of the year.
     *
     * @throws NullPointerException if `calendar` is
     * `null`.
     */
    private fun getLastMillisecond(calendar: CPCalendar): Long {
        calendar.setAll(year.toInt(), CPCalendar.DECEMBER, 31, 23, 59, 59)
        calendar.set(CPCalendar.MILLISECOND, 999)
        // in the following line, we'd rather call calendar.getTimeInMillis()
        // to avoid object creation, but that isn't supported in Java 1.3.1
        return calendar.time!!.toEpochMilliseconds()
    }

    /**
     * Returns the first millisecond of the year, evaluated using the supplied
     * calendar (which determines the time zone).
     *
     * @param calendar the calendar (`null` not permitted).
     *
     * @return The first millisecond of the year.
     *
     * @throws NullPointerException if `calendar` is
     * `null`.
     */
    private fun getFirstMillisecond(calendar: CPCalendar): Long {
        calendar.setAll(year, CPCalendar.JANUARY, 1, 0, 0, 0)
        calendar.set(CPCalendar.MILLISECOND, 0)
        // in the following line, we'd rather call calendar.getTimeInMillis()
        // to avoid object creation, but that isn't supported in Java 1.3.1
        return calendar.time!!.toEpochMilliseconds()
    }

    /**
     * Returns a string representing the year..
     *
     * @return A string representing the year.
     */
    override fun toString(): String {
        return year.toInt().toString()
    }

    /**
     * Returns a hash code for this object instance.  The approach described by
     * Joshua Bloch in "Effective Java" has been used here:
     *
     *
     * `http://developer.java.sun.com/developer/Books/effectivejava
     * /Chapter3.pdf`
     *
     * @return A hash code.
     */
    override fun hashCode(): Int {
        var result = 17
        val c = year.toInt()
        result = 37 * result + c
        return result
    }

    /**
     * Tests the equality of this `Year` object to an arbitrary
     * object.  Returns `true` if the target is a `Year`
     * instance representing the same year as this object.  In all other cases,
     * returns `false`.
     *
     * @param obj the object (`null` permitted).
     *
     * @return `true` if the year of this and the object are the
     * same.
     */
    override fun equals(obj: Any?): Boolean {
        if (obj === this) {
            return true
        }
        if (obj !is Year) {
            return false
        }
        return year == obj.year
    }

    /**
     * Returns an integer indicating the order of this `Year` object
     * relative to the specified object:
     *
     *
     * negative == before, zero == same, positive == after.
     *
     * @param o1 the object to compare.
     *
     * @return negative == before, zero == same, positive == after.
     */
    override operator fun compareTo(o1: Any?): Int {
        val result: Int

        // CASE 1 : Comparing to another Year object
        // -----------------------------------------
        result = if (o1 is Year) {
            year - o1.getYear()
        } else if (o1 is com.macrofocus.plot.guide.RegularTimePeriod) 0 else 1
        return result
    }

    /**
     * Returns the year.
     *
     * @return The year.
     */
    private fun getYear(): Int {
        return year.toInt()
    }

    /**
     * Creates a new `Year` instance, for the specified time zone
     * and locale.
     *
     * @param time   the current time (`null` not permitted).
     * @param zone   the time zone.
     * @param locale the locale.
     */
    init {
        val calendar: CPCalendar = CPCalendar()
        calendar.time = time
        year = calendar.get(CPCalendar.YEAR)
        peg(calendar)
    }
}