package com.macrofocus.plot.guide

import com.macrofocus.common.crossplatform.CPHelper
import com.macrofocus.common.format.CPFormat
import com.macrofocus.common.format.FormatFactory
import org.mkui.component.CPFactory
import org.mkui.font.CPFontFactory
import org.mkui.geom.Rectangle2D
import org.mkui.graphics.CPHeadless
import org.mkui.graphics.IGraphics
import org.mkui.graphics.IHeadless
import kotlin.math.PI
import kotlin.math.ceil
import kotlin.math.floor
import kotlin.math.max

/**
 * An axis for displaying numerical data.
 *
 *
 * If the axis is set up to automatically determine its range to fit the data,
 * you can ensure that the range includes zero (statisticians usually prefer
 * this) by setting the `autoRangeIncludesZero` flag to
 * `true`.
 */
class NumberAxis(fontFactory: CPFontFactory = CPFontFactory.instance, headless: IHeadless = CPHeadless, formatFactory: FormatFactory = FormatFactory.instance) : ValueAxis(fontFactory, headless, null, createStandardTickUnits(formatFactory)) {
    /** The tick unit for the axis.  */
    private var tickUnit: NumberTickUnit?

    /** The override number format.  */
    private var numberFormatOverride: CPFormat<Any?>?

    /**
     * Sets the tick unit for the axis and, if requested, sends an
     * to all registered listeners.  In addition, an
     * option is provided to turn off the "auto-select" feature for tick units
     * (you can restore it using the
     * [ValueAxis.setAutoTickUnitSelection] method).
     *
     * @param unit the new tick unit (`null` not permitted).
     */
    private fun setTickUnit(unit: NumberTickUnit?) {
        tickUnit = unit
    }

    /**
     * Sets the number format override.  If this is non-null, then it will be
     * used to format the numbers on the axis.
     *
     * @param formatter the number formatter (`null` permitted).
     */
    override fun setFormatOverride(formatter: CPFormat<Any?>?) {
        numberFormatOverride = formatter
    }

    /**
     * Converts a data value to a coordinate in Java2D space, assuming that the
     * axis runs along one edge of the specified dataArea.
     *
     *
     * Note that it is possible for the coordinate to fall outside the plotArea.
     *
     * @param value the data value.
     * @param area  the area for plotting the data.
     * @param edge  the axis location.
     *
     * @return The Java2D coordinate.
     */
    override fun valueToJava2D(value: Double, area: Rectangle2D, edge: RectangleEdge): Double {
        val range: Range = getRange()!!
        val axisMin: Double = range.lowerBound
        val axisMax: Double = range.upperBound
        var min = 0.0
        var max = 0.0
        if (RectangleEdge.isTopOrBottom(edge)) {
            min = area.x
            max = area.maxX
        } else if (RectangleEdge.isLeftOrRight(edge)) {
            max = area.minY
            min = area.maxY
        }
        return if (isInverted)
            max -(value - axisMin) / (axisMax - axisMin) * (max - min)
        else
            min +(value - axisMin) / (axisMax - axisMin) * (max - min)
    }

    /**
     * Calculates the value of the lowest visible tick on the axis.
     *
     * @return The value of the lowest visible tick on the axis.
     */
    private fun calculateLowestVisibleTickValue(): Double {
        val unit: Double = tickUnit!!.size
        val index: Double = ceil(getRange()!!.lowerBound / unit)
        return index * unit
    }

    /**
     * Calculates the number of visible ticks.
     *
     * @return The number of visible ticks on the axis.
     */
    private fun calculateVisibleTickCount(): Int {
        val unit: Double = tickUnit!!.size
        val range: com.macrofocus.plot.guide.Range = getRange()!!
        return (floor(range.upperBound / unit) - ceil(range.lowerBound / unit) + 1).toInt()
    }

    /**
     * Estimates the maximum tick label height.
     *
     * @param g2 the graphics device.
     *
     * @return The maximum height.
     */
    private fun estimateMaximumTickLabelHeight(g2: IGraphics): Double {
        val tickLabelInsets: RectangleInsets = tickLabelInsets
        var result: Double = tickLabelInsets.top + tickLabelInsets.bottom
        result += headless.getStringHeight(g2, tickLabelFont, "123")
        return result
    }

    /**
     * Estimates the maximum width of the tick labels, assuming the specified
     * tick unit is used.
     *
     *
     * Rather than computing the string bounds of every tick on the axis, we
     * just look at two values: the lower bound and the upper bound for the
     * axis.  These two values will usually be representative.
     *
     * @param g2   the graphics device.
     * @param unit the tick unit to use for calculation.
     *
     * @return The estimated maximum width of the tick labels.
     */
    private fun estimateMaximumTickLabelWidth(
        g2: IGraphics,
        unit: TickUnit?
    ): Double {
        val tickLabelInsets: RectangleInsets = tickLabelInsets
        var result: Double = tickLabelInsets.left + tickLabelInsets.right
        if (isVerticalTickLabels) {
            // all tick labels have the same width (equal to the height of the
            // font)...
            result += headless.getStringHeight(g2, tickLabelFont, "0")
        } else {
            // look at lower and upper bounds...
            val range: Range = getRange()!!
            val lower: Double = range.lowerBound
            val upper: Double = range.upperBound
            val lowerStr: String
            val upperStr: String
            val formatter: CPFormat<Any?>? = numberFormatOverride
            if (formatter != null) {
                lowerStr = formatter.format(lower)!!
                upperStr = formatter.format(upper)!!
            } else {
                lowerStr = unit!!.valueToString(lower)
                upperStr = unit!!.valueToString(upper)
            }
            val w1: Float = headless.getStringWidth(g2, tickLabelFont, lowerStr)
            val w2: Float = headless.getStringWidth(g2, tickLabelFont, upperStr)
            result += max(w1, w2)
        }
        return result
    }

    /**
     * Selects an appropriate tick value for the axis.  The strategy is to
     * display as many ticks as possible (selected from an array of 'standard'
     * tick units) without the labels overlapping.
     *
     * @param g2       the graphics device.
     * @param dataArea the area defined by the axes.
     * @param edge     the axis location.
     */
    private fun selectAutoTickUnit(
        g2: IGraphics,
        dataArea: Rectangle2D,
        edge: RectangleEdge
    ) {
        if (RectangleEdge.isTopOrBottom(edge)) {
            selectHorizontalAutoTickUnit(g2, dataArea, edge)
        } else if (RectangleEdge.isLeftOrRight(edge)) {
            selectVerticalAutoTickUnit(g2, dataArea, edge)
        }
    }

    /**
     * Selects an appropriate tick value for the axis.  The strategy is to
     * display as many ticks as possible (selected from an array of 'standard'
     * tick units) without the labels overlapping.
     *
     * @param g2       the graphics device.
     * @param dataArea the area defined by the axes.
     * @param edge     the axis location.
     */
    private fun selectHorizontalAutoTickUnit(
        g2: IGraphics,
        dataArea: Rectangle2D,
        edge: RectangleEdge
    ) {
        var tickLabelWidth = estimateMaximumTickLabelWidth(
            g2,
            tickUnit
        )

        // start with the current tick unit...
        val tickUnits: TickUnitSource = getStandardTickUnits()
        val unit1: TickUnit = tickUnits.getCeilingTickUnit(tickUnit as TickUnit)
        val unit1Width: Double = lengthToJava2D(unit1.size, dataArea, edge)

        // then extrapolate...
        val guess: Double = tickLabelWidth / unit1Width * unit1.size
        var unit2: NumberTickUnit = tickUnits.getCeilingTickUnit(guess) as NumberTickUnit
        val unit2Width: Double = lengthToJava2D(unit2.size, dataArea, edge)
        tickLabelWidth = estimateMaximumTickLabelWidth(g2, unit2)
        if (tickLabelWidth > unit2Width) {
            unit2 = tickUnits.getLargerTickUnit(unit2) as com.macrofocus.plot.guide.NumberTickUnit
        }
        setTickUnit(unit2)
    }

    /**
     * Selects an appropriate tick value for the axis.  The strategy is to
     * display as many ticks as possible (selected from an array of 'standard'
     * tick units) without the labels overlapping.
     *
     * @param g2       the graphics device.
     * @param dataArea the area in which the plot should be drawn.
     * @param edge     the axis location.
     */
    private fun selectVerticalAutoTickUnit(
        g2: IGraphics,
        dataArea: Rectangle2D,
        edge: com.macrofocus.plot.guide.RectangleEdge
    ) {
        var tickLabelHeight = estimateMaximumTickLabelHeight(g2)

        // start with the current tick unit...
        val tickUnits: com.macrofocus.plot.guide.TickUnitSource = getStandardTickUnits()
        val unit1: com.macrofocus.plot.guide.TickUnit = tickUnits.getCeilingTickUnit(tickUnit as NumberTickUnit)
        val unitHeight: Double = lengthToJava2D(unit1.size, dataArea, edge)

        // then extrapolate...
        val guess: Double = tickLabelHeight / unitHeight * unit1.size
        var unit2: com.macrofocus.plot.guide.NumberTickUnit = tickUnits.getCeilingTickUnit(guess) as com.macrofocus.plot.guide.NumberTickUnit
        val unit2Height: Double = lengthToJava2D(unit2.size, dataArea, edge)
        tickLabelHeight = estimateMaximumTickLabelHeight(g2)
        if (tickLabelHeight > unit2Height) {
            unit2 = tickUnits.getLargerTickUnit(unit2) as com.macrofocus.plot.guide.NumberTickUnit
        }
        setTickUnit(unit2)
    }

    /**
     * Calculates the positions of the tick labels for the axis, storing the
     * results in the tick label list (ready for drawing).
     *
     * @param g2       the graphics device.
     * @param dataArea the area in which the plot should be drawn.
     * @param edge     the location of the axis.
     *
     * @return A list of ticks.
     */
    override fun refreshTicks(g2: IGraphics, dataArea: Rectangle2D, edge: RectangleEdge): List<ValueTick> {
        var result: List<ValueTick> = ArrayList<ValueTick>()
        if (RectangleEdge.isTopOrBottom(edge)) {
            result = refreshTicksHorizontal(g2, dataArea, edge)
        } else if (RectangleEdge.isLeftOrRight(edge)) {
            result = refreshTicksVertical(g2, dataArea, edge)
        }
        return result
    }

    /**
     * Calculates the positions of the tick labels for the axis, storing the
     * results in the tick label list (ready for drawing).
     *
     * @param g2       the graphics device.
     * @param dataArea the area in which the data should be drawn.
     * @param edge     the location of the axis.
     *
     * @return A list of ticks.
     */
    private fun refreshTicksHorizontal(
        g2: IGraphics,
        dataArea: Rectangle2D, edge: RectangleEdge
    ): List<ValueTick> {
        val result: MutableList<ValueTick> = ArrayList<ValueTick>()

//        g2.setFont(getTickLabelFont());
        if (isAutoTickUnitSelection) {
            selectAutoTickUnit(g2, dataArea, edge)
        }
        val tu: TickUnit = tickUnit!!
        val size: Double = tu.size
        val count = calculateVisibleTickCount()
        val lowestTickValue = calculateLowestVisibleTickValue()
        if (count <= com.macrofocus.plot.guide.ValueAxis.Companion.MAXIMUM_TICK_COUNT) {
            var minorTickSpaces: Int = minorTickCount
            if (minorTickSpaces <= 0) {
                minorTickSpaces = tu.minorTickCount
            }
            for (minorTick in 1 until minorTickSpaces) {
                val minorTickValue = (lowestTickValue
                        - size * minorTick / minorTickSpaces)
                if (getRange()!!.contains(minorTickValue)) {
                    result.add(
                        com.macrofocus.plot.guide.NumberTick(
                            minorTickValue
                        )
                    )
                }
            }
            for (i in 0 until count) {
                val currentTickValue = lowestTickValue + i * size
                val tickLabel: String?
                val formatter: CPFormat<Any?>? = numberFormatOverride
                tickLabel = if (formatter != null) formatter.format(currentTickValue) else tickUnit!!.valueToString(currentTickValue)
                val anchor: com.macrofocus.plot.guide.Tick.TextAnchor
                val rotationAnchor: com.macrofocus.plot.guide.Tick.TextAnchor
                var angle = 0.0
                if (isVerticalTickLabels) {
                    anchor = Tick.TextAnchor.CenterRight
                    rotationAnchor = Tick.TextAnchor.CenterRight
                    angle = if (edge == RectangleEdge.Top) PI / 2.0 else -PI / 2.0
                } else {
                    if (edge == RectangleEdge.Top) {
                        anchor = Tick.TextAnchor.BottomCenter
                        rotationAnchor = Tick.TextAnchor.BottomCenter
                    } else {
                        anchor = Tick.TextAnchor.TopCenter
                        rotationAnchor = Tick.TextAnchor.TopCenter
                    }
                }
                val tick: com.macrofocus.plot.guide.ValueTick = com.macrofocus.plot.guide.NumberTick(
                    currentTickValue,
                    tickLabel!!, anchor, rotationAnchor, angle
                )
                result.add(tick)
                val nextTickValue = lowestTickValue + (i + 1) * size
                for (minorTick in 1 until minorTickSpaces) {
                    val minorTickValue = (currentTickValue
                            + (nextTickValue - currentTickValue)
                            * minorTick / minorTickSpaces)
                    if (getRange()!!.contains(minorTickValue)) {
                        result.add(
                            com.macrofocus.plot.guide.NumberTick(
                                minorTickValue
                            )
                        )
                    }
                }
            }
        }
        return result
    }

    /**
     * Calculates the positions of the tick labels for the axis, storing the
     * results in the tick label list (ready for drawing).
     *
     * @param g2       the graphics device.
     * @param dataArea the area in which the plot should be drawn.
     * @param edge     the location of the axis.
     *
     * @return A list of ticks.
     */
    private fun refreshTicksVertical(
        g2: IGraphics,
        dataArea: Rectangle2D, edge: RectangleEdge
    ): List<ValueTick> {
        val result: MutableList<ValueTick> = ArrayList<ValueTick>()

//        g2.setFont(getTickLabelFont());
        if (isAutoTickUnitSelection) {
            selectAutoTickUnit(g2, dataArea, edge)
        }
        val tu: TickUnit = tickUnit!!
        val size: Double = tu.size
        val count = calculateVisibleTickCount()
        val lowestTickValue = calculateLowestVisibleTickValue()
        if (count <= ValueAxis.Companion.MAXIMUM_TICK_COUNT) {
            var minorTickSpaces: Int = minorTickCount
            if (minorTickSpaces <= 0) {
                minorTickSpaces = tu.minorTickCount
            }
            for (minorTick in 1 until minorTickSpaces) {
                val minorTickValue = (lowestTickValue
                        - size * minorTick / minorTickSpaces)
                if (getRange()!!.contains(minorTickValue)) {
                    result.add(
                        com.macrofocus.plot.guide.NumberTick(
                            minorTickValue
                        )
                    )
                }
            }
            for (i in 0 until count) {
                val currentTickValue = lowestTickValue + i * size
                val tickLabel: String?
                val formatter: CPFormat<Any?>? = numberFormatOverride
                tickLabel = if (formatter != null) formatter!!.format(currentTickValue) else tickUnit!!.valueToString(currentTickValue)
                val anchor: com.macrofocus.plot.guide.Tick.TextAnchor
                val rotationAnchor: com.macrofocus.plot.guide.Tick.TextAnchor
                var angle = 0.0
                if (isVerticalTickLabels) {
                    if (edge == RectangleEdge.Left) {
                        anchor = Tick.TextAnchor.BottomCenter
                        rotationAnchor = Tick.TextAnchor.BottomCenter
                        angle = -PI / 2.0
                    } else {
                        anchor = Tick.TextAnchor.BottomCenter
                        rotationAnchor = Tick.TextAnchor.BottomCenter
                        angle = PI / 2.0
                    }
                } else {
                    if (edge == RectangleEdge.Left) {
                        anchor = Tick.TextAnchor.CenterRight
                        rotationAnchor = Tick.TextAnchor.CenterRight
                    } else {
                        anchor = Tick.TextAnchor.CenterLeft
                        rotationAnchor = Tick.TextAnchor.CenterLeft
                    }
                }
                val tick: ValueTick = NumberTick(currentTickValue, tickLabel!!, anchor, rotationAnchor, angle)
                result.add(tick)
                val nextTickValue = lowestTickValue + (i + 1) * size
                for (minorTick in 1 until minorTickSpaces) {
                    val minorTickValue = (currentTickValue
                            + (nextTickValue - currentTickValue)
                            * minorTick / minorTickSpaces)
                    if (getRange()!!.contains(minorTickValue)) {
                        result.add(
                            com.macrofocus.plot.guide.NumberTick(
                                minorTickValue
                            )
                        )
                    }
                }
            }
        }
        return result
    }

    companion object {
        /** The default tick unit.  */
        private var DEFAULT_TICK_UNIT: com.macrofocus.plot.guide.NumberTickUnit? = null

        /**
         * Creates the standard tick units.
         *
         *
         * If you don't like these defaults, create your own instance of TickUnits
         * and then pass it to the setStandardTickUnits() method in the
         * NumberAxis class.
         *
         * @return The standard tick units.
         */
        private fun createStandardTickUnits(formatFactory: FormatFactory): com.macrofocus.plot.guide.TickUnitSource {
            val units: com.macrofocus.plot.guide.TickUnits = com.macrofocus.plot.guide.TickUnits(formatFactory)
            val df000: CPFormat<Any?> = formatFactory.createNumberFormat("0.0000000000")
            val df00: CPFormat<Any?> = formatFactory.createNumberFormat("0.000000000")
            val df0: CPFormat<Any?> = formatFactory.createNumberFormat("0.00000000")
            val df1: CPFormat<Any?> = formatFactory.createNumberFormat("0.0000000")
            val df2: CPFormat<Any?> = formatFactory.createNumberFormat("0.000000")
            val df3: CPFormat<Any?> = formatFactory.createNumberFormat("0.00000")
            val df4: CPFormat<Any?> = formatFactory.createNumberFormat("0.0000")
            val df5: CPFormat<Any?> = formatFactory.createNumberFormat("0.000")
            val df6: CPFormat<Any?> = formatFactory.createNumberFormat("0.00")
            val df7: CPFormat<Any?> = formatFactory.createNumberFormat("0.0")
            val df8: CPFormat<Any?> = formatFactory.createNumberFormat("#,##0")
            val df9: CPFormat<Any?> = formatFactory.createNumberFormat("#,###,##0")
            val df10: CPFormat<Any?> = formatFactory.createNumberFormat("#,###,###,##0")

            // we can add the units in any order, the TickUnits collection will
            // sort them...
            units.add(NumberTickUnit(0.000000001, df00, 2))
            units.add(NumberTickUnit(0.00000001, df0, 2))
            units.add(NumberTickUnit(0.0000001, df1, 2))
            units.add(NumberTickUnit(0.000001, df2, 2))
            units.add(NumberTickUnit(0.00001, df3, 2))
            units.add(NumberTickUnit(0.0001, df4, 2))
            units.add(NumberTickUnit(0.001, df5, 2))
            units.add(NumberTickUnit(0.01, df6, 2))
            units.add(NumberTickUnit(0.1, df7, 2))
            units.add(NumberTickUnit(1.0, df8, 2))
            units.add(NumberTickUnit(10.0, df8, 2))
            units.add(NumberTickUnit(100.0, df8, 2))
            units.add(NumberTickUnit(1000.0, df8, 2))
            units.add(NumberTickUnit(10000.0, df8, 2))
            units.add(NumberTickUnit(100000.0, df8, 2))
            units.add(NumberTickUnit(1000000.0, df9, 2))
            units.add(NumberTickUnit(10000000.0, df9, 2))
            units.add(NumberTickUnit(100000000.0, df9, 2))
            units.add(NumberTickUnit(1000000000.0, df10, 2))
            units.add(NumberTickUnit(10000000000.0, df10, 2))
            units.add(NumberTickUnit(100000000000.0, df10, 2))
            units.add(NumberTickUnit(0.0000000025, df000, 5))
            units.add(NumberTickUnit(0.000000025, df00, 5))
            units.add(NumberTickUnit(0.00000025, df0, 5))
            units.add(NumberTickUnit(0.0000025, df1, 5))
            units.add(NumberTickUnit(0.000025, df2, 5))
            units.add(NumberTickUnit(0.00025, df3, 5))
            units.add(NumberTickUnit(0.0025, df4, 5))
            units.add(NumberTickUnit(0.025, df5, 5))
            units.add(NumberTickUnit(0.25, df6, 5))
            units.add(NumberTickUnit(2.5, df7, 5))
            units.add(NumberTickUnit(25.0, df8, 5))
            units.add(NumberTickUnit(250.0, df8, 5))
            units.add(NumberTickUnit(2500.0, df8, 5))
            units.add(NumberTickUnit(25000.0, df8, 5))
            units.add(NumberTickUnit(250000.0, df8, 5))
            units.add(NumberTickUnit(2500000.0, df9, 5))
            units.add(NumberTickUnit(25000000.0, df9, 5))
            units.add(NumberTickUnit(250000000.0, df9, 5))
            units.add(NumberTickUnit(2500000000.0, df10, 5))
            units.add(NumberTickUnit(25000000000.0, df10, 5))
            units.add(NumberTickUnit(250000000000.0, df10, 5))
            units.add(NumberTickUnit(0.000000005, df00, 5))
            units.add(NumberTickUnit(0.00000005, df0, 5))
            units.add(NumberTickUnit(0.0000005, df1, 5))
            units.add(NumberTickUnit(0.000005, df2, 5))
            units.add(NumberTickUnit(0.00005, df3, 5))
            units.add(NumberTickUnit(0.0005, df4, 5))
            units.add(NumberTickUnit(0.005, df5, 5))
            units.add(NumberTickUnit(0.05, df6, 5))
            units.add(NumberTickUnit(0.5, df7, 5))
            units.add(NumberTickUnit(5.0, df8, 5))
            units.add(NumberTickUnit(50.0, df8, 5))
            units.add(NumberTickUnit(500.0, df8, 5))
            units.add(NumberTickUnit(5000.0, df8, 5))
            units.add(NumberTickUnit(50000.0, df8, 5))
            units.add(NumberTickUnit(500000.0, df8, 5))
            units.add(NumberTickUnit(5000000.0, df9, 5))
            units.add(NumberTickUnit(50000000.0, df9, 5))
            units.add(NumberTickUnit(500000000.0, df9, 5))
            units.add(NumberTickUnit(5000000000.0, df10, 5))
            units.add(NumberTickUnit(50000000000.0, df10, 5))
            units.add(NumberTickUnit(500000000000.0, df10, 5))
            return units
        }
    }

    /** Constructs a number axis, using default values where necessary.  */
    init {
        if (DEFAULT_TICK_UNIT == null) {
            DEFAULT_TICK_UNIT = com.macrofocus.plot.guide.NumberTickUnit(1.0, formatFactory.createNumberFormat("0"))
        }
        tickUnit = DEFAULT_TICK_UNIT
        numberFormatOverride = null
    }
}