/*
 * Copyright (c) 2017 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.plot.guide

import com.macrofocus.common.format.CPFormat
import com.macrofocus.common.geom.Dimension
import com.macrofocus.common.transform.OneDScreenTransform
import com.macrofocus.common.transform.ScreenTransformEvent
import com.macrofocus.common.transform.ScreenTransformListener
import org.mkui.color.MkColor
import org.mkui.color.MkColorFactory
import org.mkui.color.colorOf
import org.mkui.font.CPFontFactory
import org.mkui.font.CPFontMetrics
import org.mkui.geom.Rectangle
import org.mkui.geom.Rectangle2D
import org.mkui.graphics.IGraphics
import java.awt.Font
import java.awt.event.ComponentEvent
import java.awt.event.ComponentListener

/**
 * Created by luc on 15/04/16.
 */
abstract class AbstractSwingAxisGuide @JvmOverloads constructor(private val type: Guide.Type = Guide.Type.Bottom) : AbstractSwingGuide(),
    AxisGuide {
    private var labelColor: MkColor = MkColorFactory.instance.black
    private var gridLinesColor: MkColor = colorOf(199, 199, 196)
    private var paintTicks = true
    private val coordinateAxisListener: ScreenTransformListener = object : ScreenTransformListener {
        override fun transformChanged(event: ScreenTransformEvent) {
            refresh()
        }
    }

    private fun refresh() {
//        repaint();
//            invalidate();
//        revalidate();
//        if (panel != null) {
//            panel.valideAxis();
//        }
        revalidate()
        repaint()
    }

    private var coordinateAxis: OneDScreenTransform? = null
    private var axis: ValueAxis? = null
    private var format: CPFormat<Any?>? = null

    constructor(type: Guide.Type, coordinateAxis: OneDScreenTransform?) : this(type) {
        setCoordinateAxis(coordinateAxis)
    }

    constructor(coordinateAxis: OneDScreenTransform?) : this(Guide.Type.Bottom) {
        setCoordinateAxis(coordinateAxis)
    }

    override fun setCoordinateAxis(coordinateAxis: OneDScreenTransform?) {
        if (this.coordinateAxis != null) {
            this.coordinateAxis!!.removeScreenTransformListener(coordinateAxisListener)
        }
        this.coordinateAxis = coordinateAxis
        if (this.coordinateAxis != null) {
            this.coordinateAxis!!.addScreenTransformListener(coordinateAxisListener)
        }
        refresh()
    }

    override fun computeSpace(g2: IGraphics, dataArea: Rectangle): Double {
        val space = AxisSpace()
        return when (type) {
            Guide.Type.Top -> {
                getAxis().reserveSpace(g2, dataArea, RectangleEdge.Top, space)
                space.bottom
            }
            Guide.Type.Bottom -> {
                getAxis().reserveSpace(g2, dataArea, RectangleEdge.Bottom, space)
                space.bottom + 4
            }
            Guide.Type.Left -> {
                getAxis().reserveSpace(g2, dataArea, RectangleEdge.Left, space)
                space.left
            }
            Guide.Type.Right -> {
                getAxis().reserveSpace(g2, dataArea, RectangleEdge.Right, space)
                space.left
            }
            else -> throw IllegalArgumentException()
        }
    }

    override fun setFormat(format: CPFormat<Any?>?) {
        this.format = format
        if (axis != null) {
            axis!!.setFormatOverride(this.format)
        }
        refresh()
    }

    override fun drawGrid(g2: IGraphics, d: Dimension) {
        gridRenderer.drawGrid(g2, type, coordinateAxis!!, getAxis(), gridLineRenderer, d)
    }

    override fun paintComponent(g: IGraphics) {
        val d = size
        if (coordinateAxis != null) {
            val min: Double = Math.min(coordinateAxis!!.worldMin, coordinateAxis!!.worldMax)
            val max: Double = Math.max(coordinateAxis!!.worldMin, coordinateAxis!!.worldMax)
            if (min <= max) {
                val dataArea: Rectangle2D = Rectangle2D.Double(0.0, 0.0, d.width.toDouble(), d.height.toDouble())
                getAxis().setRange(min, max)
                val ticks: List<ValueTick>
                ticks = when (type) {
                    Guide.Type.Top -> getAxis().refreshTicks(g, dataArea, RectangleEdge.Top)
                    Guide.Type.Bottom -> getAxis().refreshTicks(g, dataArea, RectangleEdge.Bottom)
                    Guide.Type.Left -> getAxis().refreshTicks(g, dataArea, RectangleEdge.Left)
                    Guide.Type.Right -> getAxis().refreshTicks(g, dataArea, RectangleEdge.Right)
                    else -> throw IllegalArgumentException()
                }

                // Draw vertical group starting lines
                for (tick in ticks) {
                    val position: Int = coordinateAxis!!.worldToScreen(tick.value)
                    if (paintTicks) {
                        g.setColor(gridLinesColor)
                        when (type) {
                            Guide.Type.Top -> g.drawLine(position, d.iheight - 5, position, d.iheight)
                            Guide.Type.Bottom -> g.drawLine(position, 0, position, 5)
                            Guide.Type.Left -> g.drawLine(d.iwidth - 5, position, d.iwidth, position)
                            Guide.Type.Right -> g.drawLine(0, position, 5, position)
                        }
                    }
                    g.setColor(labelColor)
                    val text = tick.text
                    val fontMetrics: CPFontMetrics? = g.getFontMetrics()
                    val width = g.getStringWidth(text).toInt()
                    when (type) {
                        Guide.Type.Top -> g.drawString(text, (position - (width shr 1)).toFloat(), fontMetrics!!.getHeight())
                        Guide.Type.Bottom -> g.drawString(text, (position - (width shr 1)).toFloat(), (d.height - 2).toFloat())
                        Guide.Type.Left -> g.drawString(text, (d.width - width - 7).toFloat(), (position + (fontMetrics!!.getAscent() shr 1)).toFloat())
                        Guide.Type.Right -> g.drawString(text, 7f, (position + (fontMetrics!!.getAscent() shr 1)).toFloat())
                    }
                }
            }
        }
    }

    override fun setFont(font: Font?) {
        super.setFont(font)
        getAxis().tickLabelFont = font
    }

    private fun getAxis(): ValueAxis {
        if (axis == null) {
            axis = NumberAxis()
            axis!!.tickLabelFont = Font("Tahoma", Font.PLAIN, 10)
            if (format != null) {
                axis!!.setFormatOverride(format)
            }
        }
        return axis!!
    }

    override fun setAxis(axis: ValueAxis) {
        this.axis = axis
        refresh()
    }

    fun setGridLinesColor(gridLinesColor: MkColor) {
        this.gridLinesColor = gridLinesColor
        repaint()
    }

    fun setLabelColor(labelColor: MkColor) {
        this.labelColor = labelColor
        repaint()
    }

    fun setPaintTicks(paintTicks: Boolean) {
        this.paintTicks = paintTicks
        repaint()
    }

    override fun toString(): String {
        return "AbstractAxisGuide{" +
                "coordinateAxis=" + coordinateAxis +
                ", axis=" + axis +
                '}'
    }

    init {
        addComponentListener(object : ComponentListener {
            override fun componentResized(e: ComponentEvent) {
                // ToDo: Figure out why this is needed! Otherwise, the space used by for the vertical axis isn't computed propertly.
                refresh()
            }

            override fun componentMoved(e: ComponentEvent) {}
            override fun componentShown(e: ComponentEvent) {
                refresh()
            }

            override fun componentHidden(e: ComponentEvent) {}
        })
    }
}