package com.macrofocus.slider

import com.macrofocus.common.helper.LookAndFeelHelper
import com.macrofocus.slider.plaf.RangeSliderUI
import com.macrofocus.slider.plaf.aqua.AquaRangeSliderUI
import com.macrofocus.slider.plaf.basic.BasicRangeSliderUI
import com.macrofocus.slider.plaf.metal.MetalRangeSliderUI
//import com.macrofocus.slider.plaf.synth.SynthRangeSliderUI
import com.macrofocus.slider.plaf.windows.WindowsRangeSliderUI
import java.awt.Component
import java.awt.Dimension
import java.awt.Rectangle
import java.awt.event.ComponentAdapter
import java.awt.event.ComponentEvent
import javax.swing.JComponent
import javax.swing.SwingConstants
import javax.swing.UIManager

class JRangeSlider(orientation: Int) : JComponent(), SliderView {
    private val allowOutOfRange = false
    private val rangeSliderListener: RangeSliderListener = object : RangeSliderListener {
        override fun sliderChanged(e: RangeSliderEvent) {
            updateRectangles()
            repaint()
        }

        override fun sliderScaleChanged(e: RangeSliderEvent) {
            updateRectangles()
            repaint()
        }
    }

    constructor(rangeSliderModel: RangeSliderModel?, orientation: Int) : this(orientation) {
        setModel(rangeSliderModel)
    }

    fun setModel(model: RangeSliderModel?) {
        rangeSliderModel = model
        rangeSliderModel!!.addSliderListener(rangeSliderListener)
        updateRectangles()
    }

    override fun addNotify() {
        super.addNotify()
        updateRectangles()
    }

    override fun updateUI() {
        if (UIManager.get(uIClassID) != null) {
            uI = UIManager.getUI(this) as RangeSliderUI
        } else {
            if (LookAndFeelHelper.isSynth) {
//                uI = SynthRangeSliderUI()
                throw UnsupportedOperationException("SynthRangeSliderUI")
            } else if (LookAndFeelHelper.isMetal) {
                uI = MetalRangeSliderUI()
            } else if (LookAndFeelHelper.isWindows) {
                uI = WindowsRangeSliderUI()
            } else if (LookAndFeelHelper.isAqua) {
                uI = AquaRangeSliderUI()
            } else {
                uI = BasicRangeSliderUI()
            }
        }
    }

    var uI: RangeSliderUI
        get() = ui as RangeSliderUI
        set(ui) {
            super.setUI(ui)
        }

    override fun getMinimumSize(): Dimension? {
        if (UIManager.get("ScrollBar.width") != null) {
            val scrollBarWidth = (UIManager.get("ScrollBar.width") as Int).toInt()
            if (orientation == VERTICAL) {
                return Dimension(scrollBarWidth, scrollBarWidth * 3 + 10)
            } else {
                return Dimension(scrollBarWidth * 3 + 10, scrollBarWidth)
            }
        } else {
            // JDK 1.3 scheme
            return if (orientation == VERTICAL) Dimension(16, 48) else Dimension(48, 16)
        }
    }

    override fun getPreferredSize(): Dimension? {
        return getMinimumSize()
    }

    fun updateRectangles() {
        if (orientation == VERTICAL) {
            val size: Dimension = getSize()
            val screenWidth = size.width
            val screenHeight = size.height
            val x = 0
            val width = screenWidth - 0
            val maxY = toScreen(currentMaximum)
            val minY: Int
            minY = if (rangeSliderModel!!.maximum != rangeSliderModel!!.minimum) {
                width + toScreen(currentMinimum)
            } else {
                width + screenRange
            }
            maxBounds.setBounds(x, maxY, width, width)
            minBounds.setBounds(x, minY, width, width)
        } else {
            val size: Dimension = getSize()
            val screenWidth = size.width
            val screenHeight = size.height
            val y = 0
            val height = screenHeight - 0
            val minX = toScreen(currentMinimum)
            val maxX: Int
            maxX = if (rangeSliderModel!!.maximum != rangeSliderModel!!.minimum) {
                height + toScreen(currentMaximum)
            } else {
                height + screenRange
            }
            maxBounds.setBounds(maxX, y, height, height)
            minBounds.setBounds(minX, y, height, height)
        }
        if (orientation == VERTICAL) {
            val size: Dimension = getSize()
            trackBounds.setBounds(0, minBounds.height, size.width, size.height - minBounds.height - maxBounds.height)
            rangeRectangle.setBounds(
                0, maxBounds.y + maxBounds.height, size.width,
                minBounds.y - maxBounds.y - maxBounds.height
            )
            val minY = Math.min(maxBounds.y + maxBounds.height, minBounds.y)
            val maxY = Math.max(maxBounds.y + maxBounds.height, minBounds.y)
            thumbBounds.setBounds(0, minY, size.width, maxY - minY)
        } else {
            val size: Dimension = getSize()
            trackBounds.setBounds(minBounds.width, 0, size.width - minBounds.width - maxBounds.width, size.height)
            rangeRectangle.setBounds(
                minBounds.x + minBounds.width, 0,
                maxBounds.x - minBounds.x - minBounds.width, size.height
            )
            val minX = Math.min(minBounds.x + minBounds.width, maxBounds.x)
            val maxX = Math.max(minBounds.x + minBounds.width, maxBounds.x)
            thumbBounds.setBounds(minX, 0, maxX - minX, size.height)
        }
    }

    private val currentMaximum: Double
        private get() = if (!allowOutOfRange && (!rangeSliderModel!!.isInvertedScale && rangeSliderModel!!.currentMaximum > rangeSliderModel!!.maximum || rangeSliderModel!!.isInvertedScale && rangeSliderModel!!.currentMaximum < rangeSliderModel!!.maximum)) {
            rangeSliderModel!!.maximum
        } else {
            rangeSliderModel!!.currentMaximum
        }
    private val currentMinimum: Double
        private get() = if (!allowOutOfRange && (!rangeSliderModel!!.isInvertedScale && rangeSliderModel!!.currentMinimum < rangeSliderModel!!.minimum || rangeSliderModel!!.isInvertedScale && rangeSliderModel!!.currentMinimum > rangeSliderModel!!.minimum)) {
            rangeSliderModel!!.minimum
        } else {
            rangeSliderModel!!.currentMinimum
        }

    override fun toScreen(value: Double): Int {
        return if (orientation == VERTICAL) {
            val screenRange = (getHeight() - 2 * getWidth()) as Double
            val worldRange: Double = rangeSliderModel!!.maximum - rangeSliderModel!!.minimum
            Math.round(screenRange * (worldRange - (value - rangeSliderModel!!.minimum)) / worldRange).toInt()
        } else {
            val screenRange = (getWidth() - 2 * getHeight()) as Double
            val worldRange: Double = Math.abs(rangeSliderModel!!.maximum - rangeSliderModel!!.minimum)
            Math.round(screenRange * (value - rangeSliderModel!!.minimum) / worldRange).toInt()
        }
    }

    override fun toWorld(value: Int): Double {
        return if (orientation == VERTICAL) {
            val size: Dimension = getSize()
            val screenWidth = size.width
            val screenHeight = size.height
            val screenRange = (screenHeight - 2 * screenWidth).toDouble()
            val worldRange: Double = rangeSliderModel!!.maximum - rangeSliderModel!!.minimum
            worldRange * (screenRange - value) / screenRange
        } else {
            val size: Dimension = getSize()
            val screenWidth = size.width
            val screenHeight = size.height
            val screenRange = (screenWidth - 2 * screenHeight).toDouble()
            val worldRange: Double = rangeSliderModel!!.maximum - rangeSliderModel!!.minimum
            worldRange * value.toDouble() / screenRange
        }
    }

    override val screenRange: Int
        get() {
            val size: Dimension = getSize()
            val screenWidth = size.width
            val screenHeight = size.height
            return if (orientation == VERTICAL) {
                screenHeight - 2 * screenWidth
            } else {
                screenWidth - 2 * screenHeight
            }
        }
    override val component: Component
        get() = this
    val isPaintThumb: Boolean
        get() = true
    val isPaintTrack: Boolean
        get() = true
    val isPaintHandles: Boolean
        get() = true

    //    protected boolean isFreeStanding = true;
    protected var leftToRight = true
    override var thumbBounds = Rectangle()
        protected set
    override var trackBounds = Rectangle()
        protected set
    override val orientation: Int
    override var rangeSliderModel: RangeSliderModel? = null
        set(rangeSliderModel) {
            if (this.rangeSliderModel != null) {
                this.rangeSliderModel!!.removeSliderListener(rangeSliderListener)
            }
            this.rangeSliderModel = rangeSliderModel
            this.rangeSliderModel!!.addSliderListener(rangeSliderListener)
            updateRectangles()

        }
    override var minBounds = Rectangle()
        protected set
        get() = field
    override var maxBounds = Rectangle()
        protected set
        get() = field
    var rangeRectangle = Rectangle()
        protected set
    var histogramRenderer: HistogramRangeSliderRenderer? = null

    companion object {
        val HORIZONTAL: Int = SwingConstants.HORIZONTAL
        val VERTICAL: Int = SwingConstants.VERTICAL
        val uIClassID = "MRangeSliderUI"
    }

    init {
        updateUI()
        this.orientation = orientation
        addComponentListener(object : ComponentAdapter() {
            override fun componentShown(e: ComponentEvent) {
                updateRectangles()
                repaint()
            }

            override fun componentResized(e: ComponentEvent) {
                updateRectangles()
                repaint()
            }
        })
    }
}