/*
 * Copyright (c) 2010 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.slider.plaf.basic

import com.macrofocus.slider.JRangeSlider
import com.macrofocus.slider.RangeSliderModel
import com.macrofocus.slider.plaf.RangeSliderUI
import java.awt.*
import javax.swing.JComponent
import javax.swing.UIManager
import javax.swing.plaf.ComponentUI

open class BasicRangeSliderUI : RangeSliderUI() {
    private val background: Color
    private val shadow: Color
    private val darkShadow: Color
    private val highlight: Color
    override fun paint(g: Graphics, c: JComponent) {
        val view = c as JRangeSlider
        view.updateRectangles()
        paintRange(view.rangeSliderModel, view, g, view.orientation, view.trackBounds, view.thumbBounds)
        paintThumb(view, g, view.orientation, view.thumbBounds)
        paintHandles(g, view)
    }

    private fun paintHandles(g: Graphics, view: JRangeSlider) {
        if (view.isPaintHandles) {
            val maxRectangle = view.maxBounds
            val minRectangle = view.minBounds
            if (view.orientation == RangeSliderUI.Companion.VERTICAL) {
                g.translate(maxRectangle!!.x, maxRectangle!!.y)
                paintArrowButton(view, g, RangeSliderUI.Companion.SOUTH, Rectangle(0, 0, maxRectangle!!.width, maxRectangle!!.height))
                g.translate(-maxRectangle!!.x, -maxRectangle!!.y)
                g.translate(minRectangle!!.x, minRectangle!!.y)
                paintArrowButton(view, g, RangeSliderUI.Companion.NORTH, Rectangle(0, 0, minRectangle!!.width, minRectangle!!.height))
                g.translate(-minRectangle!!.x, -minRectangle!!.y)
            } else {
                g.translate(maxRectangle!!.x, maxRectangle!!.y)
                paintArrowButton(view, g, RangeSliderUI.Companion.WEST, Rectangle(0, 0, maxRectangle!!.width, maxRectangle!!.height))
                g.translate(-maxRectangle!!.x, -maxRectangle!!.y)
                g.translate(minRectangle!!.x, minRectangle!!.y)
                paintArrowButton(view, g, RangeSliderUI.Companion.EAST, Rectangle(0, 0, minRectangle!!.width, minRectangle!!.height))
                g.translate(-minRectangle!!.x, -minRectangle!!.y)
            }
        }
    }

    override fun paintRange(
        rangeSliderModel: RangeSliderModel?,
        view: JRangeSlider,
        g: Graphics,
        orientation: Int,
        trackBounds: Rectangle?,
        thumbBounds: Rectangle?
    ) {
        if (view.isPaintTrack) {
            if (view.histogramRenderer != null) {
                val c = view.histogramRenderer?.getRangeSliderHistogramRendererComponent(view, rangeSliderModel!!.histogram)
                if (c != null) {
                    c.setSize(trackBounds!!.width, trackBounds.height)
                    val g2 = g.create(trackBounds.x, trackBounds.y, trackBounds.width, trackBounds.height)
                    c.paint(g2)
                    g2.dispose()
                }
            } else {
                if (view.isOpaque) {
                    g.color = backgroundColor
                    g.fillRect(trackBounds!!.x, trackBounds.y, trackBounds.width, trackBounds.height)
                }
            }

            //        if (trackHighlight == DECREASE_HIGHLIGHT) {
            //            paintDecreaseHighlight(g);
            //        } else if (trackHighlight == INCREASE_HIGHLIGHT) {
            //            paintIncreaseHighlight(g);
            //        }
        }
    }

    override fun paintThumb(sliderView: JRangeSlider, g: Graphics, orientation: Int, thumbBounds: Rectangle?) {
        if (sliderView.isPaintThumb && sliderView.histogramRenderer == null) {
            if (thumbBounds!!.isEmpty || !sliderView.isEnabled) {
                return
            }
            val w = thumbBounds.width
            val h = thumbBounds.height
            g.translate(thumbBounds.x, thumbBounds.y)
            g.color = thumbDarkShadowColor
            g.drawRect(0, 0, w - 1, h - 1)
            g.color = thumbColor
            g.fillRect(0, 0, w - 1, h - 1)
            g.color = thumbHighlightColor
            g.drawLine(1, 1, 1, h - 2)
            g.drawLine(2, 1, w - 3, 1)
            g.color = thumbLightShadowColor
            g.drawLine(2, h - 2, w - 2, h - 2)
            g.drawLine(w - 2, 1, w - 2, h - 3)
            g.translate(-thumbBounds.x, -thumbBounds.y)
        }
    }

    protected var backgroundColor: Color
    protected var thumbColor: Color
    protected var backgroundControlColor: Color? = null
    private val thumbLightShadowColor: Color
    private val thumbDarkShadowColor: Color
    private val trackColor: Color
    private val trackHighlightColor: Color
    private val isFreeStanding = true
    private val leftToRight = true
    override fun paintArrowButton(sliderView: JComponent, g: Graphics, direction: Int, bounds: Rectangle) {
        val origColor: Color
        val isPressed: Boolean
        val isEnabled: Boolean
        val w: Int
        val h: Int
        var size: Int
        w = bounds.width
        h = bounds.height
        origColor = g.color
        isPressed = false
        isEnabled = sliderView.isEnabled
        g.color = background
        g.fillRect(1, 1, w - 2, h - 2)

        /// Draw the proper Border
        if (isPressed) {
            g.color = shadow
            g.drawRect(0, 0, w - 1, h - 1)
        } else {
            // Using the background color set above
            g.drawLine(0, 0, 0, h - 1)
            g.drawLine(1, 0, w - 2, 0)
            g.color = highlight // inner 3D border
            g.drawLine(1, 1, 1, h - 3)
            g.drawLine(2, 1, w - 3, 1)
            g.color = shadow // inner 3D border
            g.drawLine(1, h - 2, w - 2, h - 2)
            g.drawLine(w - 2, 1, w - 2, h - 3)
            g.color = darkShadow // black drop shadow  __|
            g.drawLine(0, h - 1, w - 1, h - 1)
            g.drawLine(w - 1, h - 1, w - 1, 0)
        }

        // If there's no room to draw arrow, bail
        if (h < 5 || w < 5) {
            g.color = origColor
            return
        }
        if (isPressed) {
            g.translate(1, 1)
        }

        // Draw the arrow
        size = Math.min((h - 4) / 3, (w - 4) / 3)
        size = Math.max(size, 2)
        paintTriangle(
            g, (w - size) / 2, (h - size) / 2,
            size, direction, isEnabled
        )

        // Reset the Graphics back to it's original settings
        if (isPressed) {
            g.translate(-1, -1)
        }
        g.color = origColor
    }

    //    protected void paintDecreaseHighlight(Graphics g) {
    //        Insets insets = scrollbar.getInsets();
    //        Rectangle thumbR = getThumbBounds();
    //        g.setColor(trackHighlightColor);
    //
    //        if (scrollbar.getOrientation() == JScrollBar.VERTICAL) {
    //            int x = insets.left;
    //            int y = decrButton.getY() + decrButton.getHeight();
    //            int w = scrollbar.getWidth() - (insets.left + insets.right);
    //            int h = thumbR.y - y;
    //            g.fillRect(x, y, w, h);
    //        } else {
    //            int x = decrButton.getX() + decrButton.getHeight();
    //            int y = insets.top;
    //            int w = thumbR.x - x;
    //            int h = scrollbar.getHeight() - (insets.top + insets.bottom);
    //            g.fillRect(x, y, w, h);
    //        }
    //    }
    //
    //    protected void paintIncreaseHighlight(Graphics g, int orientation) {
    //        Insets insets = scrollbar.getInsets();
    //        Rectangle thumbR = getThumbBounds();
    //        g.setColor(trackHighlightColor);
    //
    //        if (orientation == VERTICAL) {
    //            int x = insets.left;
    //            int y = thumbR.y + thumbR.height;
    //            int w = scrollbar.getWidth() - (insets.left + insets.right);
    //            int h = incrButton.getY() - y;
    //            g.fillRect(x, y, w, h);
    //        } else {
    //            int x = thumbR.x + thumbR.width;
    //            int y = insets.top;
    //            int w = incrButton.getX() - x;
    //            int h = scrollbar.getHeight() - (insets.top + insets.bottom);
    //            g.fillRect(x, y, w, h);
    //        }
    //    }
    fun paintTriangle(
        g: Graphics, x: Int, y: Int, size: Int,
        direction: Int, isEnabled: Boolean
    ) {
        var size = size
        val oldColor = g.color
        var mid: Int
        var i: Int
        var j: Int
        j = 0
        size = Math.max(size, 2)
        mid = size / 2
        g.translate(x, y)
        if (isEnabled) g.color = UIManager.getColor("controlDkShadow") else g.color = UIManager.getColor("controlShadow")
        when (direction) {
            RangeSliderUI.Companion.NORTH -> {
                mid--
                i = 0
                while (i < size) {
                    g.drawLine(mid - i, i, mid + i, i)
                    i++
                }
                if (!isEnabled) {
                    g.color = UIManager.getColor("controlLtHighlight")
                    g.drawLine(mid - i + 2, i, mid + i, i)
                }
            }
            RangeSliderUI.Companion.SOUTH -> {
                mid--
                if (!isEnabled) {
                    g.translate(1, 1)
                    g.color = UIManager.getColor("controlLtHighlight")
                    i = size - 1
                    while (i >= 0) {
                        g.drawLine(mid - i, j, mid + i, j)
                        j++
                        i--
                    }
                    g.translate(-1, -1)
                    g.color = UIManager.getColor("controlShadow")
                }
                j = 0
                i = size - 1
                while (i >= 0) {
                    g.drawLine(mid - i, j, mid + i, j)
                    j++
                    i--
                }
            }
            RangeSliderUI.Companion.WEST -> {
                i = 0
                while (i < size) {
                    g.drawLine(i, mid - i, i, mid + i)
                    i++
                }
                if (!isEnabled) {
                    g.color = UIManager.getColor("controlLtHighlight")
                    g.drawLine(i, mid - i + 2, i, mid + i)
                }
            }
            RangeSliderUI.Companion.EAST -> {
                if (!isEnabled) {
                    g.translate(1, 1)
                    g.color = UIManager.getColor("controlLtHighlight")
                    i = size - 1
                    while (i >= 0) {
                        g.drawLine(j, mid - i, j, mid + i)
                        j++
                        i--
                    }
                    g.translate(-1, -1)
                    g.color = UIManager.getColor("controlShadow")
                }
                j = 0
                i = size - 1
                while (i >= 0) {
                    g.drawLine(j, mid - i, j, mid + i)
                    j++
                    i--
                }
            }
        }
        g.translate(-x, -y)
        g.color = oldColor
    }

    protected var trackHighlight: Int

    companion object {
        private val rangeSliderUI = BasicRangeSliderUI()
        fun createUI(c: JComponent?): ComponentUI {
            return rangeSliderUI
        }

        private lateinit var foregroundColor: Color
        private val shadowColor: Color? = null
        private val highlightColor: Color? = null
        private val darkShadowColor: Color? = null
        private val thumbShadow: Color? = null
        private lateinit var thumbHighlightColor: Color

        /*
     * Convenience function for determining ComponentOrientation.  Helps us
     * avoid having Munge directives throughout the code.
     */
        private fun isLeftToRight(c: Component): Boolean {
            return c.componentOrientation.isLeftToRight
        }

        protected const val NO_HIGHLIGHT = 0
        protected const val DECREASE_HIGHLIGHT = 1
        protected const val INCREASE_HIGHLIGHT = 2
    }

    init {
        backgroundColor = UIManager.getColor("ScrollBar.background")
        foregroundColor = UIManager.getColor("ScrollBar.foreground")
        thumbHighlightColor = UIManager.getColor("ScrollBar.thumbHighlight")
        thumbLightShadowColor = UIManager.getColor("ScrollBar.thumbShadow")
        thumbDarkShadowColor = UIManager.getColor("ScrollBar.thumbDarkShadow")
        thumbColor = UIManager.getColor("ScrollBar.thumb")
        trackColor = UIManager.getColor("ScrollBar.track")
        trackHighlightColor = UIManager.getColor("ScrollBar.trackHighlight")
        trackHighlight = DECREASE_HIGHLIGHT
        val control = UIManager.getColor("control")
        background = Color(control.red, control.green, control.blue, 127)
        shadow = UIManager.getColor("controlShadow")
        darkShadow = UIManager.getColor("controlDkShadow")
        highlight = UIManager.getColor("controlLtHighlight")
    }
}