/*
 * Copyright (c) 2010 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.slider.plaf.metal

import com.macrofocus.slider.JRangeSlider
import com.macrofocus.slider.RangeSliderModel
import com.macrofocus.slider.plaf.RangeSliderUI
import com.macrofocus.slider.plaf.basic.BasicRangeSliderUI
import java.awt.*
import java.util.*
import javax.swing.*
import javax.swing.plaf.ComponentUI
import javax.swing.plaf.metal.MetalLookAndFeel

class MetalRangeSliderUI : BasicRangeSliderUI() {
    override fun paintRange(
        rangeSliderModel: RangeSliderModel?,
        view: JRangeSlider,
        g: Graphics,
        orientation: Int,
        trackBounds: Rectangle?,
        thumbBounds: Rectangle?
    ) {
        if (view.isPaintTrack) {
            g.translate(trackBounds!!.x, trackBounds.y)
            if (view.histogramRenderer != null) {
                val c = view.histogramRenderer?.getRangeSliderHistogramRendererComponent(view, rangeSliderModel!!.histogram)
                if (c != null) {
                    c.setSize(trackBounds.width, trackBounds.height)
                    val g2 = g.create(trackBounds.x, trackBounds.y, trackBounds.width, trackBounds.height)
                    c.paint(g2)
                    g2.dispose()
                }
            } else {
                if (view.isOpaque) {
                    g.color = backgroundColor
                    g.fillRect(trackBounds.x, trackBounds.y, trackBounds.width, trackBounds.height)
                }
            }
            if (orientation == RangeSliderUI.Companion.VERTICAL) {
                if (!isFreeStanding) {
                    if (!leftToRight) {
                        trackBounds.width += 1
                        g.translate(-1, 0)
                    } else {
                        trackBounds.width += 2
                    }
                }
                if (view.isEnabled) {
                    g.color = darkShadowColor
                    g.drawLine(0, 0, 0, trackBounds.height - 1)
                    g.drawLine(trackBounds.width - 2, 0, trackBounds.width - 2, trackBounds.height - 1)
                    g.drawLine(2, trackBounds.height - 1, trackBounds.width - 1, trackBounds.height - 1)
                    g.drawLine(2, 0, trackBounds.width - 2, 0)
                    g.color = shadowColor
                    //	g.setColor( Color.red);
                    g.drawLine(1, 1, 1, trackBounds.height - 2)
                    g.drawLine(1, 1, trackBounds.width - 3, 1)
                    if (rangeSliderModel!!.currentMaximum != rangeSliderModel.maximum) {  // thumb shadow
                        val y = thumbBounds!!.y + thumbBounds.height - trackBounds.y
                        g.drawLine(1, y, trackBounds.width - 1, y)
                    }
                    g.color = highlightColor
                    g.drawLine(trackBounds.width - 1, 0, trackBounds.width - 1, trackBounds.height - 1)
                } else {
                    drawDisabledBorder(g, 0, 0, trackBounds.width, trackBounds.height)
                }
                if (!isFreeStanding) {
                    if (!leftToRight) {
                        trackBounds.width -= 1
                        g.translate(1, 0)
                    } else {
                        trackBounds.width -= 2
                    }
                }
            } else  // HORIZONTAL
            {
                if (!isFreeStanding) {
                    trackBounds.height += 2
                }
                if (view.isEnabled) {
                    g.color = darkShadowColor
                    g.drawLine(0, 0, trackBounds.width - 1, 0) // top
                    g.drawLine(0, 2, 0, trackBounds.height - 2) // left
                    g.drawLine(0, trackBounds.height - 2, trackBounds.width - 1, trackBounds.height - 2) // bottom
                    g.drawLine(trackBounds.width - 1, 2, trackBounds.width - 1, trackBounds.height - 1) // right
                    g.color = shadowColor
                    //	g.setColor( Color.red);
                    g.drawLine(1, 1, trackBounds.width - 2, 1) // top
                    g.drawLine(1, 1, 1, trackBounds.height - 3) // left
                    g.drawLine(0, trackBounds.height - 1, trackBounds.width - 1, trackBounds.height - 1) // bottom
                    if (rangeSliderModel!!.currentMaximum != rangeSliderModel.maximum) {  // thumb shadow
                        val x = thumbBounds!!.x + thumbBounds.width - trackBounds.x
                        g.drawLine(x, 1, x, trackBounds.height - 1)
                    }

                    // Bottom white line
                    g.color = highlightColor
                    g.drawLine(0, trackBounds.height - 1, trackBounds.width - 1, trackBounds.height - 1)
                } else {
                    drawDisabledBorder(g, 0, 0, trackBounds.width, trackBounds.height)
                }
                if (!isFreeStanding) {
                    trackBounds.height -= 2
                }
            }
            g.translate(-trackBounds.x, -trackBounds.y)
        }
    }

    // JDK 1.3
    //    public void paintRange(SliderModel sliderModel, JSliderView sliderView, final Graphics g, final int orientation, final Rectangle trackBounds, final Rectangle thumbBounds) {
    //        g.translate(trackBounds.x, trackBounds.y);
    //
    //        if (orientation == VERTICAL) {
    //            if (!isFreeStanding) {
    //                if (!leftToRight) {
    //                    trackBounds.width += 1;
    //                    g.translate(-1, 0);
    //                } else {
    //                    trackBounds.width += 2;
    //                }
    //            }
    //
    //            if (sliderView.isEnabled()) {
    //                g.setColor(darkShadowColor);
    //                g.drawLine(0, 0, 0, trackBounds.height - 1);
    //                g.drawLine(trackBounds.width - 2, 0, trackBounds.width - 2, trackBounds.height - 1);
    //                g.drawLine(2, trackBounds.height - 1, trackBounds.width - 1, trackBounds.height - 1);
    //                g.drawLine(2, 0, trackBounds.width - 2, 0);
    //
    //                g.setColor(shadowColor);
    //                //	g.setColor( Color.red);
    //                g.drawLine(1, 1, 1, trackBounds.height - 2);
    //                g.drawLine(1, 1, trackBounds.width - 3, 1);
    //                if (sliderModel.getRangeMaximum() != sliderModel.getMaximum()) { // thumb shadow
    //                    int y = thumbBounds.y + thumbBounds.height - trackBounds.y;
    //                    g.drawLine(1, y, trackBounds.width - 1, y);
    //                }
    //                g.setColor(highlightColor);
    //                g.drawLine(trackBounds.width - 1, 0, trackBounds.width - 1, trackBounds.height - 1);
    //            } else {
    //                drawDisabledBorder(g, 0, 0, trackBounds.width, trackBounds.height);
    //            }
    //
    //            if (!isFreeStanding) {
    //                if (!leftToRight) {
    //                    trackBounds.width -= 1;
    //                    g.translate(1, 0);
    //                } else {
    //                    trackBounds.width -= 2;
    //                }
    //            }
    //        } else  // HORIZONTAL
    //        {
    //            if (!isFreeStanding) {
    //                trackBounds.height += 2;
    //            }
    //
    //            if (sliderView.isEnabled()) {
    //                g.setColor(darkShadowColor);
    //                g.drawLine(0, 0, trackBounds.width - 1, 0);  // top
    //                g.drawLine(0, 2, 0, trackBounds.height - 2); // left
    //                g.drawLine(0, trackBounds.height - 2, trackBounds.width - 1, trackBounds.height - 2); // bottom
    //                g.drawLine(trackBounds.width - 1, 2, trackBounds.width - 1, trackBounds.height - 1); // right
    //
    //                g.setColor(shadowColor);
    //                //	g.setColor( Color.red);
    //                g.drawLine(1, 1, trackBounds.width - 2, 1);  // top
    //                g.drawLine(1, 1, 1, trackBounds.height - 3); // left
    //                g.drawLine(0, trackBounds.height - 1, trackBounds.width - 1, trackBounds.height - 1); // bottom
    //                if (sliderModel.getRangeMaximum() != sliderModel.getMaximum()) { // thumb shadow
    //                    int x = thumbBounds.x + thumbBounds.width - trackBounds.x;
    //                    g.drawLine(x, 1, x, trackBounds.height - 1);
    //                }
    //            } else {
    //                drawDisabledBorder(g, 0, 0, trackBounds.width, trackBounds.height);
    //            }
    //
    //            if (!isFreeStanding) {
    //                trackBounds.height -= 2;
    //            }
    //        }
    //
    //        g.translate(-trackBounds.x, -trackBounds.y);
    //    }
    fun paintThumb(sliderView: JComponent, g: Graphics, orientation: Int, thumbBounds: Rectangle) {
        if (!sliderView.isEnabled) {
            return
        }
        g.translate(thumbBounds.x, thumbBounds.y)
        if (orientation == RangeSliderUI.Companion.VERTICAL) {
            if (!isFreeStanding) {
                if (!leftToRight) {
                    thumbBounds.width += 1
                    g.translate(-1, 0)
                } else {
                    thumbBounds.width += 2
                }
            }
            g.color = Companion.thumbColor
            g.fillRect(0, 0, thumbBounds.width - 2, thumbBounds.height - 1)
            g.color = thumbShadow
            g.drawRect(0, 0, thumbBounds.width - 2, thumbBounds.height - 1)
            g.color = thumbHighlightColor
            g.drawLine(1, 1, thumbBounds.width - 3, 1)
            g.drawLine(1, 1, 1, thumbBounds.height - 2)
            bumps.setBumpArea(thumbBounds.width - 6, thumbBounds.height - 7)
            bumps.paintIcon(sliderView, g, 3, 4)
            if (!isFreeStanding) {
                if (!leftToRight) {
                    thumbBounds.width -= 1
                    g.translate(1, 0)
                } else {
                    thumbBounds.width -= 2
                }
            }
        } else  // HORIZONTAL
        {
            if (!isFreeStanding) {
                if (!leftToRight) {
                    thumbBounds.height += 1
                    g.translate(0, -1)
                } else {
                    thumbBounds.height += 2
                }
            }
            //            if (!isFreeStanding) {
//                thumbBounds.height += 2;
//            }
            g.color = Companion.thumbColor
            g.fillRect(0, 0, thumbBounds.width - 1, thumbBounds.height - 2)
            g.color = thumbShadow
            g.drawRect(0, 0, thumbBounds.width - 1, thumbBounds.height - 2)

            // ???
            g.color = thumbHighlightColor
            g.drawLine(1, 1, thumbBounds.width - 3, 1)
            g.drawLine(1, 1, 1, thumbBounds.height - 2)
            bumps.setBumpArea(thumbBounds.width - 7, thumbBounds.height - 6)
            bumps.paintIcon(sliderView, g, 4, 3)

//            if (!isFreeStanding) {
//                thumbBounds.height -= 2;
//            }
            if (!isFreeStanding) {
                if (!leftToRight) {
                    thumbBounds.height -= 1
                    g.translate(0, 1)
                } else {
                    thumbBounds.height -= 2
                }
            }
        }
        g.translate(-thumbBounds.x, -thumbBounds.y)
    }

    private val bumps: MetalBumps
    private val isFreeStanding = true
    private val leftToRight = true

    internal inner class MetalBumps : Icon {
        protected var xBumps = 0
        protected var yBumps = 0
        protected var topColor: Color = MetalLookAndFeel.getPrimaryControlHighlight()
        protected var shadowColor: Color = MetalLookAndFeel.getPrimaryControlDarkShadow()
        protected var backColor: Color = MetalLookAndFeel.getPrimaryControlShadow()
        protected val buffers: Vector<Any?> = Vector<Any?>()
        protected var buffer: BumpBuffer?

        constructor(bumpArea: Dimension) : this(bumpArea.width, bumpArea.height) {}
        constructor(width: Int, height: Int) {
            setBumpArea(width, height)
            buffer = getBuffer(topColor, shadowColor, backColor)
            if (buffer == null) {
                createBuffer()
            }
        }

        constructor(
            width: Int, height: Int,
            newTopColor: Color, newShadowColor: Color, newBackColor: Color
        ) {
            setBumpArea(width, height)
            setBumpColors(newTopColor, newShadowColor, newBackColor)
            buffer = getBuffer(topColor, shadowColor, backColor)
            if (buffer == null) {
                createBuffer()
            }
        }

        protected fun createBuffer() {
            buffer = BumpBuffer(topColor, shadowColor, backColor)
            buffers.addElement(buffer as Any?)
        }

        protected fun getBuffer(aTopColor: Color, aShadowColor: Color, aBackColor: Color): BumpBuffer? {
            var result: BumpBuffer? = null
            val elements = buffers.elements()
            while (elements.hasMoreElements()) {
                val aBuffer = elements.nextElement() as BumpBuffer
                if (aBuffer.hasSameColors(aTopColor, aShadowColor, aBackColor)) {
                    result = aBuffer
                    break
                }
            }
            return result
        }

        fun setBumpArea(bumpArea: Dimension) {
            setBumpArea(bumpArea.width, bumpArea.height)
        }

        fun setBumpArea(width: Int, height: Int) {
            xBumps = width / 2
            yBumps = height / 2
        }

        fun setBumpColors(newTopColor: Color, newShadowColor: Color, newBackColor: Color) {
            topColor = newTopColor
            shadowColor = newShadowColor
            backColor = newBackColor
            buffer = getBuffer(topColor, shadowColor, backColor)
            if (buffer == null) {
                createBuffer()
            }
        }

        override fun paintIcon(c: Component, g: Graphics, x: Int, y: Int) {
            var x = x
            var y = y
            val bufferWidth = buffer!!.imageSize.width
            val bufferHeight = buffer!!.imageSize.height
            val iconWidth = iconWidth
            val iconHeight = iconHeight
            val x2 = x + iconWidth
            val y2 = y + iconHeight
            val savex = x
            while (y < y2) {
                val h = Math.min(y2 - y, bufferHeight)
                x = savex
                while (x < x2) {
                    val w = Math.min(x2 - x, bufferWidth)
                    g.drawImage(
                        buffer!!.getImage(),
                        x, y, x + w, y + h,
                        0, 0, w, h,
                        null
                    )
                    x += bufferWidth
                }
                y += bufferHeight
            }
        }

        override fun getIconWidth(): Int {
            return xBumps * 2
        }

        override fun getIconHeight(): Int {
            return yBumps * 2
        }
    }

    internal inner class BumpBuffer(aTopColor: Color?, aShadowColor: Color?, aBackColor: Color?) {
        var frame: Frame? = null
        protected var component: Component? = null
        val IMAGE_SIZE = 64
        val imageSize = Dimension(IMAGE_SIZE, IMAGE_SIZE)

        @Transient
        private var image: Image?
        var topColor: Color?
        var shadowColor: Color?
        var backColor: Color?
        fun hasSameColors(aTopColor: Color, aShadowColor: Color, aBackColor: Color): Boolean {
            return topColor != null && topColor == aTopColor && shadowColor != null && shadowColor == aShadowColor && backColor != null && backColor == aBackColor
        }

        fun getImage(): Image? {
            if (image == null) {
                image = component!!.createImage(IMAGE_SIZE, IMAGE_SIZE)
                fillBumpBuffer()
            }
            return image
        }

        protected fun fillBumpBuffer() {
            val g = image!!.graphics
            g.color = backColor
            g.fillRect(0, 0, IMAGE_SIZE, IMAGE_SIZE)
            g.color = topColor
            run {
                var x = 0
                while (x < IMAGE_SIZE) {
                    var y = 0
                    while (y < IMAGE_SIZE) {
                        g.drawLine(x, y, x, y)
                        g.drawLine(x + 2, y + 2, x + 2, y + 2)
                        y += 4
                    }
                    x += 4
                }
            }
            g.color = shadowColor
            var x = 0
            while (x < IMAGE_SIZE) {
                var y = 0
                while (y < IMAGE_SIZE) {
                    g.drawLine(x + 1, y + 1, x + 1, y + 1)
                    g.drawLine(x + 3, y + 3, x + 3, y + 3)
                    y += 4
                }
                x += 4
            }
            g.dispose()
        }

        protected fun createComponent() {
            if (frame == null) {
                frame = Frame("bufferCreator")
            }
            if (component == null) {
                component = Canvas()
                frame!!.add(component, BorderLayout.CENTER)
            }
            // fix for 4185993 (moved this outside if block)
            frame!!.addNotify()
        }

        init {
            createComponent()
            image = component!!.createImage(IMAGE_SIZE, IMAGE_SIZE)
            topColor = aTopColor
            shadowColor = aShadowColor
            backColor = aBackColor
            fillBumpBuffer()
        }
    }

    override fun paintArrowButton(sliderView: JComponent, g: Graphics, direction: Int, bounds: Rectangle) {
        val leftToRight = isLeftToRight(sliderView)
        val isEnabled = sliderView.parent.isEnabled
        val arrowColor: Color = if (isEnabled) MetalLookAndFeel.getControlInfo() else MetalLookAndFeel.getControlDisabled()
        val isPressed = false
        var width = bounds.width
        var height = bounds.height
        val w = width
        val h = height
        val arrowHeight = (height + 1) / 4
        val arrowWidth = (height + 1) / 2
        if (isPressed) {
            g.color = MetalLookAndFeel.getControlShadow()
        } else {
            g.color = sliderView.background
        }
        g.fillRect(0, 0, width, height)
        if (direction == RangeSliderUI.Companion.NORTH) {
            if (!isFreeStanding) {
                height += 1
                g.translate(0, -1)
                if (!leftToRight) {
                    width += 1
                    g.translate(-1, 0)
                } else {
                    width += 2
                }
            }

            // Draw the arrow
            g.color = arrowColor
            val startY = (h + 1 - arrowHeight) / 2 - 1
            val startX = w / 2 - 1
            //		    System.out.println( "startX :" + startX + " startY :"+startY);
            for (line in 0 until arrowHeight) {
                g.drawLine(startX - line, startY + line, startX + line + 1, startY + line)
            }
            /*	g.drawLine( 7, 6, 8, 6 );
                g.drawLine( 6, 7, 9, 7 );
                g.drawLine( 5, 8, 10, 8 );
                g.drawLine( 4, 9, 11, 9 );*/if (isEnabled) {
                g.color = highlightColor
                if (!isPressed) {
                    g.drawLine(1, 0, width - 3, 0)
                    g.drawLine(1, 0, 1, height - 3)
                }
                g.drawLine(1, height - 1, width - 1, height - 1)
                g.drawLine(width - 1, 0, width - 1, height - 1)
                g.color = darkShadowColor
                g.drawLine(0, 0, 0, height - 2)
                g.drawLine(width - 2, 0, width - 2, height - 2)
                g.drawLine(2, height - 2, width - 2, height - 2)
            } else {
                drawDisabledBorder(g, 0, -1, width, height + 1)
            }
            if (!isFreeStanding) {
                height -= 1
                g.translate(0, 1)
                if (!leftToRight) {
                    width -= 1
                    g.translate(1, 0)
                } else {
                    width -= 2
                }
            }
        } else if (direction == RangeSliderUI.Companion.SOUTH) {
            if (!isFreeStanding) {
                height += 1
                if (!leftToRight) {
                    width += 1
                    g.translate(-1, 0)
                } else {
                    width += 2
                }
            }

            // Draw the arrow
            g.color = arrowColor
            val startY = (h + 1 - arrowHeight) / 2 + arrowHeight - 1 + 1
            val startX = w / 2 - 1

            //	    System.out.println( "startX2 :" + startX + " startY2 :"+startY);
            for (line in 0 until arrowHeight) {
                g.drawLine(startX - line, startY - line, startX + line + 1, startY - line)
            }

            /*	g.drawLine( 4, 5, 11, 5 );
                g.drawLine( 5, 6, 10, 6 );
                g.drawLine( 6, 7, 9, 7 );
                g.drawLine( 7, 8, 8, 8 ); */if (isEnabled) {
                g.color = highlightColor
                if (!isPressed) {
                    g.drawLine(1, 1, width - 3, 1)
                    g.drawLine(1, 1, 1, height - 1)
                }
                g.drawLine(width - 1, 1, width - 1, height - 1)
                g.color = darkShadowColor
                g.drawLine(0, 0, width - 2, 0)
                g.drawLine(0, 0, 0, height - 1)
                g.drawLine(width - 2, 2, width - 2, height - 1)
            } else {
                drawDisabledBorder(g, 0, 0, width, height + 1)
            }
            if (!isFreeStanding) {
                height -= 1
                if (!leftToRight) {
                    width -= 1
                    g.translate(1, 0)
                } else {
                    width -= 2
                }
            }
        } else if (direction == RangeSliderUI.Companion.EAST) {
            if (!isFreeStanding) {
                height += 2
                width += 1
            }

            // Draw the arrow
            g.color = arrowColor
            val startX = (w + 1 - arrowHeight) / 2 + arrowHeight - 1 + 1
            val startY = h / 2 - 1

            //System.out.println( "startX2 :" + startX + " startY2 :"+startY);
            for (line in 0 until arrowHeight) {
                g.drawLine(startX - line, startY - line, startX - line, startY + line + 1)
            }
            /*		g.drawLine( 5, 4, 5, 11 );
        g.drawLine( 6, 5, 6, 10 );
		g.drawLine( 7, 6, 7, 9 );
		g.drawLine( 8, 7, 8, 8 );*/if (isEnabled) {
                g.color = highlightColor
                if (!isPressed) {
                    g.drawLine(1, 1, width - 1, 1)
                    g.drawLine(1, 1, 1, height - 3)
                }
                g.drawLine(1, height - 1, width - 1, height - 1)
                g.color = darkShadowColor
                g.drawLine(0, 0, width - 1, 0)
                g.drawLine(0, 0, 0, height - 2)
                g.drawLine(2, height - 2, width - 1, height - 2)
            } else {
                drawDisabledBorder(g, 0, 0, width + 1, height)
            }
            if (!isFreeStanding) {
                height -= 2
                width -= 1
            }
        } else if (direction == RangeSliderUI.Companion.WEST) {
            if (!isFreeStanding) {
                height += 2
                width += 1
                g.translate(-1, 0)
            }

            // Draw the arrow
            g.color = arrowColor
            val startX = (w + 1 - arrowHeight) / 2 - 1
            val startY = h / 2 - 1
            for (line in 0 until arrowHeight) {
                g.drawLine(startX + line, startY - line, startX + line, startY + line + 1)
            }

            /*	g.drawLine( 6, 7, 6, 8 );
                g.drawLine( 7, 6, 7, 9 );
                g.drawLine( 8, 5, 8, 10 );
                g.drawLine( 9, 4, 9, 11 );*/if (isEnabled) {
                g.color = highlightColor
                if (!isPressed) {
                    g.drawLine(0, 1, width - 3, 1)
                    g.drawLine(0, 1, 0, height - 3)
                }
                g.drawLine(width - 1, 1, width - 1, height - 1)
                g.drawLine(0, height - 1, width - 1, height - 1)
                g.color = darkShadowColor
                g.drawLine(0, 0, width - 2, 0)
                g.drawLine(width - 2, 2, width - 2, height - 2)
                g.drawLine(0, height - 2, width - 2, height - 2)
            } else {
                drawDisabledBorder(g, -1, 0, width + 1, height)
            }
            if (!isFreeStanding) {
                height -= 2
                width -= 1
                g.translate(1, 0)
            }
        }
    }

    companion object {
        private val rangeSliderUI = MetalRangeSliderUI()
        fun createUI(c: JComponent?): ComponentUI {
            return rangeSliderUI
        }

        private lateinit var shadowColor: Color
        private lateinit var highlightColor: Color
        private lateinit var darkShadowColor: Color
        private lateinit var thumbColor: Color
        private lateinit var thumbShadow: Color
        private lateinit var thumbHighlightColor: Color
        private lateinit var thumbLightShadowColor: Color
        private lateinit var thumbDarkShadowColor: Color
        private lateinit var trackColor: Color
        private val trackHighlightColor: Color? = null
        private fun drawFlush3DBorder(g: Graphics, r: Rectangle) {
            drawFlush3DBorder(g, r.x, r.y, r.width, r.height)
        }

        /**
         * This draws the "Flush 3D Border" which is used throughout the Metal L&F
         */
        private fun drawFlush3DBorder(g: Graphics, x: Int, y: Int, w: Int, h: Int) {
            g.translate(x, y)
            g.color = MetalLookAndFeel.getControlDarkShadow()
            g.drawRect(0, 0, w - 2, h - 2)
            g.color = MetalLookAndFeel.getControlHighlight()
            g.drawRect(1, 1, w - 2, h - 2)
            g.color = MetalLookAndFeel.getControl()
            g.drawLine(0, h - 1, 1, h - 2)
            g.drawLine(w - 1, 0, w - 2, 1)
            g.translate(-x, -y)
        }

        /**
         * This draws a variant "Flush 3D Border" It is used for things like pressed buttons.
         */
        private fun drawPressed3DBorder(g: Graphics, r: Rectangle) {
            drawPressed3DBorder(g, r.x, r.y, r.width, r.height)
        }

        private fun drawDisabledBorder(g: Graphics, x: Int, y: Int, w: Int, h: Int) {
            g.translate(x, y)
            g.color = MetalLookAndFeel.getControlShadow()
            g.drawRect(0, 0, w - 1, h - 1)
        }

        /**
         * This draws a variant "Flush 3D Border" It is used for things like pressed buttons.
         */
        private fun drawPressed3DBorder(g: Graphics, x: Int, y: Int, w: Int, h: Int) {
            g.translate(x, y)
            drawFlush3DBorder(g, 0, 0, w, h)
            g.color = MetalLookAndFeel.getControlShadow()
            g.drawLine(1, 1, 1, h - 2)
            g.drawLine(1, 1, w - 2, 1)
            g.translate(-x, -y)
        }

        /**
         * This draws a variant "Flush 3D Border" It is used for things like active toggle buttons. This is used rarely.
         */
        private fun drawDark3DBorder(g: Graphics, r: Rectangle) {
            drawDark3DBorder(g, r.x, r.y, r.width, r.height)
        }

        /**
         * This draws a variant "Flush 3D Border" It is used for things like active toggle buttons. This is used rarely.
         */
        private fun drawDark3DBorder(g: Graphics, x: Int, y: Int, w: Int, h: Int) {
            g.translate(x, y)
            drawFlush3DBorder(g, 0, 0, w, h)
            g.color = MetalLookAndFeel.getControl()
            g.drawLine(1, 1, 1, h - 2)
            g.drawLine(1, 1, w - 2, 1)
            g.color = MetalLookAndFeel.getControlShadow()
            g.drawLine(1, h - 2, 1, h - 2)
            g.drawLine(w - 2, 1, w - 2, 1)
            g.translate(-x, -y)
        }

        private fun drawButtonBorder(g: Graphics, x: Int, y: Int, w: Int, h: Int, active: Boolean) {
            if (active) {
                drawActiveButtonBorder(g, x, y, w, h)
            } else {
                drawFlush3DBorder(g, x, y, w, h)
            }
        }

        private fun drawActiveButtonBorder(g: Graphics, x: Int, y: Int, w: Int, h: Int) {
            drawFlush3DBorder(g, x, y, w, h)
            g.color = MetalLookAndFeel.getPrimaryControl()
            g.drawLine(x + 1, y + 1, x + 1, h - 3)
            g.drawLine(x + 1, y + 1, w - 3, x + 1)
            g.color = MetalLookAndFeel.getPrimaryControlDarkShadow()
            g.drawLine(x + 2, h - 2, w - 2, h - 2)
            g.drawLine(w - 2, y + 2, w - 2, h - 2)
        }

        private fun drawDefaultButtonBorder(g: Graphics, x: Int, y: Int, w: Int, h: Int, active: Boolean) {
            drawButtonBorder(g, x + 1, y + 1, w - 1, h - 1, active)
            g.color = MetalLookAndFeel.getControlDarkShadow()
            g.drawRect(x, y, w - 3, h - 3)
            g.drawLine(w - 2, 0, w - 2, 0)
            g.drawLine(0, h - 2, 0, h - 2)
        }

        /*
     * Convenience function for determining ComponentOrientation.  Helps us
     * avoid having Munge directives throughout the code.
     */
        private fun isLeftToRight(c: Component): Boolean {
            return c.componentOrientation.isLeftToRight
        }
    }

    init {
        shadowColor = UIManager.getColor("ScrollBar.shadow")
        highlightColor = UIManager.getColor("ScrollBar.highlight")
        darkShadowColor = UIManager.getColor("ScrollBar.darkShadow")
        Companion.thumbColor = UIManager.getColor("ScrollBar.thumb")
        thumbShadow = UIManager.getColor("ScrollBar.thumbShadow")
        thumbHighlightColor = UIManager.getColor("ScrollBar.thumbHighlight")
        thumbLightShadowColor = UIManager.getColor("ScrollBar.thumbLightShadow")
        thumbDarkShadowColor = UIManager.getColor("ScrollBar.thumbDarkShadow")
        trackColor = UIManager.getColor("ScrollBar.track")
        bumps = MetalBumps(10, 10, thumbHighlightColor, thumbShadow, Companion.thumbColor)
    }
}