/*
 * Copyright (c) 2010 Macrofocus GmbH. All Rights Reserved.
 */
package org.mkui.colormap

import com.macrofocus.common.interval.Interval
import org.mkui.color.CPColor
import org.mkui.palette.Palette

/**
 * This interface defines how values can be mapped to colors.
 */
interface ColorMap {
    val assignedValues: Set<Any?>?
    val brightness: Int
    val saturation: Int
    val alpha: Float

    enum class Matching {
        EXACT, CLOSEST
    }

    enum class Assignments {
        STATIC, DYNAMIC
    }

    fun getColor(value: Any?): CPColor?
    val interval: Interval?
    val palette: Palette?

    /**
     * Get color to be used for null values.
     *
     * @return the color for null values
     */
    val nullColor: CPColor?

    /**
     * Get color to be used for high out-of-range values
     *
     * @return the high out-of-range color
     */
    val overflowColor: CPColor?

    /**
     * Get color to be used for low out-of-range values
     *
     * @return the low out-of-range color
     */
    val underflowColor: CPColor?
    val isOverflowColorSet: Boolean
    val isUnderflowColorSet: Boolean
    val isInverted: Boolean
    val colorCount: Int

    /**
     * Add a listener to the list that's notified each time a change to the colormap occurs.
     *
     * @param listener the ColorMapListener
     */
    fun addColorMapListener(listener: ColorMapListener)

    /**
     * Add a listener to the list that's notified each time a change to the colormap occurs. The listener will
     * automatically be disposed of should no other object have a reference to it.
     *
     * @param listener the ColorMapListener
     */
    fun addWeakColorMapListener(listener: ColorMapListener)

    /**
     * Remove a listener to the list that's notified each time a change to the colormap occurs.
     *
     * @param listener the ColorMapListener
     */
    fun removeColorMapListener(listener: ColorMapListener)

    companion object {
        const val PROPERTY_INTERVAL = "inverval"
        const val PROPERTY_PALETTE = "palette"
        const val PROPERTY_MATCHING = "matching"
        const val PROPERTY_ASSIGNMENTS = "assignments"
        const val PROPERTY_NULLCOLOR = "missingValuesColor"
        const val PROPERTY_UNDERCOLOR = "underflowColor"
        const val PROPERTY_OVERCOLOR = "overflowColor"
        const val PROPERTY_INVERTED = "inverted"
        const val PROPERTY_COLORCOUNT = "numberOfSteps"
        const val PROPERTY_BRIGHTNESS = "brightness"
        const val PROPERTY_SATURATION = "saturation"
        const val PROPERTY_ALPHA = "alpha"
    }
}