/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.mkui.geom

import kotlin.math.ceil
import kotlin.math.floor
import kotlin.math.max
import kotlin.math.min

/**
 * Created by luc on 10/07/15.
 */
abstract class AbstractRectangle2D : Rectangle2D {
    override val bounds: Rectangle?
        get() {
            val width: Double = width
            val height: Double = height
            if (width < 0 || height < 0) {
                return Rectangle()
            }
            val x: Double = x
            val y: Double = y
            val x1: Double = floor(x)
            val y1: Double = floor(y)
            val x2: Double = ceil(x + width)
            val y2: Double = ceil(y + height)
            return Rectangle(
                x1.toInt(), y1.toInt(),
                (x2 - x1).toInt(), (y2 - y1).toInt()
            )
        }
    override val bounds2D: Rectangle2D
        get() = this

    override operator fun contains(point: Point2D): Boolean {
        val x: Double = point.x
        val y: Double = point.y
        val x0: Double = this.x
        val y0: Double = this.y
        return x >= x0 && y >= y0 && x < x0 + width && y < y0 + height
    }

    override val centerX: Double
        get() = x + width / 2.0

    override fun intersects(rect: Rectangle2D): Boolean {
        val x: Double = rect.x
        val y: Double = rect.y
        val w: Double = rect.width
        val h: Double = rect.height
        if (w <= 0 || h <= 0) {
            return false
        }
        val x0: Double = this.x
        val y0: Double = this.y
        return x + w > x0 && y + h > y0 && x < x0 + width && y < y0 + height
    }

    override val centerY: Double
        get() = y + height / 2.0

    /**
     * Returns an iteration object that defines the boundary of this
     * `Rectangle2D`.
     * The iterator for this class is multi-threaded safe, which means
     * that this `Rectangle2D` class guarantees that
     * modifications to the geometry of this `Rectangle2D`
     * object do not affect any iterations of that geometry that
     * are already in process.
     *
     * @param at an optional `AffineTransform` to be applied to
     * the coordinates as they are returned in the iteration, or
     * `null` if untransformed coordinates are desired
     *
     * @return the `PathIterator` object that returns the
     * geometry of the outline of this
     * `Rectangle2D`, one segment at a time.
     *
     * @since 1.2
     */
    override val pathIterator: org.mkui.geom.PathIterator
        get() = getPathIterator(null)
    override val minX: Double
        get() = x

    override fun getPathIterator(at: AffineTransform?): PathIterator {
        return RectIterator(this, at)
    }

    override val minY: Double
        get() = y

    /**
     * Returns an iteration object that defines the boundary of the
     * flattened `Rectangle2D`.  Since rectangles are already
     * flat, the `flatness` parameter is ignored.
     * The iterator for this class is multi-threaded safe, which means
     * that this `Rectangle2D` class guarantees that
     * modifications to the geometry of this `Rectangle2D`
     * object do not affect any iterations of that geometry that
     * are already in process.
     *
     * @param at       an optional `AffineTransform` to be applied to
     * the coordinates as they are returned in the iteration, or
     * `null` if untransformed coordinates are desired
     * @param flatness the maximum distance that the line segments used to
     * approximate the curved segments are allowed to deviate from any
     * point on the original curve.  Since rectangles are already flat,
     * the `flatness` parameter is ignored.
     *
     * @return the `PathIterator` object that returns the
     * geometry of the outline of this
     * `Rectangle2D`, one segment at a time.
     *
     * @since 1.2
     */
    override fun getFlattenPathIterator(flatness: Double): PathIterator {
        return RectIterator(this, null)
    }

    override val maxX: Double
        get() = x + width
    override val maxY: Double
        get() = y + height

    override fun createIntersection(r: Rectangle2D): Rectangle2D {
        val x1: Double = max(minX, r.minX)
        val y1: Double = max(minY, r.minY)
        val x2: Double = min(maxX, r.maxX)
        val y2: Double = min(maxY, r.maxY)
        return Rectangle2D.Double(x1, y1, x2 - x1, y2 - y1)
    }

    override fun createUnion(r: Rectangle2D): Rectangle2D {
        val x1: Double = min(minX, r.minX)
        val y1: Double = min(minY, r.minY)
        val x2: Double = max(maxX, r.maxX)
        val y2: Double = max(maxY, r.maxY)
        return Rectangle2D.Double(x1, y1, x2 - x1, y2 - y1)
    }

    /**
     * Tests if the specified line segment intersects the interior of this
     * `Rectangle2D`.
     *
     * @param x1 the X coordinate of the start point of the specified
     * line segment
     * @param y1 the Y coordinate of the start point of the specified
     * line segment
     * @param x2 the X coordinate of the end point of the specified
     * line segment
     * @param y2 the Y coordinate of the end point of the specified
     * line segment
     *
     * @return `true` if the specified line segment intersects
     * the interior of this `Rectangle2D`; `false`
     * otherwise.
     *
     * @since 1.2
     */
    override fun intersectsLine(x1: Double, y1: Double, x2: Double, y2: Double): Boolean {
        var x1 = x1
        var y1 = y1
        var out2: Int
        if (outcode(x2, y2).also { out2 = it } == 0) {
            return true
        }
        var out1: Int
        while (outcode(x1, y1).also { out1 = it } != 0) {
            if (out1 and out2 != 0) {
                return false
            }
            if (out1 and (OUT_LEFT or OUT_RIGHT) != 0) {
                var x: Double = x
                if (out1 and OUT_RIGHT != 0) {
                    x += width
                }
                y1 = y1 + (x - x1) * (y2 - y1) / (x2 - x1)
                x1 = x
            } else {
                var y: Double = y
                if (out1 and OUT_BOTTOM != 0) {
                    y += height
                }
                x1 = x1 + (y - y1) * (x2 - x1) / (y2 - y1)
                y1 = y
            }
        }
        return true
    }

    /**
     * {@inheritDoc}
     *
     * @since 1.2
     */
    fun outcode(x: Double, y: Double): Int {
        /*
         * Note on casts to double below.  If the arithmetic of
         * x+w or y+h is done in float, then some bits may be
         * lost if the binary exponents of x/y and w/h are not
         * similar.  By converting to double before the addition
         * we force the addition to be carried out in double to
         * avoid rounding error in the comparison.
         *
         * See bug 4320890 for problems that this inaccuracy causes.
         */
        var out = 0
        if (this.width <= 0) {
            out = out or (OUT_LEFT or OUT_RIGHT)
        } else if (x < this.x) {
            out = out or OUT_LEFT
        } else if (x > this.x + this.width) {
            out = out or OUT_RIGHT
        }
        if (this.height <= 0) {
            out = out or (OUT_TOP or OUT_BOTTOM)
        } else if (y < this.y) {
            out = out or OUT_TOP
        } else if (y > this.y + this.height) {
            out = out or OUT_BOTTOM
        }
        return out
    }

    companion object {
        /**
         * The bitmask that indicates that a point lies to the left of
         * this `Rectangle2D`.
         *
         * @since 1.2
         */
        private const val OUT_LEFT = 1

        /**
         * The bitmask that indicates that a point lies above
         * this `Rectangle2D`.
         *
         * @since 1.2
         */
        private const val OUT_TOP = 2

        /**
         * The bitmask that indicates that a point lies to the right of
         * this `Rectangle2D`.
         *
         * @since 1.2
         */
        private const val OUT_RIGHT = 4

        /**
         * The bitmask that indicates that a point lies below
         * this `Rectangle2D`.
         *
         * @since 1.2
         */
        private const val OUT_BOTTOM = 8
    }
}