/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.mkui.geom

interface Ellipse2D : Shape {
    val x: kotlin.Double
    val y: kotlin.Double
    val width: kotlin.Double
    val height: kotlin.Double
    val minX: kotlin.Double
    val minY: kotlin.Double
    val maxX: kotlin.Double
    val maxY: kotlin.Double
    val centerX: kotlin.Double
    val centerY: kotlin.Double

    class Double(
        override val x: kotlin.Double,
        override val y: kotlin.Double,
        override val width: kotlin.Double,
        override val height: kotlin.Double
    ) : Ellipse2D {
        override val bounds2D: Rectangle2D
            get() = Rectangle2D.Double(x, y, width, height)

        override operator fun contains(point: Point2D): Boolean {
            val x: kotlin.Double = point.x
            val y: kotlin.Double = point.y
            val x0 = x
            val y0 = y
            return x >= x0 && y >= y0 && x < x0 + width && y < y0 + height
        }

        override fun intersects(rect: Rectangle2D): Boolean {
            val x: kotlin.Double = rect.x
            val y: kotlin.Double = rect.y
            val w: kotlin.Double = rect.width
            val h: kotlin.Double = rect.height
            if (w <= 0.0 || h <= 0.0) {
                return false
            }
            // Normalize the rectangular coordinates compared to the ellipse
            // having a center at 0,0 and a radius of 0.5.
            val ellw = width
            if (ellw <= 0.0) {
                return false
            }
            val normx0 = (x - x) / ellw - 0.5
            val ellh = height
            if (ellh <= 0.0) {
                return false
            }
            val normy0 = (y - y) / ellh - 0.5
            // find nearest x (left edge, right edge, 0.0)
            // find nearest y (top edge, bottom edge, 0.0)
            // if nearest x,y is inside circle of radius 0.5, then intersects
            val nearx: kotlin.Double
            val normx1 = normx0 + w / ellw
            nearx = if (normx0 > 0.0) {
                // center to left of X extents
                normx0
            } else if (normx1 < 0.0) {
                // center to right of X extents
                normx1
            } else {
                0.0
            }
            val neary: kotlin.Double
            val normy1 = normy0 + h / ellh
            neary = if (normy0 > 0.0) {
                // center above Y extents
                normy0
            } else if (normy1 < 0.0) {
                // center below Y extents
                normy1
            } else {
                0.0
            }
            return nearx * nearx + neary * neary < 0.25
        }

        override val minX : kotlin.Double
            get() = x

        override val minY : kotlin.Double
            get() = y

        override val maxX : kotlin.Double
            get() = x + width

        override val maxY : kotlin.Double
            get() = y + height

        override val centerX : kotlin.Double
            get() = x + width / 2.0

        override val centerY : kotlin.Double
            get() = y + height / 2.0

        /**
         * Returns an iteration object that defines the boundary of this
         * `Ellipse2D`.
         * The iterator for this class is multi-threaded safe, which means
         * that this `Ellipse2D` class guarantees that
         * modifications to the geometry of this `Ellipse2D`
         * object do not affect any iterations of that geometry that
         * are already in process.
         *
         * @param at an optional `AffineTransform` to be applied to
         * the coordinates as they are returned in the iteration, or
         * `null` if untransformed coordinates are desired
         *
         * @return the `PathIterator` object that returns the
         * geometry of the outline of this `Ellipse2D`,
         * one segment at a time.
         *
         * @since 1.2
         */
        override val pathIterator: PathIterator
            get() = getPathIterator(null)

        override fun getPathIterator(at: AffineTransform?): PathIterator {
            return EllipseIterator(this, at)
        }

        /**
         * Returns an iterator object that iterates along the
         * `Shape` object's boundary and provides access to a
         * flattened view of the outline of the `Shape`
         * object's geometry.
         *
         *
         * Only SEG_MOVETO, SEG_LINETO, and SEG_CLOSE point types will
         * be returned by the iterator.
         *
         *
         * The amount of subdivision of the curved segments is controlled
         * by the `flatness` parameter, which specifies the
         * maximum distance that any point on the unflattened transformed
         * curve can deviate from the returned flattened path segments.
         * An optional [AffineTransform] can
         * be specified so that the coordinates returned in the iteration are
         * transformed accordingly.
         *
         * @param at       an optional `AffineTransform` to be applied to the
         * coordinates as they are returned in the iteration,
         * or `null` if untransformed coordinates are desired.
         * @param flatness the maximum distance that the line segments used to
         * approximate the curved segments are allowed to deviate
         * from any point on the original curve
         *
         * @return a `PathIterator` object that provides access to
         * the `Shape` object's flattened geometry.
         *
         * @since 1.2
         */
        override fun getFlattenPathIterator(flatness: kotlin.Double): PathIterator {
            return FlatteningPathIterator(pathIterator, flatness)
        }

        override fun toString(): String {
            return "Ellipse2D.Double{" +
                    "x=" + x +
                    ", y=" + y +
                    ", width=" + width +
                    ", height=" + height +
                    '}'
        }
    }
}