/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.mkui.geom

class Path : Geometry {
    /**
     * Returns the winding rule used to determine the interior of this path.
     *
     * @return the winding rule, i.e. one of [.WIND_EVEN_ODD] or
     * [.WIND_NON_ZERO]
     */
    var windingRule = WIND_NON_ZERO
    private var segments: MutableList<Segment> = ArrayList<Segment>()

    /**
     * Creates a new empty path with a default winding rule of
     * [.WIND_NON_ZERO].
     */
    constructor() {}

    /**
     * Creates a path from the given segments, using the default winding rule
     * [.WIND_NON_ZERO].
     *
     * @param segments The segments to initialize the path with
     */
    constructor(vararg segments: Segment) : this(WIND_NON_ZERO, *segments) {}

    /**
     * Creates a path from the given segments, using the given winding rule.
     *
     * @param windingRule the winding rule to use; one of [.WIND_EVEN_ODD] or
     * [.WIND_NON_ZERO]
     * @param segments    The segments to initialize the path with
     */
    constructor(windingRule: Int, vararg segments: Segment) : this(windingRule) {
        for (s in segments) {
            this.segments.add(s)
        }
    }

    /**
     * Creates a new empty path with given winding rule.
     *
     * @param windingRule the winding rule to use; one of [.WIND_EVEN_ODD] or
     * [.WIND_NON_ZERO]
     */
    constructor(windingRule: Int) {
        this.windingRule = windingRule
    }

    /**
     * Closes the current sub-path by drawing a straight line (line-to) to the
     * location of the last move to.
     *
     * @return `this` for convenience
     */
    fun close(): Path {
        segments.add(Segment(Segment.CLOSE))
        return this
    }

    /**
     * Adds a cubic Bezier curve segment from the current position to the
     * specified end position, using the two provided control points as Bezier
     * control points.
     *
     * @param control1X The x-coordinate of the first Bezier control point
     * @param control1Y The y-coordinate of the first Bezier control point
     * @param control2X The x-coordinate of the second Bezier control point
     * @param control2Y The y-coordinate of the second Bezier control point
     * @param x         The x-coordinate of the desired target point
     * @param y         The y-coordinate of the desired target point
     *
     * @return `this` for convenience
     */
    fun cubicTo(
        control1X: Double, control1Y: Double,
        control2X: Double, control2Y: Double, x: Double, y: Double
    ): Path {
        segments.add(
            Segment(
                Segment.CUBIC_TO, Point2D.Double(
                    control1X,
                    control1Y
                ), Point2D.Double(control2X, control2Y), Point2D.Double(x, y)
            )
        )
        return this
    }

    /**
     * Adds a straight line segment from the current position to the specified
     * end position.
     *
     * @param x The x-coordinate of the desired target point
     * @param y The y-coordinate of the desired target point
     *
     * @return `this` for convenience
     */
    fun lineTo(x: Double, y: Double): Path {
        segments.add(Segment(Segment.LINE_TO, Point2D.Double(x, y)))
        return this
    }

    /**
     * Changes the current position without adding a new segment.
     *
     * @param x The x-coordinate of the desired target point
     * @param y The y-coordinate of the desired target point
     *
     * @return `this` for convenience
     */
    fun moveTo(x: Double, y: Double): Path {
        segments.add(Segment(Segment.MOVE_TO, Point2D.Double(x, y)))
        return this
    }

    /**
     * Adds a quadratic curve segment from the current position to the specified
     * end position, using the provided control point as a parametric control
     * point.
     *
     * @param controlX The x-coordinate of the control point
     * @param controlY The y-coordinate of the control point
     * @param x        The x-coordinate of the desired target point
     * @param y        The y-coordinate of the desired target point
     *
     * @return `this` for convenience
     */
    fun quadTo(
        controlX: Double, controlY: Double, x: Double,
        y: Double
    ): Path {
        segments.add(
            Segment(
                Segment.QUAD_TO,
                Point2D.Double(controlX, controlY), Point2D.Double(x, y)
            )
        )
        return this
    }

    fun getSegments(): Array<Segment> {
        return segments.toTypedArray()
    }

    fun setSegments(segments: Array<Segment>) {
        this.segments = segments.toMutableList()
    }

    override operator fun contains(point: Point2D): Boolean {
        val x: Double = point.x
        val y: Double = point.y
        return if (x * 0.0 + y * 0.0 == 0.0) {
            /* N * 0.0 is 0.0 only if N is finite.
             * Here we know that both x and y are finite.
             */
            if (segments.size < 2) {
                return false
            }
            val mask = if (windingRule == WIND_NON_ZERO) -1 else 1
            pointCrossings(x, y) and mask != 0
        } else {
            /* Either x or y was infinite or NaN.
             * A NaN always produces a negative response to any test
             * and Infinity values cannot be "inside" any path so
             * they should return false as well.
             */
            false
        }
    }

    fun pointCrossings(px: Double, py: Double): Int {
        var mov: Point2D
        mov = segments[0].getPoints()!!.get(0)
        var cur: Point2D? = mov
        var end: Point2D? = null
        var crossings = 0
        var ci = 1
        for (i in 1 until segments.size) {
            val segment: Segment = segments[i]
            when (segment.type) {
                Segment.MOVE_TO -> {
                    if (cur!!.y !== mov.y) {
                        crossings += pointCrossingsForLine(
                            px, py,
                            cur!!.x, cur!!.y,
                            mov.x, mov.y
                        )
                    }
                    run {
                        cur = segment.getPoints()!!.get(ci++)
                        mov = cur!!
                    }
                }
                Segment.LINE_TO -> {
                    crossings += pointCrossingsForLine(
                        px, py,
                        cur!!.x, cur!!.y,
                        segment.getPoints()!!.get(ci++).also { end = it }.x,
                        segment.getPoints()!!.get(ci).y
                    )
                    cur = end
                }
                Segment.QUAD_TO -> {
                    crossings += pointCrossingsForQuad(
                        px, py,
                        cur!!.x, cur!!.y,
                        segment.getPoints()!!.get(ci++).x,
                        segment.getPoints()!!.get(ci).y,
                        segment.getPoints()!!.get(ci++).also { end = it }.x,
                        segment.getPoints()!!.get(ci).y,
                        0
                    )
                    cur = end
                }
                Segment.CUBIC_TO -> {
                    crossings += pointCrossingsForCubic(
                        px, py,
                        cur!!.x, cur!!.y,
                        segment.getPoints()!!.get(ci++).x,
                        segment.getPoints()!!.get(ci).y,
                        segment.getPoints()!!.get(ci++).x,
                        segment.getPoints()!!.get(ci).y,
                        segment.getPoints()!!.get(ci++).also { end = it }.x,
                        segment.getPoints()!!.get(ci).y,
                        0
                    )
                    cur = end
                }
                Segment.CLOSE -> {
                    if (cur!!.y !== mov.y) {
                        crossings += pointCrossingsForLine(
                            px, py,
                            cur!!.x, cur!!.y,
                            mov.x, mov.y
                        )
                    }
                    cur = mov
                }
            }
        }
        if (cur!!.y !== mov.y) {
            crossings += pointCrossingsForLine(
                px, py,
                cur!!.x, cur!!.y,
                mov.x, mov.y
            )
        }
        return crossings
    }

    override fun toString(): String {
        return "Path{" +
                "windingRule=" + windingRule +
                ", segments=" + segments +
                '}'
    }

    companion object {
        /**
         * Winding rule for determining the interior of the [Path]. Indicates
         * that a [Point2D] is regarded to lie inside the [Path], if any
         * ray starting in that [Point2D] and pointing to infinity crosses the
         * [Segment]s of the [Path] an odd number of times.
         */
        const val WIND_EVEN_ODD = 0

        /**
         * Winding rule for determining the interior of the [Path]. Indicates
         * that a [Point2D] is regarded to lie inside the [Path], if any
         * ray starting from that [Point2D] and pointing to infinity is crossed
         * by [Path] [Segment]s a different number of times in the
         * counter-clockwise direction than in the clockwise direction.
         */
        const val WIND_NON_ZERO = 1
        private const val serialVersionUID = 1L

        /**
         * Calculates the number of times the line from (x0,y0) to (x1,y1)
         * crosses the ray extending to the right from (px,py).
         * If the point lies on the line, then no crossings are recorded.
         * +1 is returned for a crossing where the Y coordinate is increasing
         * -1 is returned for a crossing where the Y coordinate is decreasing
         */
        fun pointCrossingsForLine(
            px: Double, py: Double,
            x0: Double, y0: Double,
            x1: Double, y1: Double
        ): Int {
            if (py < y0 && py < y1) return 0
            if (py >= y0 && py >= y1) return 0
            // assert(y0 != y1);
            if (px >= x0 && px >= x1) return 0
            if (px < x0 && px < x1) return if (y0 < y1) 1 else -1
            val xintercept = x0 + (py - y0) * (x1 - x0) / (y1 - y0)
            if (px >= xintercept) return 0
            return if (y0 < y1) 1 else -1
        }

        /**
         * Calculates the number of times the quad from (x0,y0) to (x1,y1)
         * crosses the ray extending to the right from (px,py).
         * If the point lies on a part of the curve,
         * then no crossings are counted for that intersection.
         * the level parameter should be 0 at the top-level call and will count
         * up for each recursion level to prevent infinite recursion
         * +1 is added for each crossing where the Y coordinate is increasing
         * -1 is added for each crossing where the Y coordinate is decreasing
         */
        fun pointCrossingsForQuad(
            px: Double, py: Double,
            x0: Double, y0: Double,
            xc: Double, yc: Double,
            x1: Double, y1: Double, level: Int
        ): Int {
            var xc = xc
            var yc = yc
            if (py < y0 && py < yc && py < y1) return 0
            if (py >= y0 && py >= yc && py >= y1) return 0
            // Note y0 could equal y1...
            if (px >= x0 && px >= xc && px >= x1) return 0
            if (px < x0 && px < xc && px < x1) {
                if (py >= y0) {
                    if (py < y1) return 1
                } else {
                    // py < y0
                    if (py >= y1) return -1
                }
                // py outside of y01 range, and/or y0==y1
                return 0
            }
            // double precision only has 52 bits of mantissa
            if (level > 52) return pointCrossingsForLine(px, py, x0, y0, x1, y1)
            val x0c = (x0 + xc) / 2
            val y0c = (y0 + yc) / 2
            val xc1 = (xc + x1) / 2
            val yc1 = (yc + y1) / 2
            xc = (x0c + xc1) / 2
            yc = (y0c + yc1) / 2
            return if (xc.isNaN() || yc.isNaN()) {
                // [xy]c are NaN if any of [xy]0c or [xy]c1 are NaN
                // [xy]0c or [xy]c1 are NaN if any of [xy][0c1] are NaN
                // These values are also NaN if opposing infinities are added
                0
            } else pointCrossingsForQuad(
                px, py,
                x0, y0, x0c, y0c, xc, yc,
                level + 1
            ) +
                    pointCrossingsForQuad(
                        px, py,
                        xc, yc, xc1, yc1, x1, y1,
                        level + 1
                    )
        }

        /**
         * Calculates the number of times the cubic from (x0,y0) to (x1,y1)
         * crosses the ray extending to the right from (px,py).
         * If the point lies on a part of the curve,
         * then no crossings are counted for that intersection.
         * the level parameter should be 0 at the top-level call and will count
         * up for each recursion level to prevent infinite recursion
         * +1 is added for each crossing where the Y coordinate is increasing
         * -1 is added for each crossing where the Y coordinate is decreasing
         */
        fun pointCrossingsForCubic(
            px: Double, py: Double,
            x0: Double, y0: Double,
            xc0: Double, yc0: Double,
            xc1: Double, yc1: Double,
            x1: Double, y1: Double, level: Int
        ): Int {
            var xc0 = xc0
            var yc0 = yc0
            var xc1 = xc1
            var yc1 = yc1
            if (py < y0 && py < yc0 && py < yc1 && py < y1) return 0
            if (py >= y0 && py >= yc0 && py >= yc1 && py >= y1) return 0
            // Note y0 could equal yc0...
            if (px >= x0 && px >= xc0 && px >= xc1 && px >= x1) return 0
            if (px < x0 && px < xc0 && px < xc1 && px < x1) {
                if (py >= y0) {
                    if (py < y1) return 1
                } else {
                    // py < y0
                    if (py >= y1) return -1
                }
                // py outside of y01 range, and/or y0==yc0
                return 0
            }
            // double precision only has 52 bits of mantissa
            if (level > 52) return pointCrossingsForLine(px, py, x0, y0, x1, y1)
            var xmid = (xc0 + xc1) / 2
            var ymid = (yc0 + yc1) / 2
            xc0 = (x0 + xc0) / 2
            yc0 = (y0 + yc0) / 2
            xc1 = (xc1 + x1) / 2
            yc1 = (yc1 + y1) / 2
            val xc0m = (xc0 + xmid) / 2
            val yc0m = (yc0 + ymid) / 2
            val xmc1 = (xmid + xc1) / 2
            val ymc1 = (ymid + yc1) / 2
            xmid = (xc0m + xmc1) / 2
            ymid = (yc0m + ymc1) / 2
            return if (xmid.isNaN() || ymid.isNaN()) {
                // [xy]mid are NaN if any of [xy]c0m or [xy]mc1 are NaN
                // [xy]c0m or [xy]mc1 are NaN if any of [xy][c][01] are NaN
                // These values are also NaN if opposing infinities are added
                0
            } else pointCrossingsForCubic(
                px, py,
                x0, y0, xc0, yc0,
                xc0m, yc0m, xmid, ymid, level + 1
            ) +
                    pointCrossingsForCubic(
                        px, py,
                        xmid, ymid, xmc1, ymc1,
                        xc1, yc1, x1, y1, level + 1
                    )
        }
    }
}