/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.mkui.geom

import org.mkui.geom.PathIterator.Companion.SEG_MOVETO
import org.mkui.geom.PathIterator.Companion.SEG_QUADTO
import org.mkui.geom.PathIterator.Companion.WIND_NON_ZERO

/**
 * Created by luc on 05/07/16.
 */
class QuadIterator internal constructor(q: QuadCurve2D, at: AffineTransform?) : PathIterator {
    val quad: QuadCurve2D
    val affine: AffineTransform?
    var index = 0

    /**
     * Return the winding rule for determining the insideness of the
     * path.
     *
     * @see .WIND_EVEN_ODD
     *
     * @see .WIND_NON_ZERO
     */
    override val windingRule: Int
        get() = WIND_NON_ZERO

    /**
     * Tests if there are more points to read.
     *
     * @return true if there are more points to read
     */
    override val isDone: Boolean
        get() = index > 1

    /**
     * Moves the iterator to the next segment of the path forwards
     * along the primary direction of traversal as long as there are
     * more points in that direction.
     */
    override operator fun next() {
        index++
    }

    /**
     * Returns the coordinates and type of the current path segment in
     * the iteration.
     * The return value is the path segment type:
     * SEG_MOVETO, SEG_LINETO, SEG_QUADTO, SEG_CUBICTO, or SEG_CLOSE.
     * A float array of length 6 must be passed in and may be used to
     * store the coordinates of the point(s).
     * Each point is stored as a pair of float x,y coordinates.
     * SEG_MOVETO and SEG_LINETO types will return one point,
     * SEG_QUADTO will return two points,
     * SEG_CUBICTO will return 3 points
     * and SEG_CLOSE will not return any points.
     *
     * @see .SEG_MOVETO
     *
     * @see .SEG_LINETO
     *
     * @see .SEG_QUADTO
     *
     * @see .SEG_CUBICTO
     *
     * @see .SEG_CLOSE
     */
    override fun currentSegment(coords: FloatArray): Int {
        if (isDone) {
            throw NoSuchElementException("quad iterator iterator out of bounds")
        }
        val type: Int
        if (index == 0) {
            coords[0] = quad.x1.toFloat()
            coords[1] = quad.y1.toFloat()
            type = SEG_MOVETO
        } else {
            coords[0] = quad.ctrlX.toFloat()
            coords[1] = quad.ctrlY.toFloat()
            coords[2] = quad.x2.toFloat()
            coords[3] = quad.y2.toFloat()
            type = SEG_QUADTO
        }
        if (affine != null) {
            affine.transform(coords, 0, coords, 0, if (index == 0) 1 else 2)
        }
        return type
    }

    /**
     * Returns the coordinates and type of the current path segment in
     * the iteration.
     * The return value is the path segment type:
     * SEG_MOVETO, SEG_LINETO, SEG_QUADTO, SEG_CUBICTO, or SEG_CLOSE.
     * A double array of length 6 must be passed in and may be used to
     * store the coordinates of the point(s).
     * Each point is stored as a pair of double x,y coordinates.
     * SEG_MOVETO and SEG_LINETO types will return one point,
     * SEG_QUADTO will return two points,
     * SEG_CUBICTO will return 3 points
     * and SEG_CLOSE will not return any points.
     *
     * @see .SEG_MOVETO
     *
     * @see .SEG_LINETO
     *
     * @see .SEG_QUADTO
     *
     * @see .SEG_CUBICTO
     *
     * @see .SEG_CLOSE
     */
    override fun currentSegment(coords: DoubleArray): Int {
        if (isDone) {
            throw NoSuchElementException("quad iterator iterator out of bounds")
        }
        val type: Int
        if (index == 0) {
            coords[0] = quad.x1
            coords[1] = quad.y1
            type = SEG_MOVETO
        } else {
            coords[0] = quad.ctrlX
            coords[1] = quad.ctrlY
            coords[2] = quad.x2
            coords[3] = quad.y2
            type = SEG_QUADTO
        }
        if (affine != null) {
            affine.transform(coords, 0, coords, 0, if (index == 0) 1 else 2)
        }
        return type
    }

    init {
        quad = q
        affine = at
    }
}