/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.mkui.geom

import org.mkui.geom.PathIterator.Companion.SEG_CLOSE
import org.mkui.geom.PathIterator.Companion.SEG_LINETO
import org.mkui.geom.PathIterator.Companion.SEG_MOVETO
import org.mkui.geom.PathIterator.Companion.WIND_NON_ZERO

class RectIterator internal constructor(r: Rectangle2D, at: AffineTransform?) : PathIterator {
    val x: Double
    val y: Double
    val w: Double
    val h: Double
    val affine: AffineTransform?
    var index = 0

    /**
     * Return the winding rule for determining the insideness of the
     * path.
     *
     * @see .WIND_EVEN_ODD
     *
     * @see .WIND_NON_ZERO
     */
    override val windingRule: Int
        get() = WIND_NON_ZERO

    /**
     * Tests if there are more points to read.
     *
     * @return true if there are more points to read
     */
    override val isDone: Boolean
        get() = index > 5

    /**
     * Moves the iterator to the next segment of the path forwards
     * along the primary direction of traversal as long as there are
     * more points in that direction.
     */
    override operator fun next() {
        index++
    }

    /**
     * Returns the coordinates and type of the current path segment in
     * the iteration.
     * The return value is the path segment type:
     * SEG_MOVETO, SEG_LINETO, SEG_QUADTO, SEG_CUBICTO, or SEG_CLOSE.
     * A float array of length 6 must be passed in and may be used to
     * store the coordinates of the point(s).
     * Each point is stored as a pair of float x,y coordinates.
     * SEG_MOVETO and SEG_LINETO types will return one point,
     * SEG_QUADTO will return two points,
     * SEG_CUBICTO will return 3 points
     * and SEG_CLOSE will not return any points.
     *
     * @see .SEG_MOVETO
     *
     * @see .SEG_LINETO
     *
     * @see .SEG_QUADTO
     *
     * @see .SEG_CUBICTO
     *
     * @see .SEG_CLOSE
     */
    override fun currentSegment(coords: FloatArray): Int {
        if (isDone) {
            throw NoSuchElementException("rect iterator out of bounds")
        }
        if (index == 5) {
            return SEG_CLOSE
        }
        coords[0] = x.toFloat()
        coords[1] = y.toFloat()
        if (index == 1 || index == 2) {
            coords[0] += w.toFloat()
        }
        if (index == 2 || index == 3) {
            coords[1] += h.toFloat()
        }
        if (affine != null) {
            affine.transform(coords, 0, coords, 0, 1)
        }
        return if (index == 0) SEG_MOVETO else SEG_LINETO
    }

    /**
     * Returns the coordinates and type of the current path segment in
     * the iteration.
     * The return value is the path segment type:
     * SEG_MOVETO, SEG_LINETO, SEG_QUADTO, SEG_CUBICTO, or SEG_CLOSE.
     * A double array of length 6 must be passed in and may be used to
     * store the coordinates of the point(s).
     * Each point is stored as a pair of double x,y coordinates.
     * SEG_MOVETO and SEG_LINETO types will return one point,
     * SEG_QUADTO will return two points,
     * SEG_CUBICTO will return 3 points
     * and SEG_CLOSE will not return any points.
     *
     * @see .SEG_MOVETO
     *
     * @see .SEG_LINETO
     *
     * @see .SEG_QUADTO
     *
     * @see .SEG_CUBICTO
     *
     * @see .SEG_CLOSE
     */
    override fun currentSegment(coords: DoubleArray): Int {
        if (isDone) {
            throw NoSuchElementException("rect iterator out of bounds")
        }
        if (index == 5) {
            return SEG_CLOSE
        }
        coords[0] = x
        coords[1] = y
        if (index == 1 || index == 2) {
            coords[0] += w
        }
        if (index == 2 || index == 3) {
            coords[1] += h
        }
        if (affine != null) {
            affine.transform(coords, 0, coords, 0, 1)
        }
        return if (index == 0) SEG_MOVETO else SEG_LINETO
    }

    init {
        x = r.x
        y = r.y
        w = r.width
        h = r.height
        affine = at
        if (w < 0 || h < 0) {
            index = 6
        }
    }
}