/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.mkui.geom

import kotlin.math.max
import kotlin.math.min

class Rectangle : AbstractRectangle2D {
    var ix = 0
    var iy = 0
    var iwidth = 0
    var iheight = 0

    constructor() {}
    constructor(rectangle: Rectangle) {
        ix = rectangle.ix
        iy = rectangle.iy
        iwidth = rectangle.iwidth
        iheight = rectangle.iheight
    }

    constructor(x: Int, y: Int, width: Int, height: Int) {
        this.ix = x
        this.iy = y
        this.iwidth = width
        this.iheight = height
    }

    override fun add(x: Double, y: Double): Rectangle2D {
        val x1: Double = min(minX, x)
        val x2: Double = max(maxX, x)
        val y1: Double = min(minY, y)
        val y2: Double = max(maxY, y)
        return Rectangle2D.Double(x1, y1, x2 - x1, y2 - y1)
    }

    /**
     * @see Geometry.toPath
     */
    fun toPath(): Path {
        val path = Path()
        path.moveTo(x.toDouble(), y.toDouble())
        path.lineTo((x + width).toDouble(), y.toDouble())
        path.lineTo((x + width).toDouble(), (y + height).toDouble())
        path.lineTo(x.toDouble(), (y + height).toDouble())
        path.close()
        return path
    }

    /**
     * Computes the intersection of this `Rectangle` with the
     * specified `Rectangle`. Returns a new `Rectangle`
     * that represents the intersection of the two rectangles.
     * If the two rectangles do not intersect, the result will be
     * an empty rectangle.
     *
     * @param r the specified `Rectangle`
     *
     * @return the largest `Rectangle` contained in both the
     * specified `Rectangle` and in
     * this `Rectangle`; or if the rectangles
     * do not intersect, an empty rectangle.
     */
    fun intersection(r: Rectangle): Rectangle {
        var tx1 = ix
        var ty1 = iy
        val rx1 = r.ix
        val ry1 = r.iy
        var tx2 = tx1
        tx2 += iwidth
        var ty2 = ty1
        ty2 += iheight
        var rx2 = rx1
        rx2 += r.iwidth
        var ry2 = ry1
        ry2 += r.iheight
        if (tx1 < rx1) tx1 = rx1
        if (ty1 < ry1) ty1 = ry1
        if (tx2 > rx2) tx2 = rx2
        if (ty2 > ry2) ty2 = ry2
        tx2 -= tx1
        ty2 -= ty1
        // tx2,ty2 will never overflow (they will never be
        // larger than the smallest of the two source w,h)
        // they might underflow, though...
        if (tx2 < Int.MIN_VALUE) tx2 = Int.MIN_VALUE
        if (ty2 < Int.MIN_VALUE) ty2 = Int.MIN_VALUE
        return Rectangle(tx1, ty1, tx2, ty2)
    }

    fun setX(x: Int) {
        this.ix = x
    }

    override fun toString(): String {
        return "Rectangle{" +
                "x=" + x +
                ", y=" + y +
                ", width=" + width +
                ", height=" + height +
                '}'
    }

    override val x: Double
        get() = ix.toDouble()
    override val y: Double
        get() = iy.toDouble()
    override val width: Double
        get() = iwidth.toDouble()
    override val height: Double
        get() = iheight.toDouble()

    /**
     * Returns whether the point given by x and y is within the boundaries of
     * this Rectangle.
     *
     * @param x the x-coordinate of the point to test
     * @param y the y-coordinate of the point to test
     *
     * @return true if the Point is (imprecisely) contained within this
     * Rectangle
     */
    override operator fun contains(point: Point2D): Boolean {
        return point.y >= y && point.y <= y + height && point.x >= x && point.x <= x + width
    }

    /**
     * Normalizes this rectangle to have a positive width and height,
     * adjusting the reference position *P1* if necessary.
     *
     * @return a new rectangle or this rectangle
     */
    override fun normalize(): Rectangle2D {
        // quick check if anything needs to be done
        if (isNormalized) return this
        var normalizedX = ix
        var normalizedY = iy
        var normalizedWidth = iwidth
        var normalizedHeight = iheight
        if (normalizedWidth < 0) {
            normalizedWidth = -normalizedWidth
            normalizedX -= normalizedWidth
        }
        if (normalizedHeight < 0) {
            normalizedHeight = -normalizedHeight
            normalizedY -= iheight
        }
        return Rectangle(
            normalizedX, normalizedY, normalizedWidth,
            normalizedHeight
        )
    }

    /**
     * Returns true if both width and height are positive.
     *
     * @return true if normalized
     */
    private val isNormalized: Boolean
        private get() = width >= 0 && height >= 0

    override fun intersects(x: Double, y: Double, w: Double, h: Double): Boolean {
        return intersects(Rectangle2D.Double(x, y, w, h))
    }

    override fun contains(x: Double, y: Double, w: Double, h: Double): Boolean {
        if (isEmpty || w <= 0 || h <= 0) {
            return false
        }
        val x0 = x
        val y0 = y
        return x >= x0 && y >= y0 && x + w <= x0 + width && y + h <= y0 + height
    }

    override fun contains(x: Double, y: Double): Boolean {
        return contains(Point2D.Double(x, y))
    }

    val isEmpty: Boolean
        get() = width <= 0 || height <= 0
}