/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.mkui.geom

import org.mkui.geom.PathIterator.Companion.SEG_CLOSE
import org.mkui.geom.PathIterator.Companion.SEG_CUBICTO
import org.mkui.geom.PathIterator.Companion.SEG_LINETO
import org.mkui.geom.PathIterator.Companion.SEG_MOVETO
import org.mkui.geom.PathIterator.Companion.WIND_NON_ZERO
import kotlin.math.*

internal class RoundRectIterator(rr: RoundRectangle2D, at: AffineTransform?) : PathIterator {
    val x: Double
    val y: Double
    val w: Double
    val h: Double
    val aw: Double
    val ah: Double
    val affine: AffineTransform?
    var index = 0

    /**
     * Return the winding rule for determining the insideness of the
     * path.
     *
     * @see .WIND_EVEN_ODD
     *
     * @see .WIND_NON_ZERO
     */
    override val windingRule: Int
        get() = WIND_NON_ZERO

    /**
     * Tests if there are more points to read.
     *
     * @return true if there are more points to read
     */
    override val isDone: Boolean
        get() = index >= ctrlpts.size

    /**
     * Moves the iterator to the next segment of the path forwards
     * along the primary direction of traversal as long as there are
     * more points in that direction.
     */
    override operator fun next() {
        index++
    }

    /**
     * Returns the coordinates and type of the current path segment in
     * the iteration.
     * The return value is the path segment type:
     * SEG_MOVETO, SEG_LINETO, SEG_QUADTO, SEG_CUBICTO, or SEG_CLOSE.
     * A float array of length 6 must be passed in and may be used to
     * store the coordinates of the point(s).
     * Each point is stored as a pair of float x,y coordinates.
     * SEG_MOVETO and SEG_LINETO types will return one point,
     * SEG_QUADTO will return two points,
     * SEG_CUBICTO will return 3 points
     * and SEG_CLOSE will not return any points.
     *
     * @see .SEG_MOVETO
     *
     * @see .SEG_LINETO
     *
     * @see .SEG_QUADTO
     *
     * @see .SEG_CUBICTO
     *
     * @see .SEG_CLOSE
     */
    override fun currentSegment(coords: FloatArray): Int {
        if (isDone) {
            throw NoSuchElementException("roundrect iterator out of bounds")
        }
        val ctrls = ctrlpts[index]
        var nc = 0
        var i = 0
        while (i < ctrls.size) {
            coords[nc++] = (x + ctrls[i + 0] * w + ctrls[i + 1] * aw).toFloat()
            coords[nc++] = (y + ctrls[i + 2] * h + ctrls[i + 3] * ah).toFloat()
            i += 4
        }
        if (affine != null) {
            affine.transform(coords, 0, coords, 0, nc / 2)
        }
        return types[index]
    }

    /**
     * Returns the coordinates and type of the current path segment in
     * the iteration.
     * The return value is the path segment type:
     * SEG_MOVETO, SEG_LINETO, SEG_QUADTO, SEG_CUBICTO, or SEG_CLOSE.
     * A double array of length 6 must be passed in and may be used to
     * store the coordinates of the point(s).
     * Each point is stored as a pair of double x,y coordinates.
     * SEG_MOVETO and SEG_LINETO types will return one point,
     * SEG_QUADTO will return two points,
     * SEG_CUBICTO will return 3 points
     * and SEG_CLOSE will not return any points.
     *
     * @see .SEG_MOVETO
     *
     * @see .SEG_LINETO
     *
     * @see .SEG_QUADTO
     *
     * @see .SEG_CUBICTO
     *
     * @see .SEG_CLOSE
     */
    override fun currentSegment(coords: DoubleArray): Int {
        if (isDone) {
            throw NoSuchElementException("roundrect iterator out of bounds")
        }
        val ctrls = ctrlpts[index]
        var nc = 0
        var i = 0
        while (i < ctrls.size) {
            coords[nc++] = x + ctrls[i + 0] * w + ctrls[i + 1] * aw
            coords[nc++] = y + ctrls[i + 2] * h + ctrls[i + 3] * ah
            i += 4
        }
        if (affine != null) {
            affine.transform(coords, 0, coords, 0, nc / 2)
        }
        return types[index]
    }

    companion object {
        private val angle: Double = PI / 4.0
        private val a: Double = 1.0 - cos(angle)
        private val b: Double = tan(angle)
        private val c: Double = sqrt(1.0 + b * b) - 1 + a
        private val cv = 4.0 / 3.0 * a * b / c
        private val acv = (1.0 - cv) / 2.0

        // For each array:
        //     4 values for each point {v0, v1, v2, v3}:
        //         point = (x + v0 * w + v1 * arcWidth,
        //                  y + v2 * h + v3 * arcHeight);
        private val ctrlpts = arrayOf(
            doubleArrayOf(0.0, 0.0, 0.0, 0.5), doubleArrayOf(0.0, 0.0, 1.0, -0.5), doubleArrayOf(
                0.0, 0.0, 1.0, -acv,
                0.0, acv, 1.0, 0.0,
                0.0, 0.5, 1.0, 0.0
            ), doubleArrayOf(1.0, -0.5, 1.0, 0.0), doubleArrayOf(
                1.0, -acv, 1.0, 0.0,
                1.0, 0.0, 1.0, -acv,
                1.0, 0.0, 1.0, -0.5
            ), doubleArrayOf(1.0, 0.0, 0.0, 0.5), doubleArrayOf(
                1.0, 0.0, 0.0, acv,
                1.0, -acv, 0.0, 0.0,
                1.0, -0.5, 0.0, 0.0
            ), doubleArrayOf(0.0, 0.5, 0.0, 0.0), doubleArrayOf(
                0.0, acv, 0.0, 0.0,
                0.0, 0.0, 0.0, acv,
                0.0, 0.0, 0.0, 0.5
            ), doubleArrayOf()
        )
        private val types = intArrayOf(
            SEG_MOVETO,
            SEG_LINETO, SEG_CUBICTO,
            SEG_LINETO, SEG_CUBICTO,
            SEG_LINETO, SEG_CUBICTO,
            SEG_LINETO, SEG_CUBICTO,
            SEG_CLOSE
        )
    }

    init {
        x = rr.x
        y = rr.y
        w = rr.width
        h = rr.height
        aw = min(w, abs(rr.arcWidth))
        ah = min(h, abs(rr.arcHeight))
        affine = at
        if (aw < 0 || ah < 0) {
            // Don't draw anything...
            index = ctrlpts.size
        }
    }
}