package org.mkui.geom.curve

/**
 *
 * The Lagrange curve passes through the control-points specified by the group-iterator.
 * It uses a knot-vector to control when the curve passes through each control-point.  That is,
 * if there is a knot-value for every control-point, then the curve will pass through point i
 * when the value of t is knot[i], which is an interesting property.  Figure 1 is an example of
 * this.
 *
 *
 *
 * <center><img align="center" src="doc-files/lagrange1.gif"></img></center>
 *
 *
 *
 * In addition, when there is a knot-value for every point then the base-index should be 0, and the
 * base-length should be n-1, where n is the size of the group-iterator.
 *
 *
 *
 * A knot-vector with size less than n can still be used.  In this case the Lagrange curve is
 * generated in multiple sections.  This approach works better when the points are roughly equally
 * spaced.  Figure 2 is an example of this.
 *
 *
 *
 * <center><img align="center" src="doc-files/lagrange2.gif"></img></center>
 *
 *
 *
 * Lagrange curves and also be closed as shown in figures 3 &amp; 4.
 *
 *
 *
 * <center><img align="center" src="doc-files/lagrange3.gif"></img></center>
 *
 *
 *
 * <center><img align="center" src="doc-files/lagrange4.gif"></img></center>
 *
 *
 *
 * Notes on the knot-vector, base-index and base-length.  The size of the knot-vector specifies how many
 * points are used for each section of the curve.  The base-index specifies which point a section starts
 * at.  The base-index + base-length specify which point the section ends at.  Once a section has been
 * generated, the next section is generated starting from the end of the last section.
 */
class LagrangeCurve
/**
 * Creates a LagrangeCurve with knot vector [0, 1/3, 2/3, 1], baseIndex == 1, baseLength == 1,
 * interpolateFirst and interpolateLast are both false.  The knot vector, baseIndex and baseLength
 * along with the control points define the shape of curve.  See the appendTo method for more information.
 *
 * @see .appendTo
 */
    (cp: ControlPath?, gi: GroupIterator?) : ParametricCurve(cp, gi) {
    private var knotVector: ValueVector? = ValueVector(doubleArrayOf(0.0, 1.0 / 3.0, 2.0 / 3.0, 1.0), 4)
    private var baseIndex = 1
    private var baseLength = 1
    /**
     * If baseIndex > 0 then the first control-points will only be interpolated if interpolate-first
     * is set to true.
     *
     * @see .setInterpolateFirst
     */
    /**
     * Sets the value of the interpolateFirst flag.
     *
     * @see .getInterpolateFirst
     */
    var interpolateFirst = false
    /**
     * If baseIndex + baseLength < numKnots - 1 then the last control-points will only be interpolated if
     * interpolate-last is set to true.
     *
     * @see .setInterpolateLast
     */
    /**
     * Sets the value of the interpolateLast flag.
     *
     * @see .getInterpolateLast
     */
    var interpolateLast = false

    /**
     * Returns the base-index.  The default value is 1.
     *
     * @see .setBaseIndex
     */
    fun getBaseIndex(): Int {
        return baseIndex
    }

    /**
     * The base-index is an index location into the knot vector such that, for each section, the curve is
     * evaluated between [knot[baseIndex], knot[baseIndex + baseLength]].
     *
     * @see .getBaseIndex
     */
    fun setBaseIndex(b: Int) {
        baseIndex = b
    }

    /**
     * Returns the base-length.  The default value is 1.
     *
     * @see .setBaseLength
     */
    fun getBaseLength(): Int {
        return baseLength
    }

    /**
     * The base-length along with the base-index specify the interval to evaluate each section.
     *
     * @see .getBaseLength
     */
    fun setBaseLength(b: Int) {
        baseLength = b
    }

    /**
     * Returns the knot-vector for this curve.
     *
     * @see .setKnotVector
     */
    fun getKnotVector(): ValueVector? {
        return knotVector
    }

    /**
     * Sets the knot-vector for this curve.
     *
     * @see .getKnotVector
     */
    fun setKnotVector(v: ValueVector?) {
        knotVector = v
    }

    /**
     * For the control-points to be interpolated in order, the knot-vector values should be strictly
     * increasing, however that is not required.  The requirements are the group-iterator must be in
     * range and baseIndex + baseLength &lt; numKnots. As well, the number of points defined by the
     * group-iterator must be &gt;= numKnots, otherwise the curve does not have enough control-points
     * to define itself.  If any of these requirements are not met, then this method returns quietly.
     */
    override fun appendTo(mp: MultiPath) {
        if (!gi!!.isInRange(0, cp!!.numPoints())) return
        if (baseIndex + baseLength >= knotVector!!.size()) return
        if (pt.size < knotVector!!.size()) pt = arrayOfNulls(2 * knotVector!!.size())
        gi!!.set(0, 0)
        var b = false
        if (baseIndex != 0 && interpolateFirst) {
            for (i in 0 until knotVector!!.size()) {
                if (!gi!!.hasNext()) return
                pt[i] = cp!!.getPoint(gi!!.next())!!.location
            }
            b = doBCAA(mp, knotVector!!.get(0), knotVector!!.get(baseIndex), b)
        }
        gi!!.set(0, 0)
        var last_i = 0
        var last_j = 0
        while (true) {
            val temp_i: Int = gi!!.index_i()
            val temp_j: Int = gi!!.count_j()
            var index_i = 0
            var count_j = 0
            var j = 0
            var i = 0
            while (j < knotVector!!.size()) {
                if (i == baseLength) {
                    index_i = gi!!.index_i()
                    count_j = gi!!.count_j()
                }
                if (!gi!!.hasNext()) break
                pt[j] = cp!!.getPoint(gi!!.next())!!.location
                i++
                j++
            }
            if (j < knotVector!!.size()) {
                break
            } else {
                gi!!.set(index_i, count_j)
                last_i = temp_i
                last_j = temp_j
            }
            b = doBCAA(mp, knotVector!!.get(baseIndex), knotVector!!.get(baseIndex + baseLength), b)
        }
        if (baseIndex + baseLength < knotVector!!.size() - 1 && interpolateLast) {
            gi!!.set(last_i, last_j)
            for (i in 0 until knotVector!!.size()) {
                if (!gi!!.hasNext()) {
                    println("not enough points to interpolate last")
                    return
                }
                pt[i] = cp!!.getPoint(gi!!.next())!!.location
            }
            doBCAA(mp, knotVector!!.get(baseIndex + baseLength), knotVector!!.get(knotVector!!.size() - 1), b)
        }
    }

    private fun doBCAA(mp: MultiPath, t1: Double, t2: Double, b: Boolean): Boolean {
        var t1 = t1
        var t2 = t2
        var b = b
        if (t2 < t1) {
            val temp = t1
            t1 = t2
            t2 = temp
        }
        if (!b) {
            b = true
            val d = DoubleArray(mp.dimension + 1)
            d[mp.dimension] = t1
            eval(d)
            if (connect) mp.lineTo(d) else mp.moveTo(d)
        }
        BinaryCurveApproximationAlgorithm.genPts(this, t1, t2, mp)
        return b
    }

    override fun eval(p: DoubleArray) {
        val t = p[p.size - 1]
        val n: Int = knotVector!!.size()
        for (i in 0 until n) {
            val q = pt[i]
            val L = L(t, i)
            for (j in 0 until p.size - 1) p[j] += q!![j] * L
        }
    }

    /**
     * Returns a value of 1.
     */
    override val sampleLimit: Int
        get() = 1

    private fun L(t: Double, i: Int): Double {
        var d = 1.0
        val n: Int = knotVector!!.size()
        for (j in 0 until n) {
            val e: Double = knotVector!!.get(i) - knotVector!!.get(j)
            if (e != 0.0) d *= (t - knotVector!!.get(j)) / e
        }
        return d
    }

    override fun resetMemory() {
        if (pt.size > 0) pt = arrayOfNulls(0)
    }

    companion object {
        private var pt = arrayOfNulls<DoubleArray>(0)
    }
}