package org.mkui.geom.curve

import com.macrofocus.common.collection.arraycopy

/**
 *
 * The natural-cubic-spline is constructed using piecewise third order polynomials which pass through all the
 * control-points specified by the group-iterator.  The curve can be open or closed.  Figure 1 shows an open
 * curve and figure 2 shows a closed curve.
 *
 *
 *
 * <center><img align="center" src="doc-files/natcubic1.gif"></img></center>
 *
 *
 *
 * <center><img align="center" src="doc-files/natcubic2.gif"></img></center>
 */
class NaturalCubicSpline(cp: ControlPath?, gi: GroupIterator?) : ParametricCurve(cp, gi) {
    /**
     * Returns the value of closed.  The default value is false.
     *
     * @see .setClosed
     */
    /**
     * The closed attribute determines which tri-diagonal matrix to solve.
     *
     * @see .getClosed
     */
    var closed = false

    /**
     * The requirements for this curve are the group-iterator must be in-range and have a group size of at least 2.
     * If these requirements are not met then this method returns quietly.
     */
    override fun appendTo(mp: MultiPath) {
        if (!gi!!.isInRange(0, cp!!.numPoints())) return
        val n: Int = gi!!.groupSize
        if (n < 2) return
        val dim: Int = mp.dimension

        // make sure there is enough room
        //-------------------------------------------------------
        val x = 3 + 4 * dim + 1
        if (data.size < x) {
            val temp = arrayOfNulls<DoubleArray>(x)
            arraycopy(data, 0, temp, 0, data.size)
            data = temp
        }
        if (pt.size < n) {
            val m = 2 * n
            pt = arrayOfNulls(m)
            for (i in data.indices) data[i] = DoubleArray(m)
        }
        //-------------------------------------------------------
        gi!!.set(0, 0)
        for (i in 0 until n) pt[i] = cp!!.getPoint(gi!!.next())!!.location // assign the used points to pt
        precalc(n, dim, closed)
        ci = 0 // do not remove
        val p = DoubleArray(dim + 1)
        eval(p)
        if (connect) mp.lineTo(p) else mp.moveTo(p)

        // Note: performing a ci++ or ci = ci + 1 results in funny behavior
        for (i in 0 until n) {
            ci = i
            BinaryCurveApproximationAlgorithm.genPts(this, 0.0, 1.0, mp)
        }
    }

    override fun eval(p: DoubleArray) {
        val n = p.size - 1 // dimension
        val t = p[n]
        val t2 = t * t
        val t3 = t2 * t
        var j = 0
        for (i in 0 until n) p[i] = data[j++]!![ci] + data[j++]!![ci] * t + data[j++]!![ci] * t2 + data[j++]!![ci] * t3
    }

    /**
     * Returns a value of 1.
     */
    override val sampleLimit: Int
        get() = 1

    override fun resetMemory() {
        if (pt.size > 0) pt = arrayOfNulls(0)
        if (data.size > 0) data = arrayOfNulls(0)
    }

    companion object {
        /*
The pt array stores the points of the control-path.
The data array is used to store the result of the many calculations.

d[0] = w1  For each dimension, 4 arrays are required to store the
d[1] = x1  results of the calculations.
d[2] = y1  The length of each array is >= to the number of points.
d[3] = z1
d[4] = w2
d[5] = x2
d[6] = y2
d[7] = z2
d[8] = a   // a, b & c are used (by both open and closed) to store
d[9] = b   // the results of the calculations.
d[10] = c
d[11] = d // only used for closed cubic curves
*/
        private var pt = arrayOfNulls<DoubleArray>(0)
        private var data = arrayOfNulls<DoubleArray>(0)
        private var ci = 0

        // n is the # of points
        // dim is the dimension
        private fun precalc(n: Int, dim: Int, closed: Boolean) {
            var n = n
            n--
            val a = data[4 * dim]
            val b = data[4 * dim + 1]
            val c = data[4 * dim + 2]
            var k = 0
            if (closed) {
                val d = data[4 * dim + 3]
                for (j in 0 until dim) {
                    var e: Double
                    e = 0.25
                    a!![1] = e
                    d!![1] = a[1]
                    b!![0] = e * 3 * (pt[1]!![j] - pt[n]!![j])
                    var h = 4.0
                    var f = 3 * (pt[0]!![j] - pt[n - 1]!![j])
                    var g = 1.0
                    for (i in 1 until n) {
                        e = 1.0 / (4.0 - a[i])
                        a[i + 1] = e
                        d[i + 1] = -e * d[i]
                        b[i] = e * (3.0 * (pt[i + 1]!![j] - pt[i - 1]!![j]) - b[i - 1])
                        h -= g * d[i]
                        f -= g * b[i - 1]
                        g = -a[i] * g
                    }
                    h -= (g + 1) * (a[n] + d[n])
                    b[n] = f - (g + 1) * b[n - 1]
                    c!![n] = b[n] / h
                    c[n - 1] = b[n - 1] - (a[n] + d[n]) * c[n]
                    for (i in n - 2 downTo 0) {
                        c[i] = b[i] - a[i + 1] * c[i + 1] - d[i + 1] * c[n]
                    }
                    val w = data[k++]
                    val x = data[k++]
                    val y = data[k++]
                    val z = data[k++]
                    for (i in 0 until n) {
                        w!![i] = pt[i]!![j]
                        x!![i] = c[i]
                        y!![i] = 3 * (pt[i + 1]!![j] - pt[i]!![j]) - 2 * c[i] - c[i + 1]
                        z!![i] = 2 * (pt[i]!![j] - pt[i + 1]!![j]) + c[i] + c[i + 1]
                    }
                    w!![n] = pt[n]!![j]
                    x!![n] = c[n]
                    y!![n] = 3 * (pt[0]!![j] - pt[n]!![j]) - 2 * c[n] - c[0]
                    z!![n] = 2 * (pt[n]!![j] - pt[0]!![j]) + c[n] + c[0]
                }
            } else {
                for (j in 0 until dim) {
                    a!![0] = 0.5
                    for (i in 1 until n) {
                        a[i] = 1.0 / (4 - a[i - 1])
                    }
                    a[n] = 1.0 / (2.0 - a[n - 1])
                    b!![0] = a[0] * (3 * (pt[1]!![j] - pt[0]!![j]))
                    for (i in 1 until n) {
                        b[i] = a[i] * (3 * (pt[i + 1]!![j] - pt[i - 1]!![j]) - b[i - 1])
                    }
                    b[n] = a[n] * (3 * (pt[n]!![j] - pt[n - 1]!![j]) - b[n - 1])
                    c!![n] = b[n]
                    for (i in n - 1 downTo 0) {
                        c[i] = b[i] - a[i] * c[i + 1]
                    }
                    val w = data[k++]
                    val x = data[k++]
                    val y = data[k++]
                    val z = data[k++]
                    for (i in 0 until n) {
                        w!![i] = pt[i]!![j]
                        x!![i] = c[i]
                        y!![i] = 3 * (pt[i + 1]!![j] - pt[i]!![j]) - 2 * c[i] - c[i + 1]
                        z!![i] = 2 * (pt[i]!![j] - pt[i + 1]!![j]) + c[i] + c[i + 1]
                    }
                    w!![n] = pt[n]!![j]
                    x!![n] = 0.0
                    y!![n] = 0.0
                    z!![n] = 0.0
                }
            }
        }
    }
}