package org.mkui.labeling

import com.macrofocus.common.properties.MutableProperties
import com.macrofocus.common.properties.MutableProperty
import com.macrofocus.common.properties.Properties
import com.macrofocus.common.properties.SimpleProperties
import org.mkui.color.CPColor
import org.mkui.color.HSBtoRGB
import org.mkui.color.RGBtoHSB
import org.mkui.font.CPFont
import kotlin.math.abs
import kotlin.math.max
import kotlin.math.min

/**
 * Enhanced version of JLabel supporting text effects and word-wrap.
 */
open class EnhancedLabel() {
    var isOpaque = false

    enum class Rendering {
        Truncate, Clip, WordWrap
    }

    enum class Effect {
        Plain, Shadow, Glow, Outline, Emphasize
    }

    enum class ContrastEnhancement {
        Off {
            override fun adjustLabelColor(
                foreground: CPColor,
                background: CPColor
            ): CPColor {
                return foreground
            }
        },
        Brightness {
            override fun adjustLabelColor(
                foreground: CPColor,
                background: CPColor
            ): CPColor {
                var foreground: CPColor = foreground
                val diff: Float = foreground.diff(background)
                return if (diff <= COLOR_SIMILARITY_THRESHOLD) {
                    // Background and foreground colors are two close to one another... let's make some adjustments...

                    // Find two alternative colors, one brighter and the other one darker
                    val foregroundBrightness: Double = foreground.brightness()
                    val brighterForeground: CPColor = foreground.replaceBrightness(
                        min(
                            1.0,
                            foregroundBrightness + (COLOR_SIMILARITY_THRESHOLD - diff)
                        )
                    )
                    val darkerForeground: CPColor = foreground.replaceBrightness(
                        max(
                            0.0,
                            foregroundBrightness - (COLOR_SIMILARITY_THRESHOLD - diff)
                        )
                    )

                    // Compute their luminance compared to the one of the background
                    val backgroundLuminance: Float = background.luminance()
                    val brighterDiff: Float = abs(brighterForeground.luminance() - backgroundLuminance)
                    val darkerDiff: Float = abs(darkerForeground.luminance() - backgroundLuminance)

                    // Pick the color with the biggest contrast
                    foreground = if (brighterDiff > darkerDiff) {
                        brighterForeground
                    } else {
                        darkerForeground
                    }
                    foreground
                } else {
                    foreground
                }
            }
        },
        BlackWhite {
            override fun adjustLabelColor(
                foreground: CPColor,
                background: CPColor
            ): CPColor {
                val diff: Float = foreground.diff(background)
                return if (diff <= COLOR_SIMILARITY_THRESHOLD) {
                    val backgroundLuminance: Double = background.lum()

                    // The gamma curve applied to displays makes the middle gray value higher than you'd expect.
                    // This is easily solved by using 186 as the middle value rather than 128.
                    // Anything less than 186 should use white text, anything greater than 186 should use black text.
                    if (backgroundLuminance >= 186) Black else White
                } else {
                    foreground
                }
            }

            override fun toString(): String {
                return "Black&White"
            }
        },
        Complement {
            override fun adjustLabelColor(
                foreground: CPColor,
                background: CPColor
            ): CPColor {
                var foreground: CPColor = foreground
                val diff: Float = foreground.diff(background)
                return if (diff <= COLOR_SIMILARITY_THRESHOLD) {
                    // get existing colors
                    val alpha: UByte = foreground.getAlpha()
                    var red: UByte = foreground.getRed()
                    var blue: UByte = foreground.getBlue()
                    var green: UByte = foreground.getBlue()

                    // find compliments
                    red = red.inv()
                    blue = blue.inv()
                    green = green.inv()
                    val complement: CPColor = CPColor(red, green, blue, alpha)

                    // Compute their luminance compared to the one of the background
                    val backgroundLuminance: Float = background.luminance()
                    val brighterDiff: Float = abs(foreground.luminance() - backgroundLuminance)
                    val darkerDiff: Float = abs(complement.luminance() - backgroundLuminance)

                    // Pick the color with the biggest contrast
                    foreground = if (brighterDiff > darkerDiff) {
                        foreground
                    } else {
                        complement
                    }
                    foreground
                } else {
                    foreground
                }
            }
        };

        abstract fun adjustLabelColor(foreground: CPColor, background: CPColor): CPColor

        companion object {
            // Two colors provide good color visibility if the brightness difference and the color difference between the
            // two colors are greater than a set range. The rage for color brightness difference is 125. The range for color
            // difference is 500. We currently only base our computations on the color difference.
            const val COLOR_SIMILARITY_THRESHOLD = 0.5f
        }
    }

    var properties: MutableProperties<String?> = SimpleProperties()
    private val rendering = properties.createProperty("rendering", Rendering.Clip)
    private val justified = properties.createProperty<Boolean>("justified", false)
    private val effect = properties.createProperty("effect", Effect.Plain)
    private val effectOpacity = properties.createProperty("effectOpacity", 110f / 255f)
    private val minimumCharactersToDisplay = properties.createProperty<Int?>("minimumCharactersToDisplay", null)
    private val angle = properties.createProperty("angle", 0.0)
    private val desiredWidth = properties.createProperty("desiredWidth", Int.MAX_VALUE)
    private val desiredHeight = properties.createProperty("desiredHeight", Int.MAX_VALUE)
    private val text = properties.createProperty<String?>("text", null)
    private val html = properties.createProperty<Boolean>("html", false)
    private val font: MutableProperty<CPFont?> = properties.createProperty("font", null)
    private val verticalAlignment = properties.createProperty("verticalAlignment", CENTER)
    private val horizontalAlignment = properties.createProperty("horizontalAlignment", LEADING)
    private val width = properties.createProperty<Int?>("width", null)
    private val height = properties.createProperty<Int?>("height", null)
    private val insetTop = properties.createProperty("insetTop", 0)
    private val insetLeft = properties.createProperty("insetLeft", 0)
    private val insetBottom = properties.createProperty("insetBottom", 0)
    private val insetRight = properties.createProperty("insetRight", 0)
    private val background: MutableProperty<CPColor?> = properties.createProperty("background", null)
    private val foreground: MutableProperty<CPColor?> = properties.createProperty("foreground", null)
    private val enabled = properties.createProperty<Boolean>("enabled", true)
    private val name = properties.createProperty<String?>("name", null)

    constructor(text: String?) : this() {
        setText(text)
    }

    open fun setName(name: String?) {
        this.name.value = name
    }

    open fun getName(): String? {
        return name.value
    }

    fun getProperties(): Properties<String?>? {
        return properties
    }

    fun getText(): String? {
        return text.value
    }

    fun setText(text: String?) {
        this.text.value = text
    }

    var isHTML: Boolean
        get() = html.value
        set(html) {
            this.html.value = html
        }

    fun getFont(): CPFont? {
        return font.value
    }

    fun setFont(font: CPFont) {
        this.font.value = font
    }

    fun getVerticalAlignment(): Int {
        return verticalAlignment.value
    }

    fun setVerticalAlignment(verticalAlignment: Int) {
        this.verticalAlignment.value = verticalAlignment
    }

    fun getHorizontalAlignment(): Int {
        return horizontalAlignment.value
    }

    fun setHorizontalAlignment(horizontalAlignment: Int) {
        this.horizontalAlignment.value = horizontalAlignment
    }

    fun getHeight(): Int? {
        return height.value
    }

    fun getWidth(): Int? {
        return width.value
    }

    fun getBackground(): CPColor? {
        return background.value
    }

    fun setBackground(background: CPColor?) {
        this.background.value = background
    }

    fun getForeground(): CPColor? {
        return foreground.value
    }

    fun setForeground(foreground: CPColor) {
        this.foreground.value = foreground
    }

    fun isEnabled(): Boolean {
        return enabled.value
    }

    fun getInsetTop(): Int {
        return insetTop.value
    }

    fun getInsetLeft(): Int {
        return insetLeft.value
    }

    fun getInsetBottom(): Int {
        return insetBottom.value
    }

    fun getInsetRight(): Int {
        return insetRight.value
    }

    fun setInsetTop(value: Int) {
        insetTop.value = value
    }

    fun setInsetLeft(value: Int) {
        insetLeft.value = value
    }

    fun setInsetBottom(value: Int) {
        insetBottom.value = value
    }

    fun setInsetRight(value: Int) {
        insetRight.value = value
    }

    fun setInsets(top: Int, left: Int, bottom: Int, right: Int) {
        insetTop.value = top
        insetLeft.value = left
        insetBottom.value = bottom
        insetRight.value = right
    }

    fun getEffect() : Effect {
        return this.effect.value
    }

    fun setEffect(effect: Effect) {
        this.effect.value = effect
    }

    fun getEffectOpacity(): Float {
        return effectOpacity.value
    }

    fun setEffectOpacity(effectOpacity: Float) {
        this.effectOpacity.value = effectOpacity
    }

    fun getRendering(): Rendering {
        return rendering.value
    }

    fun setRendering(rendering: Rendering) {
        this.rendering.value = rendering
    }

    fun getMinimumCharactersToDisplay(): Int? {
        return minimumCharactersToDisplay.value
    }

    fun setMinimumCharactersToDisplay(minimumCharactersToDisplay: Int?) {
        this.minimumCharactersToDisplay.value = minimumCharactersToDisplay
    }

    fun isJustified(): Boolean {
        return justified.value
    }

    fun setJustified(justified: Boolean) {
        this.justified.value = justified
    }

    fun getAngle(): Double {
        return angle.value
    }

    /**
     * Sets the angle in degrees at which the text will be painted
     *
     * @param angle the angle in degrees
     */
    fun setAngle(angle: Double) {
        this.angle.value = angle
    }

    fun getDesiredWidth(): Int {
        return desiredWidth.value
    }

    fun setDesiredWidth(desiredSize: Int) {
        desiredWidth.value = desiredSize
    }

    fun getDesiredHeight(): Int {
        return desiredHeight.value
    }

    fun setDesiredHeight(desiredSize: Int) {
        desiredHeight.value = desiredSize
    }

    companion object {
        /**
         * The central position in an area. Used for
         * both compass-direction constants (NORTH, etc.)
         * and box-orientation constants (TOP, etc.).
         */
        const val CENTER = 0
        //
        // Box-orientation constant used to specify locations in a box.
        //
        /**
         * Box-orientation constant used to specify the top of a box.
         */
        const val TOP = 1

        /**
         * Box-orientation constant used to specify the left side of a box.
         */
        const val LEFT = 2

        /**
         * Box-orientation constant used to specify the bottom of a box.
         */
        const val BOTTOM = 3

        /**
         * Box-orientation constant used to specify the right side of a box.
         */
        const val RIGHT = 4
        //
        // These constants specify a horizontal or
        // vertical orientation. For example, they are
        // used by scrollbars and sliders.
        //
        /** Horizontal orientation. Used for scrollbars and sliders.  */
        const val HORIZONTAL = 0

        /** Vertical orientation. Used for scrollbars and sliders.  */
        const val VERTICAL = 1
        //
        // Constants for orientation support, since some languages are
        // left-to-right oriented and some are right-to-left oriented.
        // This orientation is currently used by buttons and labels.
        //
        /**
         * Identifies the leading edge of text for use with left-to-right
         * and right-to-left languages. Used by buttons and labels.
         */
        const val LEADING = 10

        /**
         * Identifies the trailing edge of text for use with left-to-right
         * and right-to-left languages. Used by buttons and labels.
         */
        const val TRAILING = 11

        /**
         * Identifies the next direction in a sequence.
         *
         * @since 1.4
         */
        const val NEXT = 12
    }

    init {
        setVerticalAlignment(TOP)
    }
}

private fun CPColor.replaceBrightness(min: Double): CPColor {
    val hsb: FloatArray = RGBtoHSB(getRed().toInt(), getGreen().toInt(), getBlue().toInt(), null)
    return getHSBColor(hsb[0], hsb[1], min.toFloat())
}

fun getHSBColor(h: Float, s: Float, b: Float): CPColor {
    return CPColor(HSBtoRGB(h, s, b))
}

private fun CPColor.brightness(): Double {
    val hsb: FloatArray = RGBtoHSB(getRed().toInt(), getGreen().toInt(), getBlue().toInt(), null)
    return hsb[2].toDouble()
}

/**
 * Return the monochrome luminance of this color.
 *
 * @return the monochrome luminance
 */
private fun CPColor.lum(): Double {
    val r = getRed().toInt()
    val g = getGreen().toInt()
    val b = getBlue().toInt()
    return .299 * r + .587 * g + .114 * b
}

/**
 * Return the monochrome luminance of given color
 */
private fun CPColor.luminance(): Float {
    val r = getRed().toInt()
    val g = getGreen().toInt()
    val b = getBlue().toInt()
    return (.299f * r + .587f * g + .114f * b) / 255f
}

private fun CPColor.diff(background: CPColor): Float {
    val r: Int = abs(getRed().toInt() - background.getRed().toInt())
    val g: Int = abs(getGreen().toInt() - background.getGreen().toInt())
    val b: Int = abs(getBlue().toInt() - background.getBlue().toInt())
    return max(r, max(g, b)) / 255f
}

val Black = CPColor(0u, 0u, 0u, 255u)
val White = CPColor(255u, 255u, 255u, 255u)