/*
 * Copyright (c) 2010 Macrofocus GmbH. All Rights Reserved.
 */
package org.mkui.palette

import com.macrofocus.common.collection.SortedSet
import com.macrofocus.common.collection.TreeSet
import com.macrofocus.common.properties.MutableProperty
import com.macrofocus.common.properties.SimpleProperty
import org.mkui.color.CPColor

/**
 * Palette where colors are interpolated according to some fixed points.
 */
class InterpolatedPalette constructor(var entries: SortedSet<Entry>) : FixedPalette(false, emptyList()) {
    private enum class Mode {
        Ramps, Bands
    }

    private val mode: MutableProperty<Mode> = SimpleProperty(Mode.Ramps)

    constructor(palette: Palette) : this(TreeSet<Entry>(compareBy<Entry> { it.fraction })) {
        for (i in 0 until palette.colorCount) {
            val c: CPColor = palette.getColorAt(i)
            entries.add(Entry(i.toDouble() / (palette.colorCount - 1), c))
        }
        updateFixedColorGradient()
    }

    constructor(vararg entries: Entry) : this(TreeSet<Entry>(compareBy<Entry> { it.fraction }, entries.asList()))
     {
        updateFixedColorGradient()
    }

    private fun updateFixedColorGradient() {
        if (!entries.isEmpty()) {
            var colors: MutableList<CPColor> = ArrayList<CPColor>()
            for (i in 0..255) {
                val fraction = 1.0 * i / 255
                val color: CPColor? = computeColor(fraction)
                if (color != null) {
                    colors.add(color)
                } else {
                    throw IllegalStateException("Null color " + fraction + ", " + color)
                }
            }
            this.colors = colors
            this.colorCount = colors.size
        } else {
            this.colors = emptyList()
            this.colorCount = 0
        }
    }
    //    public CPColor getColor(double fraction) {
    //        boolean inverted = isInverted();
    //
    //        switch (getCycle()) {
    //            case REFLECT:
    //                inverted = ((int) fraction) % 2 == 0;
    //            case REPEAT:
    //                fraction = fraction - (int) fraction;
    //        }
    //
    //        if (fraction < getLowestFraction()) {
    //            return getColor(getLowestFraction());
    //        } else if (fraction > getHighestFraction()) {
    //            return getColor(getHighestFraction());
    //        } else {
    //            final CPColor color = computeColor(!inverted ? fraction : 1 - fraction);
    //            return color;
    //        }
    //    }
    /**
     * Returns the Color at the specified position.
     *
     * @param fraction the normalized value
     * @return the color
     */
    private fun computeColor(fraction: Double): CPColor? {
        if (!entries.isEmpty()) {
            var previous: Entry? = null
            for (current in entries) {
                if (previous != null) {
                    if (previous.fraction <= fraction && fraction <= current.fraction) {
                        val v = (fraction - previous.fraction) / (current.fraction - previous.fraction)
                        return interpolate(previous.getColor(), current.getColor(), v)
                    }
                }
                previous = current
            }
            if (fraction < entries.first().fraction) {
                return entries.first().getColor()
            }
            if (fraction > entries.last().fraction) {
                return entries.last().getColor()
            }
        }
        return null
    }

    private fun interpolate(c1: CPColor, c2: CPColor, p: Double): CPColor {
        return when (mode.value) {
            Mode.Ramps -> {
                if (p == 0.0) return c1
                if (p == 1.0) c2 else CPColor(
                    (c1.getRed().toDouble() * (1.0 - p) + c2.getRed().toDouble() * p).toInt(),
                    (c1.getGreen().toDouble() * (1.0 - p) + c2.getGreen().toDouble() * p).toInt(),
                    (c1.getBlue().toDouble() * (1.0 - p) + c2.getBlue().toDouble() * p).toInt(),
                    (c1.getAlpha().toDouble() * (1.0 - p) + c2.getAlpha().toDouble() * p).toInt()
                )
            }
            Mode.Bands -> c1
        }
    }

    fun getEntries(): Set<Entry> {
        return entries
    }

    override val lowestFraction: Double
        protected get() = if (!entries.isEmpty()) {
            entries.first().fraction
        } else {
            super.lowestFraction
        }
    override val highestFraction: Double
        protected get() = if (!entries.isEmpty()) {
            entries.last().fraction
        } else {
            super.highestFraction
        }

    class Entry(val fraction: Double, color: CPColor) : Comparable<Entry> {
        private val color: CPColor
        fun getColor(): CPColor {
            return color
        }

        override operator fun compareTo(o: Entry): Int {
            return fraction.compareTo(o.fraction)
        }

        override fun equals(other: Any?): Boolean {
            if (this === other) return true
            if (other == null || this::class != other::class) return false

            other as Entry

            if (fraction != other.fraction) return false

            return true
        }

        override fun hashCode(): Int {
            return fraction.hashCode()
        }

        init {
            this.color = color
        }
    }

    init {
        updateFixedColorGradient()
    }
}