package org.mkui.transform

import com.macrofocus.common.collection.CollectionFactory
import com.macrofocus.common.collection.WeakReference
import com.macrofocus.common.transform.ScreenTransformEvent
import com.macrofocus.common.transform.ScreenTransformListener

/**
 * This class provides a skeletal implementation of the MutableTwoDScreenTransform interface to minimize the effort required to implement this interface.
 */
abstract class AbstractMutableTwoDScreenTransform protected constructor() : AbstractTwoDScreenTransform(),
    MutableTwoDScreenTransform {
    private val listeners: MutableList<ScreenTransformListener>
    override fun addScreenTransformListener(listener: ScreenTransformListener) {
        listeners.add(listener)
    }

    override fun addWeakScreenTransformListener(listener: ScreenTransformListener) {
        val weakListener = WeakScreenTransformListener(listener)
        listeners.add(weakListener)
    }

    override fun removeScreenTransformListener(listener: ScreenTransformListener) {
        if (listener is WeakScreenTransformListener) {
            val removed = listeners.remove(listener)
        } else {
            var toRemove: ScreenTransformListener? = null
            for (screenTransformListener in listeners) {
                var comparable: ScreenTransformListener
                if (screenTransformListener is WeakScreenTransformListener) {
                    comparable = (screenTransformListener as WeakScreenTransformListener).reference
                } else {
                    comparable = screenTransformListener
                }
                if (listener.equals(comparable)) {
                    toRemove = screenTransformListener
                }
            }
            if (toRemove != null) {
                val removed = listeners.remove(toRemove)
            }
        }
    }

    override fun removeScreenTransformListeners() {
        listeners.clear()
    }

    protected fun notifyTransformChanged(event: ScreenTransformEvent) {
        for (listener in listeners) {
            listener.transformChanged(event)
        }
    }

    private inner class WeakScreenTransformListener(listener: ScreenTransformListener) :
        ScreenTransformListener {
        private val l_ref: WeakReference<ScreenTransformListener>
        override fun transformChanged(event: ScreenTransformEvent) {
            val l: ScreenTransformListener = reference
            if (l != null) {
                l.transformChanged(event)
            } else {
                removeScreenTransformListener(this)
            }
        }

        val reference: ScreenTransformListener
            get() = l_ref.get()

        override fun toString(): String {
            val l: ScreenTransformListener = reference
            return if (l != null) {
                "Weak[$l]"
            } else {
                super.toString()
            }
        }

        init {
            l_ref = WeakReference<ScreenTransformListener>(listener)
        }
    }

    init {
        listeners = CollectionFactory.copyOnWriteArrayList<ScreenTransformListener>()
    }
}