package org.mkui.transform

import com.macrofocus.common.interval.BoundedInterval
import com.macrofocus.common.interval.Interval
import com.macrofocus.common.interval.IntervalEvent
import com.macrofocus.common.interval.IntervalListener
import com.macrofocus.common.transform.MutableOneDScreenTransform
import com.macrofocus.common.transform.ScreenTransformEvent
import com.macrofocus.common.transform.SimpleOneDScreenTransform
import org.mkui.geom.Point
import org.mkui.geom.Point2D
import org.mkui.geom.Rectangle
import org.mkui.geom.Rectangle2D
import kotlin.jvm.JvmOverloads

/**
 * Default implementation of a TwoDScreenTransform.
 */
class SimpleTwoDScreenTransform : AbstractMutableTwoDScreenTransform {
    override var screenWidth: Int
    override var screenHeight: Int
    override var xWorldInterval: Interval
        private set
    override var yWorldInterval: Interval
        private set
    override val x: MutableOneDScreenTransform
    override val y: MutableOneDScreenTransform
    private var maintainAspectRatio = false
    private val worldAspectRatio: Double
    private val intervalListener: IntervalListener = object : IntervalListener {
        override fun intervalChanged(event: IntervalEvent?) {
            notifyTransformChanged(ScreenTransformEvent())
        }
    }

    @JvmOverloads
    constructor(
        x: BoundedInterval,
        y: BoundedInterval,
        screenWidth: Int = 0,
        screenHeight: Int = 0,
        worldAspectRatio: Double = 1.0
    ) {
        xWorldInterval = x
        yWorldInterval = y
        this.x = SimpleOneDScreenTransform(x, screenWidth)
        this.y = SimpleOneDScreenTransform(y, screenHeight, true, false)
        this.screenWidth = screenWidth
        this.screenHeight = screenHeight
        this.worldAspectRatio = worldAspectRatio
        xWorldInterval!!.addIntervalListener(intervalListener)
        yWorldInterval!!.addIntervalListener(intervalListener)
    }

    constructor(
        x: MutableOneDScreenTransform,
        y: MutableOneDScreenTransform,
        screenWidth: Int,
        screenHeight: Int,
        worldAspectRatio: Double
    ) {
        xWorldInterval = x.worldInterval
        yWorldInterval = y.worldInterval
        this.x = x
        this.y = y
        this.screenWidth = screenWidth
        this.screenHeight = screenHeight
        this.worldAspectRatio = worldAspectRatio
        xWorldInterval!!.addIntervalListener(intervalListener)
        yWorldInterval!!.addIntervalListener(intervalListener)
    }

    override fun setIntervals(xInterval: BoundedInterval, yInterval: BoundedInterval) {
        xWorldInterval!!.removeIntervalListener(intervalListener)
        yWorldInterval!!.removeIntervalListener(intervalListener)
        xWorldInterval = xInterval
        yWorldInterval = yInterval
        if (xInterval != null) {
            xWorldInterval!!.addIntervalListener(intervalListener)
            x.setWorldInterval(xInterval)
        }
        if (yInterval != null) {
            yWorldInterval!!.addIntervalListener(intervalListener)
            y.setWorldInterval(yInterval)
        }
    }

    override fun screenToWorld(screen: Point): Point2D {
        return Point2D.Double(x.screenToWorld(screen.ix), y.screenToWorld(screen.iy))
    }

    override fun screenToWorld(screen: Rectangle): Rectangle2D {
        val x1: Double = x.screenToWorld(screen.ix)
        val y1: Double = y.screenToWorld(screen.iy)
        val x2: Double = x.screenToWorld(screen.ix + screen.iwidth)
        val y2: Double = y.screenToWorld(screen.iy + screen.iheight)
        return Rectangle2D.Double(
            x1,
            y1,
            x2 - x1,
            y2 - y1
        )
    }

    override fun worldToScreen(world: Point2D): Point {
        return Point(x.worldToScreen(world.x), y.worldToScreen(world.y))
    }

    override fun worldToScreen(world: Rectangle2D): Rectangle {
        val x1: Int = x.worldToScreen(world.x)
        val y2: Int = y.worldToScreen(world.y)
        val x2: Int = x.worldToScreen(world.x + world.width)
        val y1: Int = y.worldToScreen(world.y + world.height)
        return Rectangle(
            x1,
            y1,
            x2 - x1,
            y2 - y1
        )
    }

    override val isAffine: Boolean
        get() = true

    override fun setScreenSize(screenWidth: Int, screenHeight: Int) {
        if (this.screenWidth != screenWidth || this.screenHeight != screenHeight) {
            this.screenWidth = screenWidth
            this.screenHeight = screenHeight
            updateAspectRatio()
            x.screenSize = screenWidth
            y.screenSize = screenHeight
            notifyTransformChanged(ScreenTransformEvent())
        }
    }

    override fun setMaintainAspectRatio(maintainAspectRatio: Boolean) {
        if (this.maintainAspectRatio != maintainAspectRatio) {
            this.maintainAspectRatio = maintainAspectRatio
            updateAspectRatio()
            notifyTransformChanged(ScreenTransformEvent())
        }
    }

    private fun updateAspectRatio() {
        val width = screenWidth.toDouble()
        val height = screenHeight * worldAspectRatio
        if (maintainAspectRatio && width != height) {
            if (width < height) {
                x.setScreenMargins(0)
                y.setScreenMargins((screenHeight - screenWidth / worldAspectRatio).toInt())
            } else {
                x.setScreenMargins((screenWidth - screenHeight * worldAspectRatio).toInt())
                y.setScreenMargins(0)
            }
        }
    }

    override fun toString(): String {
        return "SimpleTwoDScreenTransform{" +
                "screenWidth=" + screenWidth +
                ", screenHeight=" + screenHeight +
                ", xInterval=" + xWorldInterval +
                ", yInterval=" + yWorldInterval +
                '}'
    }
}