/*
 * Copyright (c) 2011 Macrofocus GmbH. All Rights Reserved.
 */
package com.macrofocus.colormapping.implementation

import com.macrofocus.colormapping.DataFrameColorMapFactory
import com.macrofocus.common.format.CPFormat
import com.macrofocus.common.interval.MutableBoundedInterval
import com.macrofocus.common.interval.SimpleBoundedInterval
import com.macrofocus.common.selection.MutableSingleSelection
import com.macrofocus.common.selection.SimpleSingleSelection
import com.macrofocus.common.selection.SingleSelectionEvent
import com.macrofocus.common.selection.SingleSelectionListener
import org.mkui.color.MkColor
import org.mkui.colormap.*
import org.mkui.colormapping.AbstractColorMapping
import org.mkui.colormapping.MutableColorMapping
import org.molap.dataframe.DataFrame
import kotlin.reflect.KClass

/**
 * A simple mutable color mapping implementation.
 */
class SimpleColorMapping<R, C>(factory: ColorMapFactory, dataFrame: DataFrame<R, C, *>) : AbstractColorMapping<R, C>(),
    MutableColorMapping<R, C> {
    override var isActive = false
        get() = colorSelection.isActive && colorSelection.selected?.let { getCurrentColorMap(it) } != null
        set(active) {
            if (field != active) {
                field = active
                notifyColormappingChanged()
            }
        }
    private val dataFrame: DataFrame<R, C, *>
    private val factory: ColorMapFactory
    override val colorSelection: MutableSingleSelection<C>
    private val currentColorMaps: MutableMap<C, MutableColorMap> = HashMap<C, MutableColorMap>()
    private val categoricalColorMaps: MutableMap<C, MutableColorMap> = HashMap<C, MutableColorMap>()
    private val predefinedColorMaps: MutableMap<C, MutableColorMap> = HashMap<C, MutableColorMap>()
    private val customColorMaps: MutableMap<C, MutableColorMap> = HashMap<C, MutableColorMap>()
    private val boundedIntervalMap: MutableMap<C, MutableBoundedInterval> = HashMap<C, MutableBoundedInterval>()
    private val colorMapListener: ColorMapListener = object : ColorMapListener {
        override fun colorMapChanged(event: ColorMapEvent) {
            notifyColormappingChanged()
        }
    }
    val singleSelectionListener: SingleSelectionListener<C> = object : SingleSelectionListener<C> {
        override fun selectionChanged(event: SingleSelectionEvent<C>) {
            notifyColormappingChanged()
        }
    }

    override fun getColor(element: R): MkColor? {
        val colorMap: ColorMap? = getCurrentColorMap(colorSelection.selected!!)
        return if (colorMap != null) {
            val value: Any? = dataFrame.getValueAt(element, colorSelection.selected!!)
            colorMap.getColor(value)
        } else {
            null
        }
    }

    override operator fun iterator(): Iterator<R> {
        return if (isActive) {
            dataFrame.rows().iterator()
        } else {
            emptyList<R>().iterator()
        }
    }

    override fun getCurrentColorMap(column: C): MutableColorMap? {
        return if (column != null) {
            if (!currentColorMaps.containsKey(column)) {
                if (getNumericMin(column) != null && getNumericMax(column) != null) {
                    setCurrentColorMap(column, getPredefinedColorMap(column))
                } else {
                    setCurrentColorMap(column, getCategoricalColorMap(column))
                }
            }
            currentColorMaps[column]
        } else {
            null
        }
    }

    override fun setCurrentColorMap(column: C, colorMap: MutableColorMap?) {
        if (currentColorMaps[column] !== colorMap) {
            if (currentColorMaps.containsKey(column)) {
                currentColorMaps[column]!!.removeColorMapListener(colorMapListener)
            }
            if (colorMap != null) {
                currentColorMaps[column] = colorMap
                colorMap.addColorMapListener(colorMapListener)
            } else {
                currentColorMaps.remove(column)
            }
            notifyColormappingChanged()
        }
    }

    override fun getCategoricalColorMap(column: C): MutableColorMap? {
        if (!categoricalColorMaps.containsKey(column)) {
            categoricalColorMaps[column] = createQualitativeColorMap(column)
        }
        return categoricalColorMaps[column]
    }

    protected fun createQualitativeColorMap(column: C): MutableColorMap {
        return DataFrameColorMapFactory.createQualitativeColorMap(factory, dataFrame, column)
    }

    override fun getCustomColorMap(column: C): MutableColorMap? {
        if (!customColorMaps.containsKey(column)) {
            customColorMaps[column] = factory.createAutoSegmentedColorMap(getNumericMin(column), getNumericMax(column))
        }
        return customColorMaps[column]
    }

    override fun getPredefinedColorMap(column: C): MutableColorMap? {
        if (!predefinedColorMaps.containsKey(column)) {
            predefinedColorMaps[column] = createPredefinedColorMap(column)
        }
        return predefinedColorMaps[column]
    }

    protected fun createPredefinedColorMap(column: C): MutableColorMap {
        return factory.createAutoContinuousColorMap(getNumericMin(column), getNumericMax(column))
    }

    override fun getCurrentBoundedInterval(column: C): MutableBoundedInterval? {
        if (!boundedIntervalMap.containsKey(column)) {
            val boundedInterval: MutableBoundedInterval =
                SimpleBoundedInterval(getCurrentColorMap(column)!!.interval!!, getNumericMin(column)!!.toDouble(), getNumericMax(column)!!.toDouble())
            boundedIntervalMap[column] = boundedInterval
        }
        return boundedIntervalMap[column]
    }

    override fun getFormat(column: C): CPFormat<*>? {
        return null
    }

    override fun getNumericMin(column: C): Number? {
        return dataFrame.getColumn(column)?.min() as? Number?
    }

    override fun getNumericMax(column: C): Number? {
        return dataFrame.getColumn(column)?.max() as? Number?
    }

    override fun getType(column: C): KClass<out Any> {
        return dataFrame.getColumnClass(column)
    }

    fun resetRanges() {
        for ((key, value) in boundedIntervalMap) {
            val numericMin = getNumericMin(key)
            val numericMax = getNumericMax(key)
            if (numericMin != null && numericMax != null) {
                value.setMinMax(numericMin.toDouble(), numericMax.toDouble())
                value.setValue(numericMin.toDouble(), numericMax.toDouble() - numericMin.toDouble())
            }
        }
    }

    init {
        this.dataFrame = dataFrame
        this.factory = factory
        colorSelection = SimpleSingleSelection<C>()
        colorSelection.addSingleSelectionListener(singleSelectionListener)

//        java.util.List<Integer> numericalColumns = new ArrayList<Integer>();
//        for (int c = 0; c < tableModel.getColumnCount(); c++) {
//            if (Number.class.isAssignableFrom(tableModel.getColumnClass(c))) {
//                numericalColumns.add(c);
//            }
//        }
//
//        if (numericalColumns.size() > 0) {
//            getColorSelection().setSelected(new Column(tableModel, numericalColumns.get(0)));
////            if (numericalColumns.size() > 1) {
////                getSizeSelection().setSelected(new Column(tableModel, numericalColumns.get(1)));
////                if (numericalColumns.size() > 2) {
////                    getHeightSelection().setSelected(new Column(tableModel, numericalColumns.get(2)));
////                }
////            }
//        }
    }
}