package org.mkui.colormap.editor
/*
 * Copyright (c) 2017 Macrofocus GmbH. All Rights Reserved.
 */

import com.macrofocus.colormapping.implementation.SimpleColorMapping
import com.macrofocus.common.delegate.mutableLazy
import com.macrofocus.visual.SimpleVisual
import com.macrofocus.visual.SimpleVisualObjects
import org.mkui.color.MkColor
import org.mkui.colormap.ColorMapEvent
import org.mkui.colormap.ColorMapFactory
import org.mkui.colormap.ColorMapListener
import org.mkui.colormap.MutableColorMap
import org.mkui.component.CPComponent
import org.mkui.component.CPComponentProvider
import org.mkui.component.CPFactory
import org.mkui.component.button.CPButton
import org.mkui.component.input.CPColorPicker
import org.mkui.component.input.CPSlider
import org.mkui.component.panel.CPFormPanel
import org.mkui.component.panel.CPVerticalFlowPanel
import org.mkui.table.Table
import org.mkui.table.TableFactory
import org.molap.dataframe.AbstractMutableDataFrame
import org.molap.dataframe.DataFrameEvent
import org.molap.dataframe.MutableDataFrame
import org.molap.index.DefaultUniqueIndex
import org.molap.index.IntegerRangeUniqueIndex
import org.molap.series.Series
import kotlin.reflect.KClass

class CPCategoricalColorMapEditor(
    private val factory: CPFactory,
    private val colorMapFactory: ColorMapFactory,
    tableFactory: TableFactory
) : AbstractCategoricalColorMapEditor(), CPComponentProvider {

    private val mainPanel: CPVerticalFlowPanel = factory.createVerticalFlowPanel()
    private val generateColorsButton: CPButton
    private val table: Table<Int, String, Any?>
    private val missingValueColor: CPColorPicker
    private val resetButton: CPButton
    private val brightnessSlider: CPSlider
    private val saturationSlider: CPSlider

    init {
        val formPanel: CPFormPanel = factory.createFormPanel()
        generateColorsButton = factory.createButton("Generate")
        table = tableFactory.createTable<Int, String, Any?>()
        table.setColumnResizePolicy(Table.ColumnResizePolicy.Constrained)
        table.setPrefHeight(200.0)
        missingValueColor = factory.createColorPicker()
        resetButton = factory.createButton("Reset")
        brightnessSlider = factory.createSlider()
        brightnessSlider.setMinimum(-150.0)
        brightnessSlider.setMaximum(150.0)
        saturationSlider = factory.createSlider()
        saturationSlider.setMinimum(-150.0)
        saturationSlider.setMaximum(150.0)

        mainPanel.add(table)
        formPanel.add(factory.createLabel("Brightness:"), brightnessSlider)
        formPanel.add(factory.createLabel("Saturation:"), saturationSlider)
        formPanel.add(factory.createLabel("Missing Values Color:"), missingValueColor)
        mainPanel.add(formPanel)
    }

    override fun setColorMap(colorMap: MutableColorMap, type: KClass<*>) {
        val dataFrame: MutableDataFrame<Int, String, Any?> = ColorsDataFrame(colorMap, type)
        val visualObjects: SimpleVisualObjects<Int> = SimpleVisualObjects(dataFrame)
        val visual: SimpleVisual<Int, String> =
            SimpleVisual(visualObjects, SimpleColorMapping<Int, String>(colorMapFactory, dataFrame))
        table.setModel(dataFrame, visual)
        table.setCellRenderer("Color", ColorCellRenderer(factory, dataFrame))

        missingValueColor.setProperty(colorMap.nullColorProperty)
        brightnessSlider.setProperty(colorMap.brightnessProperty)
        saturationSlider.setProperty(colorMap.saturationProperty)
    }

//    fun setStyleClass(vararg styleClasses: String?) {
//        mainPanel.setStyleClass(styleClasses)
//    }

    override val component: CPComponent
        get() = mainPanel

    private class ColorsDataFrame(private val colorMap: MutableColorMap, private val valueType: KClass<*>) : AbstractMutableDataFrame<Int, String, Any?>() {

        override val columnIndex: DefaultUniqueIndex<String> = DefaultUniqueIndex("Value", "Color")
        val rowIndexDelegate = mutableLazy { IntegerRangeUniqueIndex(0, rowCount - 1) }
        override val rowIndex by rowIndexDelegate

        val listener: ColorMapListener = object : ColorMapListener {
            override fun colorMapChanged(event: ColorMapEvent) {
                rowIndexDelegate.forget()
                notifyDataFrameChanged(DataFrameEvent<Int, String>(null, null, true))
            }
        }

        init {
            colorMap.addWeakColorMapListener(listener)
        }

        override fun setValueAt(row: Int, column: String, value: Any?) {
            when (columnIndex.getAddress(column)) {
                1 -> colorMap.setColor(
                    colorMap.assignedValues!!.toTypedArray()[rowIndex.getAddress(row)],
                    value as MkColor?
                )
            }
        }

        override fun getRowClass(row: Int): KClass<*>? {
            return null
        }

        override fun getColumnClass(column: String): KClass<out Any> {
            return when (columnIndex.getAddress(column)) {
                0 -> valueType
                1 -> MkColor::class
                else -> Any::class
            }
        }

        override fun getValueAt(row: Int, column: String): Any? {
            val rowIndex: Int = rowIndex.getAddress(row)
            return when (columnIndex.getAddress(column)) {
                0 -> colorMap.assignedValues!!.toTypedArray()[rowIndex]
                1 -> colorMap.getColor(colorMap.assignedValues!!.toTypedArray()[rowIndex])
                else -> null
            }
        }

        override val rowCount: Int
            get() = colorMap.assignedValues!!.size

        override val columnCount: Int
            get() = 2

        fun isCellEditable(rowIndex: Int, columnIndex: Int): Boolean {
            return columnIndex == 1
        }
    }
}
